package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** Represents a Rados Block Device mount that lasts the lifetime of a pod. RBD volumes support ownership management and SELinux relabeling. */
final case class RBDPersistentVolumeSource(
  image : String,
  monitors : Seq[String],
  readOnly : Option[Boolean] = None,
  fsType : Option[String] = None,
  secretRef : Option[io.k8s.api.core.v1.SecretReference] = None,
  pool : Option[String] = None,
  keyring : Option[String] = None,
  user : Option[String] = None
) {

  /** Returns a new data with image set to new value */
  def withImage(value: String) : RBDPersistentVolumeSource = copy(image = value)
  /** transforms image to result of function */
  def mapImage(f: String => String) : RBDPersistentVolumeSource = copy(image = f(image))

  /** Returns a new data with monitors set to new value */
  def withMonitors(value: Seq[String]) : RBDPersistentVolumeSource = copy(monitors = value)
  /** Appends new values to monitors */
  def addMonitors(newValues: String*) : RBDPersistentVolumeSource = copy(monitors = monitors ++ newValues)
  /** transforms monitors to result of function */
  def mapMonitors(f: Seq[String] => Seq[String]) : RBDPersistentVolumeSource = copy(monitors = f(monitors))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : RBDPersistentVolumeSource = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : RBDPersistentVolumeSource = copy(readOnly = readOnly.map(f))

  /** Returns a new data with fsType set to new value */
  def withFsType(value: String) : RBDPersistentVolumeSource = copy(fsType = Some(value))
  /** if fsType has a value, transforms to the result of function*/
  def mapFsType(f: String => String) : RBDPersistentVolumeSource = copy(fsType = fsType.map(f))

  /** Returns a new data with secretRef set to new value */
  def withSecretRef(value: io.k8s.api.core.v1.SecretReference) : RBDPersistentVolumeSource = copy(secretRef = Some(value))
  /** if secretRef has a value, transforms to the result of function*/
  def mapSecretRef(f: io.k8s.api.core.v1.SecretReference => io.k8s.api.core.v1.SecretReference) : RBDPersistentVolumeSource = copy(secretRef = secretRef.map(f))

  /** Returns a new data with pool set to new value */
  def withPool(value: String) : RBDPersistentVolumeSource = copy(pool = Some(value))
  /** if pool has a value, transforms to the result of function*/
  def mapPool(f: String => String) : RBDPersistentVolumeSource = copy(pool = pool.map(f))

  /** Returns a new data with keyring set to new value */
  def withKeyring(value: String) : RBDPersistentVolumeSource = copy(keyring = Some(value))
  /** if keyring has a value, transforms to the result of function*/
  def mapKeyring(f: String => String) : RBDPersistentVolumeSource = copy(keyring = keyring.map(f))

  /** Returns a new data with user set to new value */
  def withUser(value: String) : RBDPersistentVolumeSource = copy(user = Some(value))
  /** if user has a value, transforms to the result of function*/
  def mapUser(f: String => String) : RBDPersistentVolumeSource = copy(user = user.map(f))
}

object RBDPersistentVolumeSource {

    implicit val encoder : Encoder[io.k8s.api.core.v1.RBDPersistentVolumeSource] = new Encoder[io.k8s.api.core.v1.RBDPersistentVolumeSource] {
        def apply[T : Builder](o: io.k8s.api.core.v1.RBDPersistentVolumeSource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("image", o.image)
            .write("monitors", o.monitors)
            .write("readOnly", o.readOnly)
            .write("fsType", o.fsType)
            .write("secretRef", o.secretRef)
            .write("pool", o.pool)
            .write("keyring", o.keyring)
            .write("user", o.user)
            .build
        }
    }

    implicit val decoder: Decoder[RBDPersistentVolumeSource] = new Decoder[RBDPersistentVolumeSource] {
      def apply[T : Reader](t: T): Either[String, RBDPersistentVolumeSource] = for {
          obj <- ObjectReader(t)
          image <- obj.read[String]("image")
          monitors <- obj.read[Seq[String]]("monitors")
          readOnly <- obj.readOpt[Boolean]("readOnly")
          fsType <- obj.readOpt[String]("fsType")
          secretRef <- obj.readOpt[io.k8s.api.core.v1.SecretReference]("secretRef")
          pool <- obj.readOpt[String]("pool")
          keyring <- obj.readOpt[String]("keyring")
          user <- obj.readOpt[String]("user")
      } yield RBDPersistentVolumeSource (
          image = image,
          monitors = monitors,
          readOnly = readOnly,
          fsType = fsType,
          secretRef = secretRef,
          pool = pool,
          keyring = keyring,
          user = user
        )
    }
}

