package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** SecretEnvSource selects a Secret to populate the environment variables with.

The contents of the target Secret's Data field will represent the key-value pairs as environment variables. */
final case class SecretEnvSource(
  name : Option[String] = None,
  optional : Option[Boolean] = None
) {

  /** Returns a new data with name set to new value */
  def withName(value: String) : SecretEnvSource = copy(name = Some(value))
  /** if name has a value, transforms to the result of function*/
  def mapName(f: String => String) : SecretEnvSource = copy(name = name.map(f))

  /** Returns a new data with optional set to new value */
  def withOptional(value: Boolean) : SecretEnvSource = copy(optional = Some(value))
  /** if optional has a value, transforms to the result of function*/
  def mapOptional(f: Boolean => Boolean) : SecretEnvSource = copy(optional = optional.map(f))
}

object SecretEnvSource {

    implicit val encoder : Encoder[io.k8s.api.core.v1.SecretEnvSource] = new Encoder[io.k8s.api.core.v1.SecretEnvSource] {
        def apply[T : Builder](o: io.k8s.api.core.v1.SecretEnvSource) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("name", o.name)
            .write("optional", o.optional)
            .build
        }
    }

    implicit val decoder: Decoder[SecretEnvSource] = new Decoder[SecretEnvSource] {
      def apply[T : Reader](t: T): Either[String, SecretEnvSource] = for {
          obj <- ObjectReader(t)
          name <- obj.readOpt[String]("name")
          optional <- obj.readOpt[Boolean]("optional")
      } yield SecretEnvSource (
          name = name,
          optional = optional
        )
    }
}

