package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** The pod this Toleration is attached to tolerates any taint that matches the triple <key,value,effect> using the matching operator <operator>. */
final case class Toleration(
  effect : Option[String] = None,
  key : Option[String] = None,
  tolerationSeconds : Option[Long] = None,
  operator : Option[String] = None,
  value : Option[String] = None
) {

  /** Returns a new data with effect set to new value */
  def withEffect(value: String) : Toleration = copy(effect = Some(value))
  /** if effect has a value, transforms to the result of function*/
  def mapEffect(f: String => String) : Toleration = copy(effect = effect.map(f))

  /** Returns a new data with key set to new value */
  def withKey(value: String) : Toleration = copy(key = Some(value))
  /** if key has a value, transforms to the result of function*/
  def mapKey(f: String => String) : Toleration = copy(key = key.map(f))

  /** Returns a new data with tolerationSeconds set to new value */
  def withTolerationSeconds(value: Long) : Toleration = copy(tolerationSeconds = Some(value))
  /** if tolerationSeconds has a value, transforms to the result of function*/
  def mapTolerationSeconds(f: Long => Long) : Toleration = copy(tolerationSeconds = tolerationSeconds.map(f))

  /** Returns a new data with operator set to new value */
  def withOperator(value: String) : Toleration = copy(operator = Some(value))
  /** if operator has a value, transforms to the result of function*/
  def mapOperator(f: String => String) : Toleration = copy(operator = operator.map(f))

  /** Returns a new data with value set to new value */
  def withValue(value: String) : Toleration = copy(value = Some(value))
  /** if value has a value, transforms to the result of function*/
  def mapValue(f: String => String) : Toleration = copy(value = value.map(f))
}

object Toleration {

    implicit val encoder : Encoder[io.k8s.api.core.v1.Toleration] = new Encoder[io.k8s.api.core.v1.Toleration] {
        def apply[T : Builder](o: io.k8s.api.core.v1.Toleration) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("effect", o.effect)
            .write("key", o.key)
            .write("tolerationSeconds", o.tolerationSeconds)
            .write("operator", o.operator)
            .write("value", o.value)
            .build
        }
    }

    implicit val decoder: Decoder[Toleration] = new Decoder[Toleration] {
      def apply[T : Reader](t: T): Either[String, Toleration] = for {
          obj <- ObjectReader(t)
          effect <- obj.readOpt[String]("effect")
          key <- obj.readOpt[String]("key")
          tolerationSeconds <- obj.readOpt[Long]("tolerationSeconds")
          operator <- obj.readOpt[String]("operator")
          value <- obj.readOpt[String]("value")
      } yield Toleration (
          effect = effect,
          key = key,
          tolerationSeconds = tolerationSeconds,
          operator = operator,
          value = value
        )
    }
}

