package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** TypedLocalObjectReference contains enough information to let you locate the typed referenced object inside the same namespace. */
final case class TypedLocalObjectReference(
  kind : String,
  name : String,
  apiGroup : Option[String] = None
) {

  /** Returns a new data with kind set to new value */
  def withKind(value: String) : TypedLocalObjectReference = copy(kind = value)
  /** transforms kind to result of function */
  def mapKind(f: String => String) : TypedLocalObjectReference = copy(kind = f(kind))

  /** Returns a new data with name set to new value */
  def withName(value: String) : TypedLocalObjectReference = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : TypedLocalObjectReference = copy(name = f(name))

  /** Returns a new data with apiGroup set to new value */
  def withApiGroup(value: String) : TypedLocalObjectReference = copy(apiGroup = Some(value))
  /** if apiGroup has a value, transforms to the result of function*/
  def mapApiGroup(f: String => String) : TypedLocalObjectReference = copy(apiGroup = apiGroup.map(f))
}

object TypedLocalObjectReference {

    implicit val encoder : Encoder[io.k8s.api.core.v1.TypedLocalObjectReference] = new Encoder[io.k8s.api.core.v1.TypedLocalObjectReference] {
        def apply[T : Builder](o: io.k8s.api.core.v1.TypedLocalObjectReference) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("kind", o.kind)
            .write("name", o.name)
            .write("apiGroup", o.apiGroup)
            .build
        }
    }

    implicit val decoder: Decoder[TypedLocalObjectReference] = new Decoder[TypedLocalObjectReference] {
      def apply[T : Reader](t: T): Either[String, TypedLocalObjectReference] = for {
          obj <- ObjectReader(t)
          kind <- obj.read[String]("kind")
          name <- obj.read[String]("name")
          apiGroup <- obj.readOpt[String]("apiGroup")
      } yield TypedLocalObjectReference (
          kind = kind,
          name = name,
          apiGroup = apiGroup
        )
    }
}

