package io.k8s.api.core.v1

import dev.hnaderi.k8s.utils._

/** VolumeMountStatus shows status of volume mounts. */
final case class VolumeMountStatus(
  mountPath : String,
  name : String,
  readOnly : Option[Boolean] = None,
  recursiveReadOnly : Option[String] = None
) {

  /** Returns a new data with mountPath set to new value */
  def withMountPath(value: String) : VolumeMountStatus = copy(mountPath = value)
  /** transforms mountPath to result of function */
  def mapMountPath(f: String => String) : VolumeMountStatus = copy(mountPath = f(mountPath))

  /** Returns a new data with name set to new value */
  def withName(value: String) : VolumeMountStatus = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : VolumeMountStatus = copy(name = f(name))

  /** Returns a new data with readOnly set to new value */
  def withReadOnly(value: Boolean) : VolumeMountStatus = copy(readOnly = Some(value))
  /** if readOnly has a value, transforms to the result of function*/
  def mapReadOnly(f: Boolean => Boolean) : VolumeMountStatus = copy(readOnly = readOnly.map(f))

  /** Returns a new data with recursiveReadOnly set to new value */
  def withRecursiveReadOnly(value: String) : VolumeMountStatus = copy(recursiveReadOnly = Some(value))
  /** if recursiveReadOnly has a value, transforms to the result of function*/
  def mapRecursiveReadOnly(f: String => String) : VolumeMountStatus = copy(recursiveReadOnly = recursiveReadOnly.map(f))
}

object VolumeMountStatus {

    implicit val encoder : Encoder[io.k8s.api.core.v1.VolumeMountStatus] = new Encoder[io.k8s.api.core.v1.VolumeMountStatus] {
        def apply[T : Builder](o: io.k8s.api.core.v1.VolumeMountStatus) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("mountPath", o.mountPath)
            .write("name", o.name)
            .write("readOnly", o.readOnly)
            .write("recursiveReadOnly", o.recursiveReadOnly)
            .build
        }
    }

    implicit val decoder: Decoder[VolumeMountStatus] = new Decoder[VolumeMountStatus] {
      def apply[T : Reader](t: T): Either[String, VolumeMountStatus] = for {
          obj <- ObjectReader(t)
          mountPath <- obj.read[String]("mountPath")
          name <- obj.read[String]("name")
          readOnly <- obj.readOpt[Boolean]("readOnly")
          recursiveReadOnly <- obj.readOpt[String]("recursiveReadOnly")
      } yield VolumeMountStatus (
          mountPath = mountPath,
          name = name,
          readOnly = readOnly,
          recursiveReadOnly = recursiveReadOnly
        )
    }
}

