package io.k8s.api.networking.v1

import dev.hnaderi.k8s.utils._

/** HTTPIngressPath associates a path with a backend. Incoming urls matching the path are forwarded to the backend. */
final case class HTTPIngressPath(
  backend : io.k8s.api.networking.v1.IngressBackend,
  pathType : String,
  path : Option[String] = None
) {

  /** Returns a new data with backend set to new value */
  def withBackend(value: io.k8s.api.networking.v1.IngressBackend) : HTTPIngressPath = copy(backend = value)
  /** transforms backend to result of function */
  def mapBackend(f: io.k8s.api.networking.v1.IngressBackend => io.k8s.api.networking.v1.IngressBackend) : HTTPIngressPath = copy(backend = f(backend))

  /** Returns a new data with pathType set to new value */
  def withPathType(value: String) : HTTPIngressPath = copy(pathType = value)
  /** transforms pathType to result of function */
  def mapPathType(f: String => String) : HTTPIngressPath = copy(pathType = f(pathType))

  /** Returns a new data with path set to new value */
  def withPath(value: String) : HTTPIngressPath = copy(path = Some(value))
  /** if path has a value, transforms to the result of function*/
  def mapPath(f: String => String) : HTTPIngressPath = copy(path = path.map(f))
}

object HTTPIngressPath {

    implicit val encoder : Encoder[io.k8s.api.networking.v1.HTTPIngressPath] = new Encoder[io.k8s.api.networking.v1.HTTPIngressPath] {
        def apply[T : Builder](o: io.k8s.api.networking.v1.HTTPIngressPath) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("backend", o.backend)
            .write("pathType", o.pathType)
            .write("path", o.path)
            .build
        }
    }

    implicit val decoder: Decoder[HTTPIngressPath] = new Decoder[HTTPIngressPath] {
      def apply[T : Reader](t: T): Either[String, HTTPIngressPath] = for {
          obj <- ObjectReader(t)
          backend <- obj.read[io.k8s.api.networking.v1.IngressBackend]("backend")
          pathType <- obj.read[String]("pathType")
          path <- obj.readOpt[String]("path")
      } yield HTTPIngressPath (
          backend = backend,
          pathType = pathType,
          path = path
        )
    }
}

