package io.k8s.api.networking.v1

import dev.hnaderi.k8s.utils._

/** IPBlock describes a particular CIDR (Ex. "192.168.1.0/24","2001:db8::/64") that is allowed to the pods matched by a NetworkPolicySpec's podSelector. The except entry describes CIDRs that should not be included within this rule. */
final case class IPBlock(
  cidr : String,
  except : Option[Seq[String]] = None
) {

  /** Returns a new data with cidr set to new value */
  def withCidr(value: String) : IPBlock = copy(cidr = value)
  /** transforms cidr to result of function */
  def mapCidr(f: String => String) : IPBlock = copy(cidr = f(cidr))

  /** Returns a new data with except set to new value */
  def withExcept(value: Seq[String]) : IPBlock = copy(except = Some(value))
  /** Appends new values to except */
  def addExcept(newValues: String*) : IPBlock = copy(except = Some(except.fold(newValues)(_ ++ newValues)))
  /** if except has a value, transforms to the result of function*/
  def mapExcept(f: Seq[String] => Seq[String]) : IPBlock = copy(except = except.map(f))
}

object IPBlock {

    implicit val encoder : Encoder[io.k8s.api.networking.v1.IPBlock] = new Encoder[io.k8s.api.networking.v1.IPBlock] {
        def apply[T : Builder](o: io.k8s.api.networking.v1.IPBlock) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("cidr", o.cidr)
            .write("except", o.except)
            .build
        }
    }

    implicit val decoder: Decoder[IPBlock] = new Decoder[IPBlock] {
      def apply[T : Reader](t: T): Either[String, IPBlock] = for {
          obj <- ObjectReader(t)
          cidr <- obj.read[String]("cidr")
          except <- obj.readOpt[Seq[String]]("except")
      } yield IPBlock (
          cidr = cidr,
          except = except
        )
    }
}

