package io.k8s.api.resource.v1alpha2

import dev.hnaderi.k8s.utils._

/** ResourceClaimConsumerReference contains enough information to let you locate the consumer of a ResourceClaim. The user must be a resource in the same namespace as the ResourceClaim. */
final case class ResourceClaimConsumerReference(
  name : String,
  resource : String,
  uid : String,
  apiGroup : Option[String] = None
) {

  /** Returns a new data with name set to new value */
  def withName(value: String) : ResourceClaimConsumerReference = copy(name = value)
  /** transforms name to result of function */
  def mapName(f: String => String) : ResourceClaimConsumerReference = copy(name = f(name))

  /** Returns a new data with resource set to new value */
  def withResource(value: String) : ResourceClaimConsumerReference = copy(resource = value)
  /** transforms resource to result of function */
  def mapResource(f: String => String) : ResourceClaimConsumerReference = copy(resource = f(resource))

  /** Returns a new data with uid set to new value */
  def withUid(value: String) : ResourceClaimConsumerReference = copy(uid = value)
  /** transforms uid to result of function */
  def mapUid(f: String => String) : ResourceClaimConsumerReference = copy(uid = f(uid))

  /** Returns a new data with apiGroup set to new value */
  def withApiGroup(value: String) : ResourceClaimConsumerReference = copy(apiGroup = Some(value))
  /** if apiGroup has a value, transforms to the result of function*/
  def mapApiGroup(f: String => String) : ResourceClaimConsumerReference = copy(apiGroup = apiGroup.map(f))
}

object ResourceClaimConsumerReference {

    implicit val encoder : Encoder[io.k8s.api.resource.v1alpha2.ResourceClaimConsumerReference] = new Encoder[io.k8s.api.resource.v1alpha2.ResourceClaimConsumerReference] {
        def apply[T : Builder](o: io.k8s.api.resource.v1alpha2.ResourceClaimConsumerReference) : T = {
          val obj = ObjectWriter[T]()
          obj
            .write("name", o.name)
            .write("resource", o.resource)
            .write("uid", o.uid)
            .write("apiGroup", o.apiGroup)
            .build
        }
    }

    implicit val decoder: Decoder[ResourceClaimConsumerReference] = new Decoder[ResourceClaimConsumerReference] {
      def apply[T : Reader](t: T): Either[String, ResourceClaimConsumerReference] = for {
          obj <- ObjectReader(t)
          name <- obj.read[String]("name")
          resource <- obj.read[String]("resource")
          uid <- obj.read[String]("uid")
          apiGroup <- obj.readOpt[String]("apiGroup")
      } yield ResourceClaimConsumerReference (
          name = name,
          resource = resource,
          uid = uid,
          apiGroup = apiGroup
        )
    }
}

