/*
 * Copyright 2024 Hydraulic Software AG
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package dev.hydraulic.conveyor.control;

import java.io.IOException;
import java.util.ServiceLoader;

/**
 * Provides access to the software update engine controlling this packaged application. To obtain an implementation of this
 * interface, use {@link #getInstance()}.
 */
public interface SoftwareUpdateController {
    /**
     * Returns an implementation of this interface, or null if the app isn't running inside a Conveyor package.
     */
    static SoftwareUpdateController getInstance() {
        return ServiceLoader.load(SoftwareUpdateController.class).findFirst().orElse(null);
    }

    /**
     * <p>Use this when the user has explicitly requested an update check from a GUI, and the {@link #getCurrentVersionFromRepository()}
     * method is advertising a newer version. Don't call it if there's no new version to upgrade to.</p>
     *
     * <p>This method will trigger a check for newly available software updates and return immediately. The upgrade process, if it proceeds,
     * may cause the process to be be forcibly shut down, so you should ensure everything is saved and you're ready for the process
     * to exit *before* calling this method. If you have any shutdown tasks that are needed, consider doing this before invoking this and
     * also.</p>
     *
     * @throws UnsupportedOperationException if {@link #canTriggerUpdateCheckUI()} doesn't return {@link Availability#AVAILABLE}
     * but you call this method anyway.
     */
    void triggerUpdateCheckUI();

    /**
     * Holds information on an available update. If it is newer than this release it will be downloaded and applied either in the
     * background automatically, or when the user accepts an update prompt, or when you call {@link #triggerUpdateCheckUI()}.
     */
    interface Version extends Comparable<Version> {
        /** Returns a version string. This string will probably be conformant with Mac/Windows version policies, but doesn't have to be. */
        String getVersion();

        /** Returns an integer that marks the revision of the package itself. */
        int getRevision();

        /**
         * Implements version comparison by comparing the version string using
         * <a href="https://maven.apache.org/ref/3.6.3/maven-artifact/apidocs/org/apache/maven/artifact/versioning/ComparableVersion.html"> the Maven algorithm</a>
         * first, and then the revision number.
         */
        @Override
        int compareTo(Version other);
    }

    /**
     * Returns a {@link Version} object initialized from system properties which are set during packaging, or null if no such properties
     * can be found.
     */
    Version getCurrentVersion();

    /**
     * Thrown if something goes wrong during an update check.
     */
    class UpdateCheckException extends IOException {
        public UpdateCheckException() {
        }

        public UpdateCheckException(String message) {
            super(message);
        }

        public UpdateCheckException(String message, Throwable cause) {
            super(message, cause);
        }

        public UpdateCheckException(Throwable cause) {
            super(cause);
        }
    }

    /**
     * Checks for the current version of the app that's being exposed by the update repository. Calling this method will block whilst making
     * an HTTP request, so it should be done on a background thread. The version returned may be the same version as this app, or a lower
     * version in case of downgrades. It is the responsibility of the app itself to compare version numbers using whatever scheme the app
     * uses to figure out if the version advertised is newer than the current one.
     *
     * @throws UpdateCheckException if something goes wrong reading the remote website.
     * @return an object implementing {@link Version} describing what the latest version is.
     */
    Version getCurrentVersionFromRepository() throws UpdateCheckException;

    /** Why update checks are or are not available. */
    enum Availability {
        /** Update checks can be done. */
        AVAILABLE,

        /** On this platform, online updates can't be triggered from code yet. */
        UNIMPLEMENTED,

        /** The package was built without upgrades support. */
        UNSUPPORTED_PACKAGE_TYPE,

        /** This app isn't a GUI app. */
        NON_GUI_APP,

        /** If other reasons are added in future that can't be mapped to one of the other entries, this one will be used instead. */
        OTHER
    }

    /**
     * <p>Returns whether the {@link #triggerUpdateCheckUI()} method will work on this platform and package.</p>
     *
     * <p>Update check triggering may not be possible in these situations:</p>
     *
     * <ul>
     *     <li>The feature isn't implemented yet on this platform.</li>
     *     <li>The package was built without upgrade capabilities.</li>
     * </ul>
     */
    Availability canTriggerUpdateCheckUI();
}
