package dev.keva.core.aof;

import dev.keva.core.config.KevaConfig;
import dev.keva.core.exception.StartupException;
import dev.keva.ioc.annotation.Autowired;
import dev.keva.ioc.annotation.Component;
import dev.keva.protocol.resp.Command;
import lombok.extern.slf4j.Slf4j;

import java.io.*;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;

@Slf4j
@Component
public class AOFContainer {
    private ReentrantLock bufferLock;
    private ObjectOutputStream output;
    private FileDescriptor fd;
    private boolean alwaysFlush;

    @Autowired
    private KevaConfig kevaConfig;

    private static class AppendOnlyObjectOutputStream extends ObjectOutputStream {
        public AppendOnlyObjectOutputStream(OutputStream out) throws IOException {
            super(out);
        }

        @Override
        protected void writeStreamHeader() throws IOException {
            reset();
        }
    }

    public void init() {
        alwaysFlush = kevaConfig.getAofInterval() == 0;
        bufferLock = new ReentrantLock();

        try {
            boolean isExists = new File(getWorkingDir() + "keva.aof").exists();
            FileOutputStream fos = new FileOutputStream(getWorkingDir() + "keva.aof", true);
            fd = fos.getFD();
            output = isExists ? new AppendOnlyObjectOutputStream(fos) : new ObjectOutputStream(fos);
        } catch (IOException e) {
            if (e instanceof FileNotFoundException) {
                log.info("AOF file not found, creating new file...");
                if (e.getMessage().contains("Permission denied")) {
                    log.error("Permission denied to access AOF file, please check your file permissions");
                    System.exit(1);
                }
            } else {
                log.error("Error writing to AOF file", e);
                System.exit(1);
            }
        }

        if (!alwaysFlush) {
            ScheduledExecutorService executorService = Executors.newSingleThreadScheduledExecutor();
            executorService.scheduleAtFixedRate(() -> {
                try {
                    flush();
                } catch (IOException e) {
                    log.error("Error writing AOF file", e);
                }
            }, kevaConfig.getAofInterval(), kevaConfig.getAofInterval(), TimeUnit.MILLISECONDS);
            log.info("AOF started with interval {} ms", kevaConfig.getAofInterval());
        } else {
            log.info("AOF will trigger for every new mutate command");
        }
    }

    public void write(Command command) {
        bufferLock.lock();
        try {
            output.writeObject(command.getObjects());
            if (alwaysFlush) {
                flush();
            }
        } catch (IOException e) {
            log.error("Error writing AOF file", e);
        } finally {
            bufferLock.unlock();
        }
    }

    private void flush() throws IOException {
        fd.sync();
    }

    public List<Command> read() throws IOException {
        final List<Command> commands = new ArrayList<>(100);
        try (FileInputStream fis = new FileInputStream(getWorkingDir() + "keva.aof");
             ObjectInputStream input = new ObjectInputStream(fis)) {
            log.info("AOF size is: {}", fis.getChannel().size());
            while (true) {
                byte[][] objects = (byte[][]) input.readObject();
                commands.add(Command.newInstance(objects, false));
            }
        } catch (final FileNotFoundException | EOFException ignored) {
            return commands;
        } catch (final ClassNotFoundException e) {
            final String msg = "Error reading AOF file";
            log.error(msg, e);
            throw new StartupException(msg, e);
        }
    }

    private String getWorkingDir() {
        String workingDir = kevaConfig.getWorkDirectory();
        return workingDir.equals("./") ? "" : workingDir + "/";
    }
}
