/*
 * Decompiled with CFR 0.152.
 */
package dev.langchain4j.community.data.document.loader.alloydb;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import dev.langchain4j.community.store.embedding.alloydb.AlloyDBEngine;
import dev.langchain4j.data.document.DefaultDocument;
import dev.langchain4j.data.document.Document;
import dev.langchain4j.data.document.Metadata;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.BiFunction;

public class AlloyDBLoader {
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private static final String DEFAULT_METADATA_COL = "langchain_metadata";
    private final AlloyDBEngine engine;
    private final String query;
    private final List<String> contentColumns;
    private final List<String> metadataColumns;
    private final BiFunction<Map<String, Object>, List<String>, String> formatter;
    private final String metadataJsonColumn;

    private AlloyDBLoader(Builder builder) {
        this.engine = builder.engine;
        this.query = builder.query;
        this.formatter = builder.formatter;
        this.contentColumns = builder.contentColumns;
        this.metadataColumns = builder.metadataColumns;
        this.metadataJsonColumn = builder.metadataJsonColumn;
    }

    private static String textFormatter(Map<String, Object> row, List<String> contentColumns) {
        StringBuilder sb = new StringBuilder();
        for (String column : contentColumns) {
            if (!row.containsKey(column)) continue;
            sb.append(row.get(column)).append(" ");
        }
        return sb.toString().trim();
    }

    private static String csvFormatter(Map<String, Object> row, List<String> contentColumns) {
        StringBuilder sb = new StringBuilder();
        for (String column : contentColumns) {
            if (!row.containsKey(column)) continue;
            sb.append(row.get(column)).append(", ");
        }
        return sb.toString().trim().replaceAll(", $", "");
    }

    private static String yamlFormatter(Map<String, Object> row, List<String> contentColumns) {
        StringBuilder sb = new StringBuilder();
        for (String column : contentColumns) {
            if (!row.containsKey(column)) continue;
            sb.append(column).append(": ").append(row.get(column)).append("\n");
        }
        return sb.toString().trim();
    }

    private static String jsonFormatter(Map<String, Object> row, List<String> contentColumns) {
        ObjectNode json = objectMapper.createObjectNode();
        for (String column : contentColumns) {
            if (!row.containsKey(column)) continue;
            json.put(column, (String)row.get(column));
        }
        return json.toString();
    }

    public static Builder builder(AlloyDBEngine engine) {
        return new Builder(engine);
    }

    public List<Document> load() throws SQLException {
        ArrayList<Document> documents = new ArrayList<Document>();
        try (Connection pool = this.engine.getConnection();
             PreparedStatement statement = pool.prepareStatement(this.query);){
            ResultSet resultSet = statement.executeQuery();
            while (resultSet.next()) {
                HashMap<String, Object> rowData = new HashMap<String, Object>();
                for (String column : this.contentColumns) {
                    rowData.put(column, resultSet.getString(column));
                }
                for (String column : this.metadataColumns) {
                    rowData.put(column, resultSet.getObject(column));
                }
                if (this.metadataJsonColumn != null) {
                    rowData.put(this.metadataJsonColumn, resultSet.getObject(this.metadataJsonColumn));
                }
                Document doc = this.parseDocFromRow(rowData);
                documents.add(doc);
            }
        }
        return documents;
    }

    private Document parseDocFromRow(Map<String, Object> row) {
        String pageContent = this.formatter.apply(row, this.contentColumns);
        HashMap<String, Object> metaDataMap = new HashMap<String, Object>();
        if (this.metadataJsonColumn != null && row.containsKey(this.metadataJsonColumn)) {
            try {
                metaDataMap.putAll((Map)objectMapper.readValue(row.get(this.metadataJsonColumn).toString(), Map.class));
            }
            catch (JsonProcessingException e) {
                throw new RuntimeException("Failed to parse JSON: " + e.getMessage() + ". Ensure metadata JSON structure matches the expected format.", e);
            }
        }
        for (String column : this.metadataColumns) {
            if (!row.containsKey(column) || column.equals(this.metadataJsonColumn)) continue;
            metaDataMap.put(column, row.get(column));
        }
        Metadata metadata = Metadata.from(metaDataMap);
        return new DefaultDocument(pageContent, metadata);
    }

    public static class Builder {
        private final AlloyDBEngine engine;
        private String tableName;
        private String query;
        private String metadataJsonColumn;
        private String schemaName = "public";
        private List<String> contentColumns;
        private List<String> metadataColumns;
        private String format;
        private BiFunction<Map<String, Object>, List<String>, String> formatter;

        public Builder(AlloyDBEngine engine) {
            this.engine = engine;
        }

        public Builder schemaName(String schemaName) {
            this.schemaName = schemaName;
            return this;
        }

        public Builder query(String query) {
            this.query = query;
            return this;
        }

        public Builder tableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public Builder formatter(BiFunction<Map<String, Object>, List<String>, String> formatter) {
            this.formatter = formatter;
            return this;
        }

        public Builder format(String format) {
            this.format = format;
            return this;
        }

        public Builder contentColumns(List<String> contentColumns) {
            this.contentColumns = contentColumns;
            return this;
        }

        public Builder metadataColumns(List<String> metadataColumns) {
            this.metadataColumns = metadataColumns;
            return this;
        }

        public Builder metadataJsonColumn(String metadataJsonColumn) {
            this.metadataJsonColumn = metadataJsonColumn;
            return this;
        }

        public AlloyDBLoader build() throws SQLException {
            if ((this.query == null || this.query.isEmpty()) && (this.tableName == null || this.tableName.isEmpty())) {
                throw new IllegalArgumentException("Either query or tableName must be specified.");
            }
            if (this.query == null) {
                this.query = String.format("SELECT * FROM \"%s\".\"%s\"", this.schemaName, this.tableName);
            }
            if (this.format != null && this.formatter != null) {
                throw new IllegalArgumentException("Only one of 'format' or 'formatter' should be specified.");
            }
            if (this.format != null) {
                switch (this.format) {
                    case "csv": {
                        this.formatter = AlloyDBLoader::csvFormatter;
                        break;
                    }
                    case "text": {
                        this.formatter = AlloyDBLoader::textFormatter;
                        break;
                    }
                    case "JSON": {
                        this.formatter = AlloyDBLoader::jsonFormatter;
                        break;
                    }
                    case "YAML": {
                        this.formatter = AlloyDBLoader::yamlFormatter;
                        break;
                    }
                    default: {
                        throw new IllegalArgumentException("format must be type: 'csv', 'text', 'JSON', 'YAML'");
                    }
                }
            } else if (this.formatter == null) {
                this.formatter = AlloyDBLoader::textFormatter;
            }
            ArrayList<String> columnNames = new ArrayList<String>();
            try (Connection pool = this.engine.getConnection();
                 PreparedStatement statement = pool.prepareStatement(this.query);){
                statement.setMaxRows(1);
                ResultSet resultSet = statement.executeQuery();
                for (int i = 1; i <= resultSet.getMetaData().getColumnCount(); ++i) {
                    columnNames.add(resultSet.getMetaData().getColumnName(i));
                }
            }
            this.contentColumns = this.contentColumns == null || this.contentColumns.isEmpty() ? List.of((String)columnNames.get(0)) : this.contentColumns;
            List<String> list = this.metadataColumns = this.metadataColumns == null || this.metadataColumns.isEmpty() ? columnNames.stream().filter(col -> !this.contentColumns.contains(col)).toList() : this.metadataColumns;
            if (this.metadataJsonColumn != null && !columnNames.contains(this.metadataJsonColumn)) {
                throw new IllegalArgumentException(String.format("Column %s not found in query result %s.", this.metadataJsonColumn, columnNames));
            }
            if (this.metadataJsonColumn == null && columnNames.contains(AlloyDBLoader.DEFAULT_METADATA_COL)) {
                this.metadataJsonColumn = AlloyDBLoader.DEFAULT_METADATA_COL;
            }
            ArrayList<String> allNames = new ArrayList<String>(this.contentColumns);
            allNames.addAll(this.metadataColumns);
            for (String name : allNames) {
                if (columnNames.contains(name)) continue;
                throw new IllegalArgumentException(String.format("Column %s not found in query result %s.", name, columnNames));
            }
            return new AlloyDBLoader(this);
        }
    }
}

