//
// Copyright (c) Like Magic e.U. and contributors. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for details.
//

package dev.likemagic.bluebreeze

import BBUUID

object BBConstants {
    object UUID {
        val cccd = BBUUID.fromString("2902").uuid
    }

    // https://www.bluetooth.com/wp-content/uploads/Files/Specification/Assigned_Numbers.html
    object Advertisement {
        const val FLAGS: UByte = 0x01u
        const val UUIDS_16_BIT_INCOMPLETE: UByte = 0x02u
        const val UUIDS_16_BIT_COMPLETE: UByte = 0x03u
        const val UUIDS_32_BIT_INCOMPLETE: UByte = 0x04u
        const val UUIDS_32_BIT_COMPLETE: UByte = 0x05u
        const val UUIDS_128_BIT_INCOMPLETE: UByte = 0x06u
        const val UUIDS_128_BIT_COMPLETE: UByte = 0x07u
        const val LOCAL_NAME_SHORTENED: UByte = 0x08u
        const val LOCAL_NAME: UByte = 0x09u
        const val TX_POWER_LEVEL: UByte = 0x0Au
        const val DEVICE_CLASS: UByte = 0x0Du
        const val SIMPLE_PAIRING_HASH: UByte = 0x0Eu
        const val SIMPLE_PAIRING_RANDOMIZER: UByte = 0x0Fu
        const val DEVICE_ID: UByte = 0x10u
        const val SECURITY_MANAGER_TK: UByte = 0x10u
        const val SECURITY_MANAGER_OUT_OF_BAND: UByte = 0x11u
        const val PERIPHERAL_CONNECTION_INTERVAL: UByte = 0x12u
        const val SOLICITATION_UUID_16_BIT: UByte = 0x14u
        const val SOLICITATION_UUID_128_BIT: UByte = 0x15u
        const val SERVICE_DATA_16_BIT: UByte = 0x16u
        const val TARGET_ADDRESS_PUBLIC: UByte = 0x17u
        const val TARGET_ADDRESS_RANDOM: UByte = 0x18u
        const val APPEARANCE: UByte = 0x19u
        const val ADVERTISING_INTERVAL: UByte = 0x1Au
        const val LE_DEVICE_ADDRESS: UByte = 0x1Bu
        const val LE_ROLE: UByte = 0x1Cu
        const val SIMPLE_PAIRING_HASH_C256: UByte = 0x1Du
        const val SIMPLE_PAIRING_RANDOMIZER_C256: UByte = 0x1Eu
        const val SOLICITATION_UUID_32_BIT: UByte = 0x1Fu
        const val SERVICE_DATA_32_BIT: UByte = 0x20u
        const val SERVICE_DATA_128_BIT: UByte = 0x21u
        const val LE_SECURE_CONNECTIONS_CONFIRMATION: UByte = 0x22u
        const val LE_SECURE_CONNECTIONS_RANDOM: UByte = 0x23u
        const val URI: UByte = 0x24u
        const val INDOOR_POSITIONING: UByte = 0x25u
        const val TRANSPORT_DISCOVERY: UByte = 0x26u
        const val LE_SUPPORTED_FEATURES: UByte = 0x27u
        const val CHANNEL_MAP_UPDATE: UByte = 0x28u
        const val PB_ADV: UByte = 0x29u
        const val MESH_MESSAGE: UByte = 0x2Au
        const val MESH_BEACON: UByte = 0x2Bu
        const val BIG_INFO: UByte = 0x2Cu
        const val BROADCAST_CODE: UByte = 0x2Du
        const val RESOLVABLE_SET_IDENTIFIER: UByte = 0x2Eu
        const val ADVERTISING_INTERVAL_LONG: UByte = 0x2Fu
        const val BROADCAST_NAME: UByte = 0x30u
        const val ENCRYPTED_ADVERTISING: UByte = 0x31u
        const val PERIODIC_ADVERTISING_RESPONSE_TIMING: UByte = 0x32u
        const val ELECTRONIC_SHELF_LABEL: UByte = 0x34u
        const val INFORMATION_3D: UByte = 0x3Du
        const val MANUFACTURER: UByte = 0xFFu
    }

    const val DEFAULT_MTU: Int = 23

    // https://bitbucket.org/bluetooth-SIG/public/src/main/assigned_numbers/uuids/service_uuids.yaml
    object Service {
        val knownUUIDs: Map<BBUUID, String> = mapOf(
            BBUUID.fromString("1800") to "GAP",
            BBUUID.fromString("1801") to "GATT",
            BBUUID.fromString("1802") to "Immediate Alert",
            BBUUID.fromString("1803") to "Link Loss",
            BBUUID.fromString("1804") to "Tx Power",
            BBUUID.fromString("1805") to "Current Time",
            BBUUID.fromString("1806") to "Reference Time Update",
            BBUUID.fromString("1807") to "Next DST Change",
            BBUUID.fromString("1808") to "Glucose",
            BBUUID.fromString("1809") to "Health Thermometer",
            BBUUID.fromString("180A") to "Device Information",
            BBUUID.fromString("180D") to "Heart Rate",
            BBUUID.fromString("180E") to "Phone Alert Status",
            BBUUID.fromString("180F") to "Battery",
            BBUUID.fromString("1810") to "Blood Pressure",
            BBUUID.fromString("1811") to "Alert Notification",
            BBUUID.fromString("1812") to "Human Interface Device",
            BBUUID.fromString("1813") to "Scan Parameters",
            BBUUID.fromString("1814") to "Running Speed and Cadence",
            BBUUID.fromString("1815") to "Automation IO",
            BBUUID.fromString("1816") to "Cycling Speed and Cadence",
            BBUUID.fromString("1818") to "Cycling Power",
            BBUUID.fromString("1819") to "Location and Navigation",
            BBUUID.fromString("181A") to "Environmental Sensing",
            BBUUID.fromString("181B") to "Body Composition",
            BBUUID.fromString("181C") to "User Data",
            BBUUID.fromString("181D") to "Weight Scale",
            BBUUID.fromString("181E") to "Bond Management",
            BBUUID.fromString("181F") to "Continuous Glucose Monitoring",
            BBUUID.fromString("1820") to "Internet Protocol Support",
            BBUUID.fromString("1821") to "Indoor Positioning",
            BBUUID.fromString("1822") to "Pulse Oximeter",
            BBUUID.fromString("1823") to "HTTP Proxy",
            BBUUID.fromString("1824") to "Transport Discovery",
            BBUUID.fromString("1825") to "Object Transfer",
            BBUUID.fromString("1826") to "Fitness Machine",
            BBUUID.fromString("1827") to "Mesh Provisioning",
            BBUUID.fromString("1828") to "Mesh Proxy",
            BBUUID.fromString("1829") to "Reconnection Configuration",
            BBUUID.fromString("183A") to "Insulin Delivery",
            BBUUID.fromString("183B") to "Binary Sensor",
            BBUUID.fromString("183C") to "Emergency Configuration",
            BBUUID.fromString("183D") to "Authorization Control",
            BBUUID.fromString("183E") to "Physical Activity Monitor",
            BBUUID.fromString("183F") to "Elapsed Time",
            BBUUID.fromString("1840") to "Generic Health Sensor",
            BBUUID.fromString("1843") to "Audio Input Control",
            BBUUID.fromString("1844") to "Volume Control",
            BBUUID.fromString("1845") to "Volume Offset Control",
            BBUUID.fromString("1846") to "Coordinated Set Identification",
            BBUUID.fromString("1847") to "Device Time",
            BBUUID.fromString("1848") to "Media Control",
            BBUUID.fromString("1849") to "Generic Media Control",
            BBUUID.fromString("184A") to "Constant Tone Extension",
            BBUUID.fromString("184B") to "Telephone Bearer",
            BBUUID.fromString("184C") to "Generic Telephone Bearer",
            BBUUID.fromString("184D") to "Microphone Control",
            BBUUID.fromString("184E") to "Audio Stream Control",
            BBUUID.fromString("184F") to "Broadcast Audio Scan",
            BBUUID.fromString("1850") to "Published Audio Capabilities",
            BBUUID.fromString("1851") to "Basic Audio Announcement",
            BBUUID.fromString("1852") to "Broadcast Audio Announcement",
            BBUUID.fromString("1853") to "Common Audio",
            BBUUID.fromString("1854") to "Hearing Access",
            BBUUID.fromString("1855") to "Telephony and Media Audio",
            BBUUID.fromString("1856") to "Public Broadcast Announcement",
            BBUUID.fromString("1857") to "Electronic Shelf Label",
            BBUUID.fromString("1858") to "Gaming Audio",
            BBUUID.fromString("1859") to "Mesh Proxy Solicitation"
        )
    }

    // https://bitbucket toorg/bluetooth-SIG/public/src/main/assigned_numbers/uuids/characteristic_uuids.yaml
    object Characteristic {
        val knownUUIDs: Map<BBUUID, String> = mapOf(
            BBUUID.fromString("2A00") to "Device Name",
            BBUUID.fromString("2A01") to "Appearance",
            BBUUID.fromString("2A02") to "Peripheral Privacy Flag",
            BBUUID.fromString("2A03") to "Reconnection Address",
            BBUUID.fromString("2A04") to "Peripheral Preferred Connection Parameters",
            BBUUID.fromString("2A05") to "Service Changed",
            BBUUID.fromString("2A06") to "Alert Level",
            BBUUID.fromString("2A07") to "Tx Power Level",
            BBUUID.fromString("2A08") to "Date Time",
            BBUUID.fromString("2A09") to "Day of Week",
            BBUUID.fromString("2A0A") to "Day Date Time",
            BBUUID.fromString("2A0C") to "Exact Time 256",
            BBUUID.fromString("2A0D") to "DST Offset",
            BBUUID.fromString("2A0E") to "Time Zone",
            BBUUID.fromString("2A0F") to "Local Time Information",
            BBUUID.fromString("2A11") to "Time with DST",
            BBUUID.fromString("2A12") to "Time Accuracy",
            BBUUID.fromString("2A13") to "Time Source",
            BBUUID.fromString("2A14") to "Reference Time Information",
            BBUUID.fromString("2A16") to "Time Update Control Point",
            BBUUID.fromString("2A17") to "Time Update State",
            BBUUID.fromString("2A18") to "Glucose Measurement",
            BBUUID.fromString("2A19") to "Battery Level",
            BBUUID.fromString("2A1C") to "Temperature Measurement",
            BBUUID.fromString("2A1D") to "Temperature Type",
            BBUUID.fromString("2A1E") to "Intermediate Temperature",
            BBUUID.fromString("2A21") to "Measurement Interval",
            BBUUID.fromString("2A22") to "Boot Keyboard Input Report",
            BBUUID.fromString("2A23") to "System ID",
            BBUUID.fromString("2A24") to "Model Number String",
            BBUUID.fromString("2A25") to "Serial Number String",
            BBUUID.fromString("2A26") to "Firmware Revision String",
            BBUUID.fromString("2A27") to "Hardware Revision String",
            BBUUID.fromString("2A28") to "Software Revision String",
            BBUUID.fromString("2A29") to "Manufacturer Name String",
            BBUUID.fromString("2A2A") to "IEEE 11073-20601 Regulatory Certification Data List",
            BBUUID.fromString("2A2B") to "Current Time",
            BBUUID.fromString("2A2C") to "Magnetic Declination",
            BBUUID.fromString("2A31") to "Scan Refresh",
            BBUUID.fromString("2A32") to "Boot Keyboard Output Report",
            BBUUID.fromString("2A33") to "Boot Mouse Input Report",
            BBUUID.fromString("2A34") to "Glucose Measurement Context",
            BBUUID.fromString("2A35") to "Blood Pressure Measurement",
            BBUUID.fromString("2A36") to "Intermediate Cuff Pressure",
            BBUUID.fromString("2A37") to "Heart Rate Measurement",
            BBUUID.fromString("2A38") to "Body Sensor Location",
            BBUUID.fromString("2A39") to "Heart Rate Control Point",
            BBUUID.fromString("2A3F") to "Alert Status",
            BBUUID.fromString("2A40") to "Ringer Control Point",
            BBUUID.fromString("2A41") to "Ringer Setting",
            BBUUID.fromString("2A42") to "Alert Category ID Bit Mask",
            BBUUID.fromString("2A43") to "Alert Category ID",
            BBUUID.fromString("2A44") to "Alert Notification Control Point",
            BBUUID.fromString("2A45") to "Unread Alert Status",
            BBUUID.fromString("2A46") to "New Alert",
            BBUUID.fromString("2A47") to "Supported New Alert Category",
            BBUUID.fromString("2A48") to "Supported Unread Alert Category",
            BBUUID.fromString("2A49") to "Blood Pressure Feature",
            BBUUID.fromString("2A4A") to "HID Information",
            BBUUID.fromString("2A4B") to "Report Map",
            BBUUID.fromString("2A4C") to "HID Control Point",
            BBUUID.fromString("2A4D") to "Report",
            BBUUID.fromString("2A4E") to "Protocol Mode",
            BBUUID.fromString("2A4F") to "Scan Interval Window",
            BBUUID.fromString("2A50") to "PnP ID",
            BBUUID.fromString("2A51") to "Glucose Feature",
            BBUUID.fromString("2A52") to "Record Access Control Point",
            BBUUID.fromString("2A53") to "RSC Measurement",
            BBUUID.fromString("2A54") to "RSC Feature",
            BBUUID.fromString("2A55") to "SC Control Point",
            BBUUID.fromString("2A5A") to "Aggregate",
            BBUUID.fromString("2A5B") to "CSC Measurement",
            BBUUID.fromString("2A5C") to "CSC Feature",
            BBUUID.fromString("2A5D") to "Sensor Location",
            BBUUID.fromString("2A5E") to "PLX Spot-Check Measurement",
            BBUUID.fromString("2A5F") to "PLX Continuous Measurement",
            BBUUID.fromString("2A60") to "PLX Features",
            BBUUID.fromString("2A63") to "Cycling Power Measurement",
            BBUUID.fromString("2A64") to "Cycling Power Vector",
            BBUUID.fromString("2A65") to "Cycling Power Feature",
            BBUUID.fromString("2A66") to "Cycling Power Control Point",
            BBUUID.fromString("2A67") to "Location and Speed",
            BBUUID.fromString("2A68") to "Navigation",
            BBUUID.fromString("2A69") to "Position Quality",
            BBUUID.fromString("2A6A") to "LN Feature",
            BBUUID.fromString("2A6B") to "LN Control Point",
            BBUUID.fromString("2A6C") to "Elevation",
            BBUUID.fromString("2A6D") to "Pressure",
            BBUUID.fromString("2A6E") to "Temperature",
            BBUUID.fromString("2A6F") to "Humidity",
            BBUUID.fromString("2A70") to "True Wind Speed",
            BBUUID.fromString("2A71") to "True Wind Direction",
            BBUUID.fromString("2A72") to "Apparent Wind Speed",
            BBUUID.fromString("2A73") to "Apparent Wind Direction",
            BBUUID.fromString("2A74") to "Gust Factor",
            BBUUID.fromString("2A75") to "Pollen Concentration",
            BBUUID.fromString("2A76") to "UV Index",
            BBUUID.fromString("2A77") to "Irradiance",
            BBUUID.fromString("2A78") to "Rainfall",
            BBUUID.fromString("2A79") to "Wind Chill",
            BBUUID.fromString("2A7A") to "Heat Index",
            BBUUID.fromString("2A7B") to "Dew Point",
            BBUUID.fromString("2A7D") to "Descriptor Value Changed",
            BBUUID.fromString("2A7E") to "Aerobic Heart Rate Lower Limit",
            BBUUID.fromString("2A7F") to "Aerobic Threshold",
            BBUUID.fromString("2A80") to "Age",
            BBUUID.fromString("2A81") to "Anaerobic Heart Rate Lower Limit",
            BBUUID.fromString("2A82") to "Anaerobic Heart Rate Upper Limit",
            BBUUID.fromString("2A83") to "Anaerobic Threshold",
            BBUUID.fromString("2A84") to "Aerobic Heart Rate Upper Limit",
            BBUUID.fromString("2A85") to "Date of Birth",
            BBUUID.fromString("2A86") to "Date of Threshold Assessment",
            BBUUID.fromString("2A87") to "Email Address",
            BBUUID.fromString("2A88") to "Fat Burn Heart Rate Lower Limit",
            BBUUID.fromString("2A89") to "Fat Burn Heart Rate Upper Limit",
            BBUUID.fromString("2A8A") to "First Name",
            BBUUID.fromString("2A8B") to "Five Zone Heart Rate Limits",
            BBUUID.fromString("2A8C") to "Gender",
            BBUUID.fromString("2A8D") to "Heart Rate Max",
            BBUUID.fromString("2A8E") to "Height",
            BBUUID.fromString("2A8F") to "Hip Circumference",
            BBUUID.fromString("2A90") to "Last Name",
            BBUUID.fromString("2A91") to "Maximum Recommended Heart Rate",
            BBUUID.fromString("2A92") to "Resting Heart Rate",
            BBUUID.fromString("2A93") to "Sport Type for Aerobic and Anaerobic Thresholds",
            BBUUID.fromString("2A94") to "Three Zone Heart Rate Limits",
            BBUUID.fromString("2A95") to "Two Zone Heart Rate Limits",
            BBUUID.fromString("2A96") to "VO2 Max",
            BBUUID.fromString("2A97") to "Waist Circumference",
            BBUUID.fromString("2A98") to "Weight",
            BBUUID.fromString("2A99") to "Database Change Increment",
            BBUUID.fromString("2A9A") to "User Index",
            BBUUID.fromString("2A9B") to "Body Composition Feature",
            BBUUID.fromString("2A9C") to "Body Composition Measurement",
            BBUUID.fromString("2A9D") to "Weight Measurement",
            BBUUID.fromString("2A9E") to "Weight Scale Feature",
            BBUUID.fromString("2A9F") to "User Control Point",
            BBUUID.fromString("2AA0") to "Magnetic Flux Density - 2D",
            BBUUID.fromString("2AA1") to "Magnetic Flux Density - 3D",
            BBUUID.fromString("2AA2") to "Language",
            BBUUID.fromString("2AA3") to "Barometric Pressure Trend",
            BBUUID.fromString("2AA4") to "Bond Management Control Point",
            BBUUID.fromString("2AA5") to "Bond Management Feature",
            BBUUID.fromString("2AA6") to "Central Address Resolution",
            BBUUID.fromString("2AA7") to "CGM Measurement",
            BBUUID.fromString("2AA8") to "CGM Feature",
            BBUUID.fromString("2AA9") to "CGM Status",
            BBUUID.fromString("2AAA") to "CGM Session Start Time",
            BBUUID.fromString("2AAB") to "CGM Session Run Time",
            BBUUID.fromString("2AAC") to "CGM Specific Ops Control Point",
            BBUUID.fromString("2AAD") to "Indoor Positioning Configuration",
            BBUUID.fromString("2AAE") to "Latitude",
            BBUUID.fromString("2AAF") to "Longitude",
            BBUUID.fromString("2AB0") to "Local North Coordinate",
            BBUUID.fromString("2AB1") to "Local East Coordinate",
            BBUUID.fromString("2AB2") to "Floor Number",
            BBUUID.fromString("2AB3") to "Altitude",
            BBUUID.fromString("2AB4") to "Uncertainty",
            BBUUID.fromString("2AB5") to "Location Name",
            BBUUID.fromString("2AB6") to "URI",
            BBUUID.fromString("2AB7") to "HTTP Headers",
            BBUUID.fromString("2AB8") to "HTTP Status Code",
            BBUUID.fromString("2AB9") to "HTTP Entity Body",
            BBUUID.fromString("2ABA") to "HTTP Control Point",
            BBUUID.fromString("2ABB") to "HTTPS Security",
            BBUUID.fromString("2ABC") to "TDS Control Point",
            BBUUID.fromString("2ABD") to "OTS Feature",
            BBUUID.fromString("2ABE") to "Object Name",
            BBUUID.fromString("2ABF") to "Object Type",
            BBUUID.fromString("2AC0") to "Object Size",
            BBUUID.fromString("2AC1") to "Object First-Created",
            BBUUID.fromString("2AC2") to "Object Last-Modified",
            BBUUID.fromString("2AC3") to "Object ID",
            BBUUID.fromString("2AC4") to "Object Properties",
            BBUUID.fromString("2AC5") to "Object Action Control Point",
            BBUUID.fromString("2AC6") to "Object List Control Point",
            BBUUID.fromString("2AC7") to "Object List Filter",
            BBUUID.fromString("2AC8") to "Object Changed",
            BBUUID.fromString("2AC9") to "Resolvable Private Address Only",
            BBUUID.fromString("2ACC") to "Fitness Machine Feature",
            BBUUID.fromString("2ACD") to "Treadmill Data",
            BBUUID.fromString("2ACE") to "Cross Trainer Data",
            BBUUID.fromString("2ACF") to "Step Climber Data",
            BBUUID.fromString("2AD0") to "Stair Climber Data",
            BBUUID.fromString("2AD1") to "Rower Data",
            BBUUID.fromString("2AD2") to "Indoor Bike Data",
            BBUUID.fromString("2AD3") to "Training Status",
            BBUUID.fromString("2AD4") to "Supported Speed Range",
            BBUUID.fromString("2AD5") to "Supported Inclination Range",
            BBUUID.fromString("2AD6") to "Supported Resistance Level Range",
            BBUUID.fromString("2AD7") to "Supported Heart Rate Range",
            BBUUID.fromString("2AD8") to "Supported Power Range",
            BBUUID.fromString("2AD9") to "Fitness Machine Control Point",
            BBUUID.fromString("2ADA") to "Fitness Machine Status",
            BBUUID.fromString("2ADB") to "Mesh Provisioning Data In",
            BBUUID.fromString("2ADC") to "Mesh Provisioning Data Out",
            BBUUID.fromString("2ADD") to "Mesh Proxy Data In",
            BBUUID.fromString("2ADE") to "Mesh Proxy Data Out",
            BBUUID.fromString("2AE0") to "Average Current",
            BBUUID.fromString("2AE1") to "Average Voltage",
            BBUUID.fromString("2AE2") to "Boolean",
            BBUUID.fromString("2AE3") to "Chromatic Distance from Planckian",
            BBUUID.fromString("2AE4") to "Chromaticity Coordinates",
            BBUUID.fromString("2AE5") to "Chromaticity in CCT and Duv Values",
            BBUUID.fromString("2AE6") to "Chromaticity Tolerance",
            BBUUID.fromString("2AE7") to "CIE 13.3-1995 Color Rendering Index",
            BBUUID.fromString("2AE8") to "Coefficient",
            BBUUID.fromString("2AE9") to "Correlated Color Temperature",
            BBUUID.fromString("2AEA") to "Count 16",
            BBUUID.fromString("2AEB") to "Count 24",
            BBUUID.fromString("2AEC") to "Country Code",
            BBUUID.fromString("2AED") to "Date UTC",
            BBUUID.fromString("2AEE") to "Electric Current",
            BBUUID.fromString("2AEF") to "Electric Current Range",
            BBUUID.fromString("2AF0") to "Electric Current Specification",
            BBUUID.fromString("2AF1") to "Electric Current Statistics",
            BBUUID.fromString("2AF2") to "Energy",
            BBUUID.fromString("2AF3") to "Energy in a Period of Day",
            BBUUID.fromString("2AF4") to "Event Statistics",
            BBUUID.fromString("2AF5") to "Fixed String 16",
            BBUUID.fromString("2AF6") to "Fixed String 24",
            BBUUID.fromString("2AF7") to "Fixed String 36",
            BBUUID.fromString("2AF8") to "Fixed String 8",
            BBUUID.fromString("2AF9") to "Generic Level",
            BBUUID.fromString("2AFA") to "Global Trade Item Number",
            BBUUID.fromString("2AFB") to "Illuminance",
            BBUUID.fromString("2AFC") to "Luminous Efficacy",
            BBUUID.fromString("2AFD") to "Luminous Energy",
            BBUUID.fromString("2AFE") to "Luminous Exposure",
            BBUUID.fromString("2AFF") to "Luminous Flux",
            BBUUID.fromString("2B00") to "Luminous Flux Range",
            BBUUID.fromString("2B01") to "Luminous Intensity",
            BBUUID.fromString("2B02") to "Mass Flow",
            BBUUID.fromString("2B03") to "Perceived Lightness",
            BBUUID.fromString("2B04") to "Percentage 8",
            BBUUID.fromString("2B05") to "Power",
            BBUUID.fromString("2B06") to "Power Specification",
            BBUUID.fromString("2B07") to "Relative Runtime in a Current Range",
            BBUUID.fromString("2B08") to "Relative Runtime in a Generic Level Range",
            BBUUID.fromString("2B09") to "Relative Value in a Voltage Range",
            BBUUID.fromString("2B0A") to "Relative Value in an Illuminance Range",
            BBUUID.fromString("2B0B") to "Relative Value in a Period of Day",
            BBUUID.fromString("2B0C") to "Relative Value in a Temperature Range",
            BBUUID.fromString("2B0D") to "Temperature 8",
            BBUUID.fromString("2B0E") to "Temperature 8 in a Period of Day",
            BBUUID.fromString("2B0F") to "Temperature 8 Statistics",
            BBUUID.fromString("2B10") to "Temperature Range",
            BBUUID.fromString("2B11") to "Temperature Statistics",
            BBUUID.fromString("2B12") to "Time Decihour 8",
            BBUUID.fromString("2B13") to "Time Exponential 8",
            BBUUID.fromString("2B14") to "Time Hour 24",
            BBUUID.fromString("2B15") to "Time Millisecond 24",
            BBUUID.fromString("2B16") to "Time Second 16",
            BBUUID.fromString("2B17") to "Time Second 8",
            BBUUID.fromString("2B18") to "Voltage",
            BBUUID.fromString("2B19") to "Voltage Specification",
            BBUUID.fromString("2B1A") to "Voltage Statistics",
            BBUUID.fromString("2B1B") to "Volume Flow",
            BBUUID.fromString("2B1C") to "Chromaticity Coordinate",
            BBUUID.fromString("2B1D") to "RC Feature",
            BBUUID.fromString("2B1E") to "RC Settings",
            BBUUID.fromString("2B1F") to "Reconnection Configuration Control Point",
            BBUUID.fromString("2B20") to "IDD Status Changed",
            BBUUID.fromString("2B21") to "IDD Status",
            BBUUID.fromString("2B22") to "IDD Annunciation Status",
            BBUUID.fromString("2B23") to "IDD Features",
            BBUUID.fromString("2B24") to "IDD Status Reader Control Point",
            BBUUID.fromString("2B25") to "IDD Command Control Point",
            BBUUID.fromString("2B26") to "IDD Command Data",
            BBUUID.fromString("2B27") to "IDD Record Access Control Point",
            BBUUID.fromString("2B28") to "IDD History Data",
            BBUUID.fromString("2B29") to "Client Supported Features",
            BBUUID.fromString("2B2A") to "Database Hash",
            BBUUID.fromString("2B2B") to "BSS Control Point",
            BBUUID.fromString("2B2C") to "BSS Response",
            BBUUID.fromString("2B2D") to "Emergency ID",
            BBUUID.fromString("2B2E") to "Emergency Text",
            BBUUID.fromString("2B2F") to "ACS Status",
            BBUUID.fromString("2B30") to "ACS Data In",
            BBUUID.fromString("2B31") to "ACS Data Out Notify",
            BBUUID.fromString("2B32") to "ACS Data Out Indicate",
            BBUUID.fromString("2B33") to "ACS Control Point",
            BBUUID.fromString("2B34") to "Enhanced Blood Pressure Measurement",
            BBUUID.fromString("2B35") to "Enhanced Intermediate Cuff Pressure",
            BBUUID.fromString("2B36") to "Blood Pressure Record",
            BBUUID.fromString("2B37") to "Registered User",
            BBUUID.fromString("2B38") to "BR-EDR Handover Data",
            BBUUID.fromString("2B39") to "Bluetooth SIG Data",
            BBUUID.fromString("2B3A") to "Server Supported Features",
            BBUUID.fromString("2B3B") to "Physical Activity Monitor Features",
            BBUUID.fromString("2B3C") to "General Activity Instantaneous Data",
            BBUUID.fromString("2B3D") to "General Activity Summary Data",
            BBUUID.fromString("2B3E") to "CardioRespiratory Activity Instantaneous Data",
            BBUUID.fromString("2B3F") to "CardioRespiratory Activity Summary Data",
            BBUUID.fromString("2B40") to "Step Counter Activity Summary Data",
            BBUUID.fromString("2B41") to "Sleep Activity Instantaneous Data",
            BBUUID.fromString("2B42") to "Sleep Activity Summary Data",
            BBUUID.fromString("2B43") to "Physical Activity Monitor Control Point",
            BBUUID.fromString("2B44") to "Physical Activity Current Session",
            BBUUID.fromString("2B45") to "Physical Activity Session Descriptor",
            BBUUID.fromString("2B46") to "Preferred Units",
            BBUUID.fromString("2B47") to "High Resolution Height",
            BBUUID.fromString("2B48") to "Middle Name",
            BBUUID.fromString("2B49") to "Stride Length",
            BBUUID.fromString("2B4A") to "Handedness",
            BBUUID.fromString("2B4B") to "Device Wearing Position",
            BBUUID.fromString("2B4C") to "Four Zone Heart Rate Limits",
            BBUUID.fromString("2B4D") to "High Intensity Exercise Threshold",
            BBUUID.fromString("2B4E") to "Activity Goal",
            BBUUID.fromString("2B4F") to "Sedentary Interval Notification",
            BBUUID.fromString("2B50") to "Caloric Intake",
            BBUUID.fromString("2B51") to "TMAP Role",
            BBUUID.fromString("2B77") to "Audio Input State",
            BBUUID.fromString("2B78") to "Gain Settings Attribute",
            BBUUID.fromString("2B79") to "Audio Input Type",
            BBUUID.fromString("2B7A") to "Audio Input Status",
            BBUUID.fromString("2B7B") to "Audio Input Control Point",
            BBUUID.fromString("2B7C") to "Audio Input Description",
            BBUUID.fromString("2B7D") to "Volume State",
            BBUUID.fromString("2B7E") to "Volume Control Point",
            BBUUID.fromString("2B7F") to "Volume Flags",
            BBUUID.fromString("2B80") to "Volume Offset State",
            BBUUID.fromString("2B81") to "Audio Location",
            BBUUID.fromString("2B82") to "Volume Offset Control Point",
            BBUUID.fromString("2B83") to "Audio Output Description",
            BBUUID.fromString("2B84") to "Set Identity Resolving Key",
            BBUUID.fromString("2B85") to "Coordinated Set Size",
            BBUUID.fromString("2B86") to "Set Member Lock",
            BBUUID.fromString("2B87") to "Set Member Rank",
            BBUUID.fromString("2B88") to "Encrypted Data Key Material",
            BBUUID.fromString("2B89") to "Apparent Energy 32",
            BBUUID.fromString("2B8A") to "Apparent Power",
            BBUUID.fromString("2B8B") to "Live Health Observations",
            BBUUID.fromString("2B8C") to "CO2 Concentration",
            BBUUID.fromString("2B8D") to "Cosine of the Angle",
            BBUUID.fromString("2B8E") to "Device Time Feature",
            BBUUID.fromString("2B8F") to "Device Time Parameters",
            BBUUID.fromString("2B90") to "Device Time",
            BBUUID.fromString("2B91") to "Device Time Control Point",
            BBUUID.fromString("2B92") to "Time Change Log Data",
            BBUUID.fromString("2B93") to "Media Player Name",
            BBUUID.fromString("2B94") to "Media Player Icon Object ID",
            BBUUID.fromString("2B95") to "Media Player Icon URL",
            BBUUID.fromString("2B96") to "Track Changed",
            BBUUID.fromString("2B97") to "Track Title",
            BBUUID.fromString("2B98") to "Track Duration",
            BBUUID.fromString("2B99") to "Track Position",
            BBUUID.fromString("2B9A") to "Playback Speed",
            BBUUID.fromString("2B9B") to "Seeking Speed",
            BBUUID.fromString("2B9C") to "Current Track Segments Object ID",
            BBUUID.fromString("2B9D") to "Current Track Object ID",
            BBUUID.fromString("2B9E") to "Next Track Object ID",
            BBUUID.fromString("2B9F") to "Parent Group Object ID",
            BBUUID.fromString("2BA0") to "Current Group Object ID",
            BBUUID.fromString("2BA1") to "Playing Order",
            BBUUID.fromString("2BA2") to "Playing Orders Supported",
            BBUUID.fromString("2BA3") to "Media State",
            BBUUID.fromString("2BA4") to "Media Control Point",
            BBUUID.fromString("2BA5") to "Media Control Point Opcodes Supported",
            BBUUID.fromString("2BA6") to "Search Results Object ID",
            BBUUID.fromString("2BA7") to "Search Control Point",
            BBUUID.fromString("2BA8") to "Energy 32",
            BBUUID.fromString("2BAD") to "Constant Tone Extension Enable",
            BBUUID.fromString("2BAE") to "Advertising Constant Tone Extension Minimum Length",
            BBUUID.fromString("2BAF") to "Advertising Constant Tone Extension Minimum Transmit Count",
            BBUUID.fromString("2BB0") to "Advertising Constant Tone Extension Transmit Duration",
            BBUUID.fromString("2BB1") to "Advertising Constant Tone Extension Interval",
            BBUUID.fromString("2BB2") to "Advertising Constant Tone Extension PHY",
            BBUUID.fromString("2BB3") to "Bearer Provider Name",
            BBUUID.fromString("2BB4") to "Bearer UCI",
            BBUUID.fromString("2BB5") to "Bearer Technology",
            BBUUID.fromString("2BB6") to "Bearer URI Schemes Supported List",
            BBUUID.fromString("2BB7") to "Bearer Signal Strength",
            BBUUID.fromString("2BB8") to "Bearer Signal Strength Reporting Interval",
            BBUUID.fromString("2BB9") to "Bearer List Current Calls",
            BBUUID.fromString("2BBA") to "Content Control ID",
            BBUUID.fromString("2BBB") to "Status Flags",
            BBUUID.fromString("2BBC") to "Incoming Call Target Bearer URI",
            BBUUID.fromString("2BBD") to "Call State",
            BBUUID.fromString("2BBE") to "Call Control Point",
            BBUUID.fromString("2BBF") to "Call Control Point Optional Opcodes",
            BBUUID.fromString("2BC0") to "Termination Reason",
            BBUUID.fromString("2BC1") to "Incoming Call",
            BBUUID.fromString("2BC2") to "Call Friendly Name",
            BBUUID.fromString("2BC3") to "Mute",
            BBUUID.fromString("2BC4") to "Sink ASE",
            BBUUID.fromString("2BC5") to "Source ASE",
            BBUUID.fromString("2BC6") to "ASE Control Point",
            BBUUID.fromString("2BC7") to "Broadcast Audio Scan Control Point",
            BBUUID.fromString("2BC8") to "Broadcast Receive State",
            BBUUID.fromString("2BC9") to "Sink PAC",
            BBUUID.fromString("2BCA") to "Sink Audio Locations",
            BBUUID.fromString("2BCB") to "Source PAC",
            BBUUID.fromString("2BCC") to "Source Audio Locations",
            BBUUID.fromString("2BCD") to "Available Audio Contexts",
            BBUUID.fromString("2BCE") to "Supported Audio Contexts",
            BBUUID.fromString("2BCF") to "Ammonia Concentration",
            BBUUID.fromString("2BD0") to "Carbon Monoxide Concentration",
            BBUUID.fromString("2BD1") to "Methane Concentration",
            BBUUID.fromString("2BD2") to "Nitrogen Dioxide Concentration",
            BBUUID.fromString("2BD3") to "Non-Methane Volatile Organic Compounds Concentration",
            BBUUID.fromString("2BD4") to "Ozone Concentration",
            BBUUID.fromString("2BD5") to "Particulate Matter - PM1 Concentration",
            BBUUID.fromString("2BD6") to "Particulate Matter - PM2.5 Concentration",
            BBUUID.fromString("2BD7") to "Particulate Matter - PM10 Concentration",
            BBUUID.fromString("2BD8") to "Sulfur Dioxide Concentration",
            BBUUID.fromString("2BD9") to "Sulfur Hexafluoride Concentration",
            BBUUID.fromString("2BDA") to "Hearing Aid Features",
            BBUUID.fromString("2BDB") to "Hearing Aid Preset Control Point",
            BBUUID.fromString("2BDC") to "Active Preset Index",
            BBUUID.fromString("2BDD") to "Stored Health Observations",
            BBUUID.fromString("2BDE") to "Fixed String 64",
            BBUUID.fromString("2BDF") to "High Temperature",
            BBUUID.fromString("2BE0") to "High Voltage",
            BBUUID.fromString("2BE1") to "Light Distribution",
            BBUUID.fromString("2BE2") to "Light Output",
            BBUUID.fromString("2BE3") to "Light Source Type",
            BBUUID.fromString("2BE4") to "Noise",
            BBUUID.fromString("2BE5") to "Relative Runtime in a Correlated Color Temperature Range",
            BBUUID.fromString("2BE6") to "Time Second 32",
            BBUUID.fromString("2BE7") to "VOC Concentration",
            BBUUID.fromString("2BE8") to "Voltage Frequency",
            BBUUID.fromString("2BE9") to "Battery Critical Status",
            BBUUID.fromString("2BEA") to "Battery Health Status",
            BBUUID.fromString("2BEB") to "Battery Health Information",
            BBUUID.fromString("2BEC") to "Battery Information",
            BBUUID.fromString("2BED") to "Battery Level Status",
            BBUUID.fromString("2BEE") to "Battery Time Status",
            BBUUID.fromString("2BEF") to "Estimated Service Date",
            BBUUID.fromString("2BF0") to "Battery Energy Status",
            BBUUID.fromString("2BF1") to "Observation Schedule Changed",
            BBUUID.fromString("2BF2") to "Current Elapsed Time",
            BBUUID.fromString("2BF3") to "Health Sensor Features",
            BBUUID.fromString("2BF4") to "GHS Control Point",
            BBUUID.fromString("2BF5") to "LE GATT Security Levels",
            BBUUID.fromString("2BF6") to "ESL Address",
            BBUUID.fromString("2BF7") to "AP Sync Key Material",
            BBUUID.fromString("2BF8") to "ESL Response Key Material",
            BBUUID.fromString("2BF9") to "ESL Current Absolute Time",
            BBUUID.fromString("2BFA") to "ESL Display Information",
            BBUUID.fromString("2BFB") to "ESL Image Information",
            BBUUID.fromString("2BFC") to "ESL Sensor Information",
            BBUUID.fromString("2BFD") to "ESL LED Information",
            BBUUID.fromString("2BFE") to "ESL Control Point",
            BBUUID.fromString("2BFF") to "UDI for Medical Devices",
            BBUUID.fromString("2C00") to "GMAP Role",
            BBUUID.fromString("2C01") to "UGG Features",
            BBUUID.fromString("2C02") to "UGT Features",
            BBUUID.fromString("2C03") to "BGS Features",
            BBUUID.fromString("2C04") to "BGR Features",
            BBUUID.fromString("2C05") to "Percentage 8 Steps",
        )
    }

    object Manufacturer {
        val knownIds: Map<Int, String> = mapOf(
            3515 to "Nexis Link Technology Co., Ltd.",
            3514 to "Veo Technologies ApS",
            3513 to "CompanyDeep Ltd",
            3512 to "Shenzhen Chuangyuan Digital Technology Co., Ltd",
            3511 to "Morningstar Corporation",
            3510 to "SAFEGUARD EQUIPMENT, INC.",
            3509 to "Djup AB",
            3508 to "Franklin Control Systems",
            3507 to "SHENZHEN REFLYING ELECTRONIC CO., LTD",
            3506 to "Whirlpool",
            3505 to "TELE System Communications Pte. Ltd.",
            3504 to "Invisalert Solutions, Inc.",
            3503 to "Hexagon Aura Reality AG",
            3502 to "TITUM AUDIO, INC.",
            3501 to "linktop",
            3500 to "ITALTRACTOR ITM S.P.A.",
            3499 to "Efento",
            3498 to "Shenzhen EBELONG Technology Co., Ltd.",
            3497 to "lnventronics GmbH",
            3496 to "Airwallet ApS",
            3495 to "Novoferm tormatic GmbH",
            3494 to "Generac Corporation",
            3493 to "PIXELA CORPORATION",
            3492 to "HP Tuners",
            3491 to "Airgraft Inc.",
            3490 to "KIWI.KI GmbH",
            3489 to "Fen Systems Ltd.",
            3488 to "SICK AG",
            3487 to "MML US, Inc",
            3486 to "Impulse Wellness LLC",
            3485 to "Cear, Inc.",
            3484 to "Skytech Creations Limited",
            3483 to "Boxyz, Inc.",
            3482 to "Yeasound (Xiamen) Hearing Technology Co., Ltd",
            3481 to "Caire Inc.",
            3480 to "E.F. Johnson Company",
            3479 to "Zhejiang Huanfu Technology Co., LTD",
            3478 to "NEOKOHM SISTEMAS ELETRONICOS LTDA",
            3477 to "Hunter Industries Incorporated",
            3476 to "Shrooly Inc",
            3475 to "HagerEnergy GmbH",
            3474 to "TACHIKAWA CORPORATION",
            3473 to "Beamex Oy Ab",
            3472 to "LAAS ApS",
            3471 to "Canon Electronics Inc.",
            3470 to "Optivolt Labs, Inc.",
            3469 to "RF Electronics Limited",
            3468 to "Ultimea Technology (Shenzhen) Limited",
            3467 to "Software Development, LLC",
            3466 to "Simply Embedded Inc.",
            3465 to "Nanohex Corp",
            3464 to "Geocene Inc.",
            3463 to "Quectel Wireless Solutions Co., Ltd.",
            3462 to "ROCKWELL AUTOMATION, INC.",
            3461 to "SEW-EURODRIVE GmbH & Co KG",
            3460 to "Testo SE & Co. KGaA",
            3459 to "ATLANTIC SOCIETE FRANCAISE DE DEVELOPPEMENT THERMIQUE",
            3458 to "VELCO",
            3457 to "Beyerdynamic GmbH & Co. KG",
            3456 to "Gravaa B.V.",
            3455 to "Konova",
            3454 to "Daihatsu Motor Co., Ltd.",
            3453 to "Taiko Audio B.V.",
            3452 to "BeiJing SmartChip Microelectronics Technology Co.,Ltd",
            3451 to "MindMaze SA",
            3450 to "Xiamen Intretech Inc.",
            3449 to "VIVIWARE JAPAN, Inc.",
            3448 to "MITACHI CO.,LTD.",
            3447 to "DualNetworks SA",
            3446 to "i-focus Co.,Ltd",
            3445 to "Indistinguishable From Magic, Inc.",
            3444 to "ANUME s.r.o.",
            3443 to "iota Biosciences, Inc.",
            3442 to "Earfun Technology (HK) Limited",
            3441 to "Kiteras Inc.",
            3440 to "Kindhome",
            3439 to "Closed Joint Stock Company NVP BOLID",
            3438 to "Look Cycle International",
            3437 to "DYNAMOX S/A",
            3436 to "Ambient IoT Pty Ltd",
            3435 to "Crane Payment Innovations, Inc.",
            3434 to "Helge Kaiser GmbH",
            3433 to "AIR AROMA INTERNATIONAL PTY LTD",
            3432 to "Status Audio LLC",
            3431 to "BLACK BOX NETWORK SERVICES INDIA PRIVATE LIMITED",
            3430 to "Hendrickson USA , L.L.C",
            3429 to "Molnlycke Health Care AB",
            3428 to "Southco",
            3427 to "SKF France",
            3426 to "MEBSTER s.r.o.",
            3425 to "F.I.P. FORMATURA INIEZIONE POLIMERI - S.P.A.",
            3424 to "Smart Products Connection, S.A.",
            3423 to "SiChuan Homme Intelligent Technology co.,Ltd.",
            3422 to "Pella Corp",
            3421 to "Stogger B.V.",
            3420 to "Pison Technology, Inc.",
            3419 to "Axis Communications AB",
            3418 to "Gunnebo Aktiebolag",
            3417 to "HYUPSUNG MACHINERY ELECTRIC CO., LTD.",
            3416 to "ifm electronic gmbh",
            3415 to "Nanjing Xinxiangyuan Microelectronics Co., Ltd.",
            3414 to "Wellang.Co,.Ltd",
            3413 to "NO CLIMB PRODUCTS LTD",
            3412 to "ISEKI FRANCE S.A.S",
            3411 to "Luxottica Group S.p.A",
            3410 to "DIVAN TRADING CO., LTD.",
            3409 to "Genetus inc.",
            3408 to "NINGBO FOTILE KITCHENWARE CO., LTD.",
            3407 to "Movano Inc.",
            3406 to "NIKAT SOLUTIONS PRIVATE LIMITED",
            3405 to "Optec, LLC",
            3404 to "IotGizmo Corporation",
            3403 to "Soundwave Hearing, LLC",
            3402 to "Rockpile Solutions, LLC",
            3401 to "Refrigerated Transport Electronics, Inc.",
            3400 to "Vemcon GmbH",
            3399 to "Shenzhen DOKE Electronic Co., Ltd",
            3398 to "Thales Simulation & Training AG",
            3397 to "Odeon, Inc.",
            3396 to "Ex Makhina Inc.",
            3395 to "Gosuncn Technology Group Co., Ltd.",
            3394 to "TEKTRO TECHNOLOGY CORPORATION",
            3393 to "CPAC Systems AB",
            3392 to "SignalFire Telemetry, Inc.",
            3391 to "Vogels Products B.V.",
            3390 to "LUMINOAH, INC.",
            3389 to "bHaptics Inc.",
            3388 to "SIRONA Dental Systems GmbH",
            3387 to "Lone Star Marine Pty Ltd",
            3386 to "Frost Solutions, LLC",
            3385 to "Systemic Games, LLC",
            3384 to "CycLock",
            3383 to "Zerene Inc.",
            3382 to "XIHAO INTELLIGENGT TECHNOLOGY CO., LTD",
            3381 to "Universidad Politecnica de Madrid",
            3380 to "ZILLIOT TECHNOLOGIES PRIVATE LIMITED",
            3379 to "Micropower Group AB",
            3378 to "Badger Meter",
            3377 to "SYNCHRON, INC.",
            3376 to "Laxmi Therapeutic Devices, Inc.",
            3375 to "Delta Development Team, Inc",
            3374 to "Advanced Electronic Applications, Inc",
            3373 to "Cooler Pro, LLC",
            3372 to "SIL System Integration Laboratory GmbH",
            3371 to "Sensoryx AG",
            3370 to "PhysioLogic Devices, Inc.",
            3369 to "MIYAKAWA ELECTRIC WORKS LTD.",
            3368 to "FUJITSU COMPONENT LIMITED",
            3367 to "velocitux",
            3366 to "Burkert Werke GmbH & Co. KG",
            3365 to "System Elite Holdings Group Limited",
            3364 to "Japan Display Inc.",
            3363 to "GREE Electric Appliances, Inc. of Zhuhai",
            3362 to "Cedarware, Corp.",
            3361 to "Cennox Group Limited",
            3360 to "SCIENTERRA LIMITED",
            3359 to "Synkopi, Inc.",
            3358 to "FESTINA LOTUS SA",
            3357 to "Electronics4All Inc.",
            3356 to "LIMBOID LLC",
            3355 to "RACHIO, INC.",
            3354 to "Maturix ApS",
            3353 to "C.G. Air Systemes Inc.",
            3352 to "Bioliberty Ltd",
            3351 to "Akix S.r.l.",
            3350 to "Nations Technologies Inc.",
            3349 to "Spark",
            3348 to "Merry Electronics (S) Pte Ltd",
            3347 to "MERRY ELECTRONICS CO., LTD.",
            3346 to "Spartek Systems Inc.",
            3345 to "Great Dane LLC",
            3344 to "JVC KENWOOD Corporation",
            3343 to "Timebirds Australia Pty Ltd",
            3342 to "PetVoice Co., Ltd.",
            3341 to "C.Ed. Schulte GmbH Zylinderschlossfabrik",
            3340 to "Planmeca Oy",
            3339 to "Research Products Corporation",
            3338 to "CATEYE Co., Ltd.",
            3337 to "Leica Geosystems AG",
            3336 to "Datalogic USA, Inc.",
            3335 to "Datalogic S.r.l.",
            3334 to "doubleO Co., Ltd.",
            3333 to "Energy Technology and Control Limited",
            3332 to "Bartec Auto Id Ltd",
            3331 to "MakuSafe Corp",
            3330 to "Rocky Mountain ATV/MC Jake Wilson",
            3329 to "KEEPEN",
            3328 to "Sparkpark AS",
            3327 to "Ergodriven Inc",
            3326 to "Thule Group AB",
            3325 to "Wuhan Woncan Construction Technologies Co., Ltd.",
            3324 to "ElectronX design",
            3323 to "Tyromotion GmbH",
            3322 to "Protect Animals With Satellites LLC",
            3321 to "Tamblue Oy",
            3320 to "core sensing GmbH",
            3319 to "TVS Motor Company Ltd.",
            3318 to "OJ Electronics A/S",
            3317 to "BOS Balance of Storage Systems AG",
            3316 to "SOLUX PTY LTD",
            3315 to "Radio Sound",
            3314 to "BestSens AG",
            3313 to "Midmark",
            3312 to "THOTAKA TEKHNOLOGIES INDIA PRIVATE LIMITED",
            3311 to "POGS B.V.",
            3310 to "MadgeTech, Inc",
            3309 to "CV. NURI TEKNIK",
            3308 to "Pacific Coast Fishery Services (2003) Inc.",
            3307 to "Shenzhen Tingting Technology Co. LTD",
            3306 to "HAYWARD INDUSTRIES, INC.",
            3305 to "PEAG, LLC dba JLab Audio",
            3304 to "Dongguan Yougo Electronics Co.,Ltd.",
            3303 to "TAG HEUER SA",
            3302 to "McWong International, Inc.",
            3301 to "Amina Distribution AS",
            3300 to "Off-Highway Powertrain Services Germany GmbH",
            3299 to "Taiwan Fuhsing",
            3298 to "CORVENT MEDICAL, INC.",
            3297 to "Regal Beloit America, Inc.",
            3296 to "VODALOGIC PTY LTD",
            3295 to "SHENZHEN CHENYUN ELECTRONICS  CO., LTD",
            3294 to "RESPONSE TECHNOLOGIES, LTD.",
            3293 to "Alif Semiconductor, Inc.",
            3292 to "Ypsomed AG",
            3291 to "Circus World Displays Limited",
            3290 to "Wolf Steel ltd",
            3289 to "Minami acoustics Limited",
            3288 to "SIA Mesh Group",
            3287 to "Maztech Industries, LLC",
            3286 to "HHO (Hangzhou) Digital Technology Co., Ltd.",
            3285 to "Numa Products, LLC",
            3284 to "Reoqoo IoT Technology Co., Ltd.",
            3283 to "TechSwipe",
            3282 to "EQOM SSC B.V.",
            3281 to "Imagine Marketing Limited",
            3280 to "MooreSilicon Semiconductor Technology (Shanghai) Co., LTD.",
            3279 to "Shenzhen CESI Information Technology Co., Ltd.",
            3278 to "SENOSPACE LLC",
            3277 to "YanFeng Visteon(Chongqing) Automotive Electronic Co.,Ltd",
            3276 to "Kord Defence Pty Ltd",
            3275 to "NOTHING TECHNOLOGY LIMITED",
            3274 to "Cyclops Marine Ltd",
            3273 to "Innocent Technology Co., Ltd.",
            3272 to "TrikThom",
            3271 to "SB C&S Corp.",
            3270 to "Serial Technology Corporation",
            3269 to "Open Road Solutions, Inc.",
            3268 to "ABUS August Bremicker Soehne Kommanditgesellschaft",
            3267 to "HMD Global Oy",
            3266 to "Anker Innovations Limited",
            3265 to "CLEIO Inc.",
            3264 to "Garnet Instruments Ltd.",
            3263 to "Forward Thinking Systems LLC.",
            3261 to "Pricer AB",
            3260 to "TROX GmbH",
            3259 to "Emlid Tech Kft.",
            3258 to "Ameso Tech (OPC) Private Limited",
            3257 to "seca GmbH & Co. KG",
            3256 to "Shanghai Proxy Network Technology Co., Ltd.",
            3255 to "Cucumber Lighting Controls Limited",
            3254 to "THE EELECTRIC MACARON LLC",
            3253 to "Racketry, d. o. o.",
            3252 to "Eberspaecher Climate Control Systems GmbH",
            3251 to "janova GmbH",
            3250 to "SHINKAWA Sensor Technology, Inc.",
            3249 to "RF Creations",
            3248 to "SwipeSense, Inc.",
            3247 to "NEURINNOV",
            3246 to "Evident Corporation",
            3245 to "Shenzhen Openhearing Tech CO., LTD .",
            3244 to "Shenzhen Shokz Co.,Ltd.",
            3243 to "HERUTU ELECTRONICS CORPORATION",
            3242 to "Shenzhen Poseidon Network Technology Co., Ltd",
            3241 to "Mievo Technologies Private Limited",
            3240 to "Sonas, Inc.",
            3239 to "Verve InfoTec Pty Ltd",
            3238 to "Megger Ltd",
            3237 to "Princess Cruise Lines, Ltd.",
            3236 to "MITSUBISHI ELECTRIC LIGHTING CO, LTD",
            3235 to "MAQUET GmbH",
            3234 to "XSENSE LTD",
            3233 to "YAMAHA MOTOR CO.,LTD.",
            3232 to "BIGBEN",
            3231 to "Dragonfly Energy Corp.",
            3230 to "ECCEL CORPORATION SAS",
            3229 to "Ribbiot, INC.",
            3228 to "Sunstone-RTLS Ipari Szolgaltato Korlatolt Felelossegu Tarsasag",
            3227 to "NTT sonority, Inc.",
            3226 to "ALF Inc.",
            3225 to "Vire Health Oy",
            3224 to "MiX Telematics International (PTY) LTD",
            3223 to "Deako",
            3222 to "H+B Hightech GmbH",
            3221 to "Gemstone Lights Canada Ltd.",
            3220 to "Baxter Healthcare Corporation",
            3219 to "Movesense Oy",
            3218 to "KessebÃ¶hmer Ergonomietechnik GmbH",
            3217 to "Yashu Systems",
            3216 to "WESCO AG",
            3215 to "Radar Automobile Sales(Shandong)Co.,Ltd.",
            3214 to "Technocon Engineering Ltd.",
            3213 to "tonies GmbH",
            3212 to "T-Mobile USA",
            3211 to "Heavys Inc",
            3210 to "A.GLOBAL co.,Ltd.",
            3209 to "AGZZX OPTOELECTRONICS TECHNOLOGY CO., LTD",
            3208 to "Nextivity Inc.",
            3207 to "Weltek Technologies Company Limited",
            3206 to "Qingdao Eastsoft Communication Technology Co.,Ltd",
            3205 to "Amlogic, Inc.",
            3204 to "MAXON INDUSTRIES, INC.",
            3203 to "Watchdog Systems LLC",
            3202 to "NACON",
            3201 to "Carrier Corporation",
            3200 to "CARDIOID - TECHNOLOGIES, LDA",
            3199 to "Rochester Sensors, LLC",
            3198 to "BOOMING OF THINGS",
            3197 to "3ALogics, Inc.",
            3196 to "Mopeka Products LLC",
            3195 to "PT SADAMAYA GRAHA TEKNOLOGI",
            3194 to "Triductor Technology (Suzhou), Inc.",
            3193 to "Zhuhai Smartlink Technology Co., Ltd",
            3192 to "CHARGTRON IOT PRIVATE LIMITED",
            3191 to "TEAC Corporation",
            3190 to "Shenzhen Gwell Times Technology Co. , Ltd",
            3189 to "Embedded Engineering Solutions LLC",
            3188 to "yupiteru",
            3187 to "Truma GerÃ¤tetechnik GmbH & Co. KG",
            3186 to "StreetCar ORV, LLC",
            3185 to "BitGreen Technolabz (OPC) Private Limited",
            3184 to "SCARAB SOLUTIONS LTD",
            3183 to "Parakey AB",
            3182 to "Sensa LLC",
            3181 to "Fidure Corp.",
            3180 to "SNIFF LOGIC LTD",
            3179 to "GILSON SAS",
            3178 to "CONSORCIO TRUST CONTROL - NETTEL",
            3177 to "BLITZ electric motors. LTD",
            3176 to "Emerja Corporation",
            3175 to "TRACKTING S.R.L.",
            3174 to "DEN Smart Home B.V.",
            3173 to "WAKO CO,.LTD",
            3172 to "dormakaba Holding AG",
            3171 to "phg Peter Hengstler GmbH + Co. KG",
            3170 to "Phiaton Corporation",
            3169 to "NNOXX, Inc",
            3168 to "KEBA Energy Automation GmbH",
            3167 to "Wuxi Linkpower Microelectronics Co.,Ltd",
            3166 to "BlueID GmbH",
            3165 to "StepUp Solutions ApS",
            3164 to "MGM WIRELESSS HOLDINGS PTY LTD",
            3163 to "Alban Giacomo S.P.A.",
            3162 to "Lockswitch Sdn Bhd",
            3161 to "CYBERDYNE Inc.",
            3160 to "Hykso Inc.",
            3159 to "UNEEG medical A/S",
            3158 to "Rheem Sales Company, Inc.",
            3157 to "Zintouch B.V.",
            3156 to "HiViz Lighting, Inc.",
            3155 to "Taco, Inc.",
            3154 to "ESCEA LIMITED",
            3153 to "INNOVA S.R.L.",
            3152 to "Imostar Technologies Inc.",
            3151 to "SharkNinja Operating LLC",
            3150 to "Tactile Engineering, Inc.",
            3149 to "Seekwave Technology Co.,ltd.",
            3148 to "Orpyx Medical Technologies Inc.",
            3147 to "ADTRAN, Inc.",
            3146 to "atSpiro ApS",
            3145 to "twopounds gmbh",
            3144 to "VALEO MANAGEMENT SERVICES",
            3143 to "Epsilon Electronics,lnc",
            3142 to "Granwin IoT Technology (Guangzhou) Co.,Ltd",
            3141 to "Brose Verwaltung SE, Bamberg",
            3140 to "ONCELABS LLC",
            3139 to "Berlinger & Co. AG",
            3138 to "Heath Consultants Inc.",
            3137 to "Control Solutions LLC",
            3136 to "HORIBA, Ltd.",
            3135 to "Stinger Equipment, Inc.",
            3134 to "BELLDESIGN Inc.",
            3133 to "Wrmth Corp.",
            3132 to "Classified Cycling",
            3131 to "ORB Innovations Ltd",
            3130 to "Equinosis, LLC",
            3129 to "TIGER CORPORATION",
            3128 to "Noritz Corporation.",
            3127 to "SignalQuest, LLC",
            3126 to "Cosmicnode BV",
            3125 to "Thermokon-Sensortechnik GmbH",
            3124 to "BYD Company Limited",
            3123 to "Exeger Operations AB",
            3122 to "Xian Yisuobao Electronic Technology Co., Ltd.",
            3121 to "KINDOO LLP",
            3120 to "McIntosh Group Inc",
            3119 to "BEEHERO, INC.",
            3118 to "Easee AS",
            3117 to "OTF Product Sourcing, LLC",
            3116 to "Zeku Technology (Shanghai) Corp., Ltd.",
            3115 to "GigaDevice Semiconductor Inc.",
            3114 to "Caresix Inc.",
            3113 to "DENSO AIRCOOL CORPORATION",
            3112 to "Embecta Corp.",
            3111 to "Pal Electronics",
            3110 to "Performance Electronics, Ltd.",
            3109 to "JURA Elektroapparate AG",
            3108 to "SMARTD TECHNOLOGIES INC.",
            3107 to "KEYTEC,Inc.",
            3106 to "Glamo Inc.",
            3105 to "Foshan Viomi Electrical Technology Co., Ltd",
            3104 to "COMELIT GROUP S.P.A.",
            3103 to "LVI Co.",
            3102 to "EC sense co., Ltd",
            3101 to "OFF Line Japan Co., Ltd.",
            3100 to "GEMU",
            3099 to "Rockchip Electronics Co., Ltd.",
            3098 to "Catapult Group International Ltd",
            3097 to "Arlo Technologies, Inc.",
            3096 to "CORROHM",
            3095 to "SomnoMed Limited",
            3094 to "TYKEE PTY. LTD.",
            3093 to "Geva Sol B.V.",
            3092 to "Fasetto, Inc.",
            3091 to "Scandinavian Health Limited",
            3090 to "IoSA",
            3089 to "Gordon Murray Design Limited",
            3088 to "Cosmed s.r.l.",
            3087 to "AETERLINK",
            3086 to "ALEX DENKO CO.,LTD.",
            3085 to "Mereltron bv",
            3084 to "Mendeltron, Inc.",
            3083 to "aconno GmbH",
            3082 to "Automated Pet Care Products, LLC",
            3081 to "Senic Inc.",
            3080 to "limited liability company \"Red\"",
            3079 to "CONSTRUKTS, INC.",
            3078 to "LED Smart Inc.",
            3077 to "Montage Connect, Inc.",
            3076 to "Happy Health, Inc.",
            3075 to "Puff Corp",
            3074 to "Loomanet, Inc.",
            3073 to "NEOWRK SISTEMAS INTELIGENTES S.A.",
            3072 to "MQA Limited",
            3071 to "Ratio Electric BV",
            3070 to "Media-Cartec GmbH",
            3069 to "EsmÃ© Solutions",
            3068 to "T+A elektroakustik GmbH & Co.KG",
            3067 to "Dodam Enersys Co., Ltd",
            3066 to "CleanBands Systems Ltd.",
            3065 to "Alio, Inc",
            3064 to "Innovacionnye Resheniya",
            3063 to "Wacker Neuson SE",
            3062 to "greenTEG AG",
            3061 to "T5 tek, Inc.",
            3060 to "ER Lab LLC",
            3059 to "PONE BIOMETRICS AS",
            3058 to "Angel Medical Systems, Inc.",
            3057 to "Site IQ LLC",
            3056 to "KIDO SPORTS CO., LTD.",
            3055 to "Safetytest GmbH",
            3054 to "LINKSYS USA, INC.",
            3053 to "CORAL-TAIYI Co. Ltd.",
            3052 to "Miracle-Ear, Inc.",
            3051 to "Luna Health, Inc.",
            3050 to "Twenty Five Seven, prodaja in storitve, d.o.o.",
            3049 to "Shindengen Electric Manufacturing Co., Ltd.",
            3048 to "Sensormate AG",
            3047 to "Fresnel Technologies, Inc.",
            3046 to "Puratap Pty Ltd",
            3045 to "ZWILLING J.A. Henckels Aktiengesellschaft",
            3044 to "Deepfield Connect GmbH",
            3043 to "Comtel Systems Ltd.",
            3042 to "OTC engineering",
            3041 to "Back40 Precision",
            3040 to "Koizumi Lighting Technology corp.",
            3039 to "WINKEY ENTERPRISE (HONG KONG) LIMITED",
            3038 to "Yale",
            3037 to "Coroflo Limited",
            3036 to "Ledworks S.r.l.",
            3035 to "CHAR-BROIL, LLC",
            3034 to "Aardex Ltd.",
            3033 to "Elics Basis Ltd.",
            3032 to "PURA SCENTS, INC.",
            3031 to "VINFAST TRADING AND PRODUCTION JOINT STOCK COMPANY",
            3030 to "Shenzhen Injoinic Technology Co., Ltd.",
            3029 to "Super B Lithium Power B.V.",
            3028 to "ndd Medizintechnik AG",
            3027 to "Procon Analytics, LLC",
            3026 to "IDEC",
            3025 to "Hubei Yuan Times Technology Co., Ltd.",
            3024 to "Durag GmbH",
            3023 to "LL Tec Group LLC",
            3022 to "Neurosity, Inc.",
            3021 to "Amiko srl",
            3020 to "Sylvac sa",
            3019 to "Divesoft s.r.o.",
            3018 to "Perimeter Technologies, Inc.",
            3017 to "Neuvatek Inc.",
            3016 to "OTF Distribution, LLC",
            3015 to "Signtle Inc.",
            3014 to "TCL COMMUNICATION EQUIPMENT CO.,LTD.",
            3013 to "Aperia Technologies, Inc.",
            3012 to "TECHTICS ENGINEERING B.V.",
            3011 to "MCOT INC.",
            3010 to "EntWick Co.",
            3009 to "Miele & Cie. KG",
            3008 to "READY FOR SKY LLP",
            3007 to "HIMSA II K/S",
            3006 to "SAAB Aktiebolag",
            3005 to "ETHEORY PTY LTD",
            3004 to "T2REALITY SOLUTIONS PRIVATE LIMITED",
            3003 to "SWISSINNO SOLUTIONS AG",
            3002 to "Huso, INC",
            3001 to "SaluStim Group Oy",
            3000 to "INNOVAG PTY. LTD.",
            2999 to "IONA Tech LLC",
            2998 to "Build With Robots Inc.",
            2997 to "Xirgo Technologies, LLC",
            2996 to "New Cosmos USA, Inc.",
            2995 to "Flender GmbH",
            2994 to "Fjorden Electra AS",
            2993 to "Beijing ranxin intelligence technology Co.,LTD",
            2992 to "Ecolab Inc.",
            2991 to "NITTO KOGYO CORPORATION",
            2990 to "Soma Labs LLC",
            2989 to "Roambotics, Inc.",
            2988 to "Machfu Inc.",
            2987 to "Grandex International Corporation",
            2986 to "Infinitegra, Inc.",
            2985 to "Allterco Robotics ltd",
            2984 to "GLOWFORGE INC.",
            2983 to "hearX Group (Pty) Ltd",
            2982 to "Nissan Motor Co., Ltd.",
            2981 to "SONICOS ENTERPRISES, LLC",
            2980 to "Vervent Audio Group",
            2979 to "Sonova Consumer Hearing GmbH",
            2978 to "TireCheck GmbH",
            2977 to "Bunn-O-Matic Corporation",
            2976 to "Data Sciences International",
            2975 to "Group Lotus Limited",
            2974 to "Audio Partnership Plc",
            2973 to "Sensoria Holdings LTD",
            2972 to "Komatsu Ltd.",
            2971 to "GISMAN",
            2970 to "Beijing Wisepool Infinite Intelligence Technology Co.,Ltd",
            2969 to "The Goodyear Tire & Rubber Company",
            2968 to "Gymstory B.V.",
            2967 to "SILVER TREE LABS, INC.",
            2966 to "Telecom Design",
            2965 to "Netwake GmbH",
            2964 to "Dreem SAS",
            2963 to "Hangzhou BroadLink Technology Co., Ltd.",
            2962 to "Citisend Solutions, SL",
            2961 to "Alfen ICU B.V.",
            2960 to "Ineos Automotive Limited",
            2959 to "Senscomm Semiconductor Co., Ltd.",
            2958 to "Gentle Energy Corp.",
            2957 to "Pertech Industries Inc",
            2956 to "MOTREX",
            2955 to "American Technology Components, Incorporated",
            2954 to "Seiko Instruments Inc.",
            2953 to "Rotronic AG",
            2952 to "Muguang (Guangdong) Intelligent Lighting Technology Co., Ltd",
            2951 to "Ampetronic Ltd",
            2950 to "Trek Bicycle",
            2949 to "VIMANA TECH PTY LTD",
            2948 to "Presidio Medical, Inc.",
            2947 to "Taiga Motors Inc.",
            2946 to "Mammut Sports Group AG",
            2945 to "SCM Group",
            2944 to "AXELIFE",
            2943 to "ICU tech GmbH",
            2942 to "Offcode Oy",
            2941 to "FoundersLane GmbH",
            2940 to "Scangrip A/S",
            2939 to "Hardcoder Oy",
            2938 to "Shenzhen KTC Technology Co.,Ltd.",
            2937 to "Sankyo Air Tech Co.,Ltd.",
            2936 to "FIELD DESIGN INC.",
            2935 to "Aixlink(Chengdu) Co., Ltd.",
            2934 to "MAX-co., ltd",
            2933 to "Triple W Japan Inc.",
            2932 to "BQN",
            2931 to "HARADA INDUSTRY CO., LTD.",
            2930 to "Geeknet, Inc.",
            2929 to "lilbit ODM AS",
            2928 to "JDRF Electromag Engineering Inc",
            2927 to "Shenzhen Malide Technology Co.,Ltd",
            2926 to "React Mobile",
            2925 to "SOLUM CO., LTD",
            2924 to "Sensitech, Inc.",
            2923 to "Samsara Networks, Inc",
            2922 to "Dymo",
            2921 to "Addaday",
            2920 to "Quha oy",
            2919 to "CleanSpace Technology Pty Ltd",
            2918 to "MITSUBISHI ELECTRIC AUTOMATION (THAILAND) COMPANY LIMITED",
            2917 to "The Apache Software Foundation",
            2916 to "NingBo klite Electric Manufacture Co.,LTD",
            2915 to "Innolux Corporation",
            2914 to "NOVEA ENERGIES",
            2913 to "Sentek Pty Ltd",
            2912 to "RATOC Systems, Inc.",
            2911 to "Rivieh, Inc.",
            2910 to "CELLCONTROL, INC.",
            2909 to "Fujian Newland Auto-ID Tech. Co., Ltd.",
            2908 to "Exponential Power, Inc.",
            2907 to "Shenzhen ImagineVision Technology Limited",
            2906 to "H.P. Shelby Manufacturing, LLC.",
            2905 to "Versa Group B.V.",
            2904 to "TOKAI-DENSHI INC",
            2903 to "CONVERTRONIX TECHNOLOGIES AND SERVICES LLP",
            2902 to "BORA - Vertriebs GmbH & Co KG",
            2901 to "H G M Automotive Electronics, Inc.",
            2900 to "Emotion Fitness GmbH & Co. KG",
            2899 to "SHENZHEN KAADAS INTELLIGENT TECHNOLOGY CO.,Ltd",
            2898 to "ZIIP Inc",
            2897 to "FUN FACTORY GmbH",
            2896 to "Mesh Systems LLC",
            2895 to "Breezi.io, Inc.",
            2894 to "ICP Systems B.V.",
            2893 to "Adam Hall GmbH",
            2892 to "BiosBob.Biz",
            2891 to "EMS Integrators, LLC",
            2890 to "Nomono AS",
            2889 to "SkyHawke Technologies",
            2888 to "NIO USA, Inc.",
            2887 to "Gentex Corporation",
            2886 to "Bird Rides, Inc.",
            2885 to "Electronic Sensors, Inc.",
            2884 to "nFore Technology Co., Ltd.",
            2883 to "INCITAT ENVIRONNEMENT",
            2882 to "TSI",
            2881 to "Sentrax GmbH",
            2880 to "Havells India Limited",
            2879 to "MindRhythm, Inc.",
            2878 to "ISEO Serrature S.p.a.",
            2877 to "REALTIMEID AS",
            2876 to "Dodge Industrial, Inc.",
            2875 to "AIC semiconductor (Shanghai) Co., Ltd.",
            2874 to "Impact Biosystems, Inc.",
            2873 to "Red 100 Lighting Co., ltd.",
            2872 to "WISYCOM S.R.L.",
            2871 to "Omnivoltaic Energy Solutions Limited Company",
            2870 to "SINTEF",
            2869 to "BH SENS",
            2868 to "CONZUMEX INDUSTRIES PRIVATE LIMITED",
            2867 to "ARMATURA LLC",
            2866 to "Hala Systems, Inc.",
            2865 to "Silver Wolf Vehicles Inc.",
            2864 to "ART SPA",
            2863 to "Duke Manufacturing Co",
            2862 to "MOCA System Inc.",
            2861 to "REDARC ELECTRONICS PTY LTD",
            2860 to "ILLUMAGEAR, Inc.",
            2859 to "MAINBOT",
            2858 to "ACL Airshop B.V.",
            2857 to "Tech-Venom Entertainment Private Limited",
            2856 to "CHACON",
            2855 to "Lumi United Technology Co., Ltd",
            2854 to "Baracoda Daily Healthtech.",
            2853 to "NIBROTECH LTD",
            2852 to "BeiJing ZiJie TiaoDong KeJi Co.,Ltd.",
            2851 to "iRhythm Technologies, Inc.",
            2850 to "Hygiene IQ, LLC.",
            2849 to "ams AG",
            2848 to "TKH Security B.V.",
            2847 to "Beijing ESWIN Computing Technology Co., Ltd.",
            2846 to "PB INC.",
            2845 to "Accelerated Systems",
            2844 to "Nanoleq AG",
            2843 to "Enerpac Tool Group Corp.",
            2842 to "Roca Sanitario, S.A.",
            2841 to "WBS PROJECT H PTY LTD",
            2840 to "DECATHLON SE",
            2839 to "SIG SAUER, INC.",
            2838 to "Guard RFID Solutions Inc.",
            2837 to "NAOS JAPAN K.K.",
            2836 to "Olumee",
            2835 to "IOTOOLS",
            2834 to "ToughBuilt Industries LLC",
            2833 to "ThermoWorks, Inc.",
            2832 to "Alfa Laval Corporate AB",
            2831 to "B.E.A. S.A.",
            2830 to "Honda Lock Mfg. Co.,Ltd.",
            2829 to "SANYO DENKO Co.,Ltd.",
            2828 to "BluPeak",
            2827 to "Sanistaal A/S",
            2826 to "Belun Technology Company Limited",
            2825 to "soonisys",
            2824 to "Shenzhen Qianfenyi Intelligent Technology Co., LTD",
            2823 to "Workaround Gmbh",
            2822 to "FAZUA GmbH",
            2821 to "Marquardt GmbH",
            2820 to "I-PERCUT",
            2819 to "Precision Triathlon Systems Limited",
            2818 to "IORA Technology Development Ltd. Sti.",
            2817 to "RESIDEO TECHNOLOGIES, INC.",
            2816 to "Flaircomm Microelectronics Inc.",
            2815 to "FUSEAWARE LIMITED",
            2814 to "Earda Technologies Co.,Ltd",
            2813 to "Weber Sensors, LLC",
            2812 to "Cerebrum Sensor Technologies Inc.",
            2811 to "SMT ELEKTRONIK GmbH",
            2810 to "Chengdu Ambit Technology Co., Ltd.",
            2809 to "Unisto AG",
            2808 to "First Design System Inc.",
            2807 to "Irdeto",
            2806 to "AMETEK, Inc.",
            2805 to "Unitech Electronic Inc.",
            2804 to "Radioworks Microelectronics PTY LTD",
            2803 to "701x Inc.",
            2802 to "Shanghai All Link Microelectronics Co.,Ltd",
            2801 to "CRADERS,CO.,LTD",
            2800 to "Leupold & Stevens, Inc.",
            2799 to "GLP German Light Products GmbH",
            2798 to "Velentium, LLC",
            2797 to "Saxonar GmbH",
            2796 to "FUTEK ADVANCED SENSOR TECHNOLOGY, INC",
            2795 to "Square, Inc.",
            2794 to "Borda Technology",
            2793 to "FLIR Systems AB",
            2792 to "LEVEL, s.r.o.",
            2791 to "Sunplus Technology Co., Ltd.",
            2790 to "Hexology",
            2789 to "unu GmbH",
            2788 to "DALI Alliance",
            2787 to "GlobalMed",
            2786 to "IMATRIX SYSTEMS, INC.",
            2785 to "ChengDu ForThink Technology Co., Ltd.",
            2784 to "Viceroy Devices Corporation",
            2783 to "Douglas Dynamics L.L.C.",
            2782 to "Vocera Communications, Inc.",
            2781 to "Boss Audio",
            2780 to "Duravit AG",
            2779 to "Reelables, Inc.",
            2778 to "Codefabrik GmbH",
            2777 to "Shenzhen Aimore. Co.,Ltd",
            2776 to "Franz Kaldewei GmbH&Co KG",
            2775 to "AL-KO Geraete GmbH",
            2774 to "nymea GmbH",
            2773 to "Streamit B.V.",
            2772 to "Zhuhai Pantum Electronisc Co., Ltd",
            2771 to "SSV Software Systems GmbH",
            2770 to "Lautsprecher Teufel GmbH",
            2769 to "EAGLE KINGDOM TECHNOLOGIES LIMITED",
            2768 to "Nordic Strong ApS",
            2767 to "CACI Technologies",
            2766 to "KOBATA GAUGE MFG. CO., LTD.",
            2765 to "Visuallex Sport International Limited",
            2764 to "Nuvoton",
            2763 to "ise Individuelle Software und Elektronik GmbH",
            2762 to "Shenzhen CoolKit Technology Co., Ltd",
            2761 to "Swedlock AB",
            2760 to "Keepin Co., Ltd.",
            2759 to "Chengdu Aich Technology Co.,Ltd",
            2758 to "Barnes Group Inc.",
            2757 to "Flexoptix GmbH",
            2756 to "CODIUM",
            2755 to "Kenzen, Inc.",
            2754 to "RealMega Microelectronics technology (Shanghai) Co. Ltd.",
            2753 to "Shenzhen Jingxun Technology Co., Ltd.",
            2752 to "Omni-ID USA, INC.",
            2751 to "PAUL HARTMANN AG",
            2750 to "Robkoo Information & Technologies Co., Ltd.",
            2749 to "Inventas AS",
            2748 to "KCCS Mobile Engineering Co., Ltd.",
            2747 to "R-DAS, s.r.o.",
            2746 to "Open Bionics Ltd.",
            2745 to "STL",
            2744 to "Sens.ai Incorporated",
            2743 to "LogTag North America Inc.",
            2742 to "Xenter, Inc.",
            2741 to "Elstat Electronics Ltd.",
            2740 to "Ellenby Technologies, Inc.",
            2739 to "INNER RANGE PTY. LTD.",
            2738 to "TouchTronics, Inc.",
            2737 to "InVue Security Products Inc",
            2736 to "Visiontronic s.r.o.",
            2735 to "AIAIAI ApS",
            2734 to "PS Engineering, Inc.",
            2733 to "Adevo Consulting AB",
            2732 to "OSM HK Limited",
            2731 to "Anhui Listenai Co",
            2730 to "Computime International Ltd",
            2729 to "Spintly, Inc.",
            2728 to "Zencontrol Pty Ltd",
            2727 to "Urbanista AB",
            2726 to "Realityworks, inc.",
            2725 to "Shenzhen Uascent Technology Co., Ltd",
            2724 to "FAZEPRO LLC",
            2723 to "DIC Corporation",
            2722 to "Care Bloom, LLC",
            2721 to "LINCOGN TECHNOLOGY CO. LIMITED",
            2720 to "Loy Tec electronics GmbH",
            2719 to "ista International GmbH",
            2718 to "LifePlus, Inc.",
            2717 to "Canon Finetech Nisca Inc.",
            2716 to "Xi'an Fengyu Information Technology Co., Ltd.",
            2715 to "Eello LLC",
            2714 to "TEMKIN ASSOCIATES, LLC",
            2713 to "Shanghai high-flying electronics technology Co.,Ltd",
            2712 to "Foil, Inc.",
            2711 to "SensTek",
            2710 to "Lightricity Ltd",
            2709 to "Pamex Inc.",
            2708 to "OOBIK Inc.",
            2707 to "GiPStech S.r.l.",
            2706 to "Carestream Dental LLC",
            2705 to "Monarch International Inc.",
            2704 to "Shenzhen Grandsun Electronic Co.,Ltd.",
            2703 to "TOTO LTD.",
            2702 to "Perfect Company",
            2701 to "JCM TECHNOLOGIES S.A.",
            2700 to "DelpSys, s.r.o.",
            2699 to "SANlight GmbH",
            2698 to "HAINBUCH GMBH SPANNENDE TECHNIK",
            2697 to "SES-Imagotag",
            2696 to "PSA Peugeot Citroen",
            2695 to "Shanghai Smart System Technology Co., Ltd",
            2694 to "ALIZENT International",
            2693 to "Snowball Technology Co., Ltd.",
            2692 to "Greennote Inc,",
            2691 to "Rivata, Inc.",
            2690 to "Corsair",
            2689 to "Universal Biosensors Pty Ltd",
            2688 to "Cleer Limited",
            2687 to "Intuity Medical",
            2686 to "KEBA Handover Automation GmbH",
            2685 to "Freedman Electronics Pty Ltd",
            2684 to "WAFERLOCK",
            2683 to "UniqAir Oy",
            2682 to "Emlid Limited",
            2681 to "Webasto SE",
            2680 to "Shenzhen Sunricher Technology Limited",
            2679 to "AXTRO PTE. LTD.",
            2678 to "Synaptics Incorporated",
            2677 to "Delta Cycle Corporation",
            2676 to "MICROSON S.A.",
            2675 to "Innohome Oy",
            2674 to "Jumo GmbH & Co. KG",
            2673 to "Senquip Pty Ltd",
            2672 to "Ooma",
            2671 to "Warner Bros.",
            2670 to "Pac Sane Limited",
            2669 to "KUUKANJYOKIN Co.,Ltd.",
            2668 to "Pokkels",
            2667 to "Olympic Ophthalmics, Inc.",
            2666 to "Scribble Design Inc.",
            2665 to "HAPPIEST BABY, INC.",
            2664 to "Focus Ingenieria SRL",
            2663 to "Beijing SuperHexa Century Technology CO. Ltd",
            2662 to "JUSTMORPH PTE. LTD.",
            2661 to "Lytx, INC.",
            2660 to "Geopal system A/S",
            2659 to "Gremsy JSC",
            2658 to "MOKO TECHNOLOGY Ltd",
            2657 to "Smart Parks B.V.",
            2656 to "DATANG SEMICONDUCTOR TECHNOLOGY CO.,LTD",
            2655 to "stryker",
            2654 to "LaceClips llc",
            2653 to "MG Energy Systems B.V.",
            2652 to "Innovative Design Labs Inc.",
            2651 to "LEGIC Identsystems AG",
            2650 to "Sontheim Industrie Elektronik GmbH",
            2649 to "TourBuilt, LLC",
            2648 to "Indigo Diabetes",
            2647 to "Meizhou Guo Wei Electronics Co., Ltd",
            2646 to "ambie",
            2645 to "Inugo Systems Limited",
            2644 to "SQL Technologies Corp.",
            2643 to "KKM COMPANY LIMITED",
            2642 to "Follow Sense Europe B.V.",
            2641 to "CSIRO",
            2640 to "Nextscape Inc.",
            2639 to "VANMOOF Global Holding B.V.",
            2638 to "Toytec Corporation",
            2637 to "Lockn Technologies Private Limited",
            2636 to "SiFli Technologies (shanghai) Inc.",
            2635 to "MistyWest Energy and Transport Ltd.",
            2634 to "Map Large, Inc.",
            2633 to "Venture Research Inc.",
            2632 to "JRC Mobility Inc.",
            2631 to "The Wand Company Ltd",
            2630 to "Beijing HC-Infinite Technology Limited",
            2629 to "3SI Security Systems, Inc",
            2628 to "Novidan, Inc.",
            2627 to "Busch Systems International Inc.",
            2626 to "Motionalysis, Inc.",
            2625 to "OPEX Corporation",
            2624 to "GEWISS S.p.A.",
            2623 to "Shenzhen Yopeak Optoelectronics Technology Co., Ltd.",
            2622 to "Hefei Yunlian Semiconductor Co., Ltd",
            2621 to "DELABIE",
            2620 to "Siteco GmbH",
            2619 to "Galileo Technology Limited",
            2618 to "Incotex Co. Ltd.",
            2617 to "BLUETICKETING SRL",
            2616 to "Bouffalo Lab (Nanjing)., Ltd.",
            2615 to "2587702 Ontario Inc.",
            2614 to "NGK SPARK PLUG CO., LTD.",
            2613 to "safectory GmbH",
            2612 to "Luxer Corporation",
            2611 to "WMF AG",
            2610 to "Pinnacle Technology, Inc.",
            2609 to "Nevro Corp.",
            2608 to "Air-Weigh",
            2607 to "Instamic, Inc.",
            2606 to "Zuma Array Limited",
            2605 to "Shenzhen Feasycom Technology Co., Ltd.",
            2604 to "Shenzhen H&T Intelligent Control Co., Ltd",
            2603 to "PaceBait IVS",
            2602 to "Yamaha Corporation",
            2601 to "Worthcloud Technology Co.,Ltd",
            2600 to "NanoFlex Power Corporation",
            2599 to "AYU DEVICES PRIVATE LIMITED",
            2598 to "Louis Vuitton",
            2597 to "Eran Financial Services LLC",
            2596 to "Atmosic Technologies, Inc.",
            2595 to "BIXOLON CO.,LTD",
            2594 to "DAIICHIKOSHO CO., LTD.",
            2593 to "Apollogic Sp. z o.o.",
            2592 to "Jiangxi Innotech Technology Co., Ltd",
            2591 to "DeVilbiss Healthcare LLC",
            2590 to "CombiQ AB",
            2589 to "API-K",
            2588 to "INPEAK S.C.",
            2587 to "Embrava Pty Ltd",
            2586 to "Link Labs, Inc.",
            2585 to "Maxell, Ltd.",
            2584 to "Cambridge Animal Technologies Ltd",
            2583 to "Plume Design Inc",
            2582 to "RIDE VISION LTD",
            2581 to "Syng Inc",
            2580 to "CROXEL, INC.",
            2579 to "Tec4med LifeScience GmbH",
            2578 to "Dyson Technology Limited",
            2577 to "Sensolus",
            2576 to "SUBARU Corporation",
            2575 to "LIXIL Corporation",
            2574 to "Roland Corporation",
            2573 to "Blue Peacock GmbH",
            2572 to "Shanghai Yidian Intelligent Technology Co., Ltd.",
            2571 to "SIANA Systems",
            2570 to "Volan Technology Inc.",
            2569 to "ECCT",
            2568 to "Oras Oy",
            2567 to "Reflow Pty Ltd",
            2566 to "Shanghai wuqi microelectronics Co.,Ltd",
            2565 to "Southwire Company, LLC",
            2564 to "Flosonics Medical",
            2563 to "donutrobotics Co., Ltd.",
            2562 to "Ayxon-Dynamics GmbH",
            2561 to "Cleveron AS",
            2560 to "Ampler Bikes OU",
            2559 to "AIRSTAR",
            2558 to "Lichtvision Engineering GmbH",
            2557 to "Keep Technologies, Inc.",
            2556 to "Confidex",
            2555 to "TOITU CO., LTD.",
            2554 to "Listen Technologies Corporation",
            2553 to "Hangzhou Yaguan Technology Co. LTD",
            2552 to "R.O. S.R.L.",
            2551 to "SENSATEC Co., Ltd.",
            2550 to "Mobile Action Technology Inc.",
            2549 to "OKI Electric Industry Co., Ltd",
            2548 to "Spectrum Technologies, Inc.",
            2547 to "Beijing Zero Zero Infinity Technology Co.,Ltd.",
            2546 to "Audeara Pty Ltd",
            2545 to "OM Digital Solutions Corporation",
            2544 to "WatchGas B.V.",
            2543 to "Steinel Solutions AG",
            2542 to "OJMAR SA",
            2541 to "Sibel Inc.",
            2540 to "Yukon advanced optics worldwide, UAB",
            2539 to "KEAN ELECTRONICS PTY LTD",
            2538 to "Athlos Oy",
            2537 to "LumenRadio AB",
            2536 to "Melange Systems Pvt. Ltd.",
            2535 to "Kabushikigaisha HANERON",
            2534 to "Masonite Corporation",
            2533 to "Mobilogix",
            2532 to "CPS AS",
            2531 to "Friday Home Aps",
            2530 to "Wuhan Linptech Co.,Ltd.",
            2529 to "Tag-N-Trac Inc",
            2528 to "Preddio Technologies Inc.",
            2527 to "Magnus Technology Sdn Bhd",
            2526 to "JLD Technology Solutions, LLC",
            2525 to "Innoware Development AB",
            2524 to "AON2 Ltd.",
            2523 to "Bionic Avionics Inc.",
            2522 to "Nagravision SA",
            2521 to "VivoSensMedical GmbH",
            2520 to "Synergy Tecnologia em Sistemas Ltda",
            2519 to "Coyotta",
            2518 to "EAR TEKNIK ISITME VE ODIOMETRI CIHAZLARI SANAYI VE TICARET ANONIM SIRKETI",
            2517 to "GEAR RADIO ELECTRONICS CORP.",
            2516 to "ORBIS Inc.",
            2515 to "HeartHero, inc.",
            2514 to "Temperature Sensitive Solutions Systems Sweden AB",
            2513 to "ABLEPAY TECHNOLOGIES AS",
            2512 to "Chess Wise B.V.",
            2511 to "BlueStreak IoT, LLC",
            2510 to "Julius Blum GmbH",
            2509 to "Blyott",
            2508 to "Senso4s d.o.o.",
            2507 to "Hx Engineering, LLC",
            2506 to "Mobitrace",
            2505 to "CrowdGlow Ltd",
            2504 to "XUNTONG",
            2503 to "Combustion, LLC",
            2502 to "Honor Device Co., Ltd.",
            2501 to "HungYi Microelectronics Co.,Ltd.",
            2500 to "UVISIO",
            2499 to "JAPAN TOBACCO INC.",
            2498 to "Universal Audio, Inc.",
            2497 to "Rosewill",
            2496 to "AnotherBrain inc.",
            2495 to "Span.IO, Inc.",
            2494 to "Vessel Ltd.",
            2493 to "Centre Suisse d'Electronique et de Microtechnique SA",
            2492 to "Aerosens LLC",
            2491 to "SkyStream Corporation",
            2490 to "Elimo Engineering Ltd",
            2489 to "SAVOY ELECTRONIC LIGHTING",
            2488 to "PlayerData Limited",
            2487 to "Bout Labs, LLC",
            2486 to "Pegasus Technologies, Inc.",
            2485 to "AUTEC Gesellschaft fuer Automationstechnik mbH",
            2484 to "PentaLock Aps.",
            2483 to "BlueX Microelectronics Corp Ltd.",
            2482 to "DYPHI",
            2481 to "BLINQY",
            2480 to "Deublin Company, LLC",
            2479 to "ifLink Open Community",
            2478 to "Pozyx NV",
            2477 to "Narhwall Inc.",
            2476 to "Ambiq",
            2475 to "DashLogic, Inc.",
            2474 to "PHOTODYNAMIC INCORPORATED",
            2473 to "Nippon Ceramic Co.,Ltd.",
            2472 to "KHN Solutions LLC",
            2471 to "Paybuddy ApS",
            2470 to "BEIJING ELECTRIC VEHICLE CO.,LTD",
            2469 to "Security Enhancement Systems, LLC",
            2468 to "KUMHO ELECTRICS, INC",
            2467 to "ARDUINO SA",
            2466 to "ENGAGENOW DATA SCIENCES PRIVATE LIMITED",
            2465 to "VOS Systems, LLC",
            2464 to "Proof Diagnostics, Inc.",
            2463 to "Koya Medical, Inc.",
            2462 to "Step One Limited",
            2461 to "YKK AP Inc.",
            2460 to "deister electronic GmbH",
            2459 to "Sendum Wireless Corporation",
            2458 to "New Audio LLC",
            2457 to "eTactica ehf",
            2456 to "Pixie Dust Technologies, Inc.",
            2455 to "NextMind",
            2454 to "C. & E. Fein GmbH",
            2453 to "Bronkhorst High-Tech B.V.",
            2452 to "VT42 Pty Ltd",
            2451 to "Absolute Audio Labs B.V.",
            2450 to "Big Kaiser Precision Tooling Ltd",
            2449 to "Telenor ASA",
            2448 to "Anton Paar GmbH",
            2447 to "Aktiebolaget Regin",
            2446 to "ADVEEZ",
            2445 to "C3-WIRELESS, LLC",
            2444 to "bGrid B.V.",
            2443 to "Mequonic Engineering, S.L.",
            2442 to "Biovigil",
            2441 to "WIKA Alexander Wiegand SE & Co.KG",
            2440 to "BHM-Tech Produktionsgesellschaft m.b.H",
            2439 to "TSE BRAKES, INC.",
            2438 to "Cello Hill, LLC",
            2437 to "Lumos Health Inc.",
            2436 to "TeraTron GmbH",
            2435 to "Feedback Sports LLC",
            2434 to "ELPRO-BUCHS AG",
            2433 to "Bernard Krone Holding SE & Co.KG",
            2432 to "DEKRA TESTING AND CERTIFICATION, S.A.U.",
            2431 to "ISEMAR S.R.L.",
            2430 to "SonicSensory Inc",
            2429 to "CLB B.V.",
            2428 to "Thorley Industries, LLC",
            2427 to "CTEK Sweden AB",
            2426 to "CORE CORPORATION",
            2425 to "BIOTRONIK SE & Co. KG",
            2424 to "ZifferEins GmbH & Co. KG",
            2423 to "TOYOTA motor corporation",
            2422 to "Fauna Audio GmbH",
            2421 to "BlueIOT(Beijing) Technology Co.,Ltd",
            2420 to "ABEYE",
            2419 to "Popit Oy",
            2418 to "Closed Joint Stock Company \"Zavod Flometr\" (\"Zavod Flometr\" CJSC)",
            2417 to "GA",
            2416 to "IBA Dosimetry GmbH",
            2415 to "Lund Motion Products, Inc.",
            2414 to "Band Industries, inc.",
            2413 to "Gunwerks, LLC",
            2412 to "9374-7319 Quebec inc",
            2411 to "Guide ID B.V.",
            2410 to "dricos, Inc.",
            2409 to "Woan Technology (Shenzhen) Co., Ltd.",
            2408 to "Actev Motors, Inc.",
            2407 to "Neo Materials and Consulting Inc.",
            2406 to "PointGuard, LLC",
            2405 to "Asahi Kasei Corporation",
            2404 to "Countrymate Technology Limited",
            2403 to "Moonbird BV",
            2402 to "GL Solutions K.K.",
            2401 to "Linkura AB",
            2400 to "Sena Technologies Inc.",
            2399 to "NUANCE HEARING LTD",
            2398 to "BioEchoNet inc.",
            2397 to "Electronic Theatre Controls",
            2396 to "LogiLube, LLC",
            2395 to "Lismore Instruments Limited",
            2394 to "Selekt Bilgisayar, lletisim Urunleri lnsaat Sanayi ve Ticaret Limited Sirketi",
            2393 to "HerdDogg, Inc",
            2392 to "ZTE Corporation",
            2391 to "Ohsung Electronics",
            2390 to "Kerlink",
            2389 to "Breville Group",
            2388 to "Julbo",
            2387 to "LogiLube, LLC",
            2386 to "Apptricity Corporation",
            2385 to "PPRS",
            2384 to "Capetech",
            2383 to "Limited Liability Company \"Mikrotikls\"",
            2382 to "PassiveBolt, Inc.",
            2381 to "tkLABS INC.",
            2380 to "GimmiSys GmbH",
            2379 to "Kindeva Drug Delivery L.P.",
            2378 to "Zwift, Inc.",
            2377 to "Metronom Health Europe",
            2376 to "Wearable Link Limited",
            2375 to "First Light Technologies Ltd.",
            2374 to "AMC International Alfa Metalcraft Corporation AG",
            2373 to "Globe (Jiangsu) Co., Ltd",
            2372 to "Agitron d.o.o.",
            2370 to "TRANSSION HOLDINGS LIMITED",
            2369 to "Rivian Automotive, LLC",
            2368 to "Hero Workout GmbH",
            2367 to "JEPICO Corporation",
            2366 to "Catalyft Labs, Inc.",
            2365 to "Adolf Wuerth GmbH & Co KG",
            2364 to "Xenoma Inc.",
            2363 to "ENSESO LLC",
            2362 to "LinkedSemi Microelectronics (Xiamen) Co., Ltd",
            2361 to "ASTEM Co.,Ltd.",
            2360 to "Henway Technologies, LTD.",
            2359 to "RealThingks GmbH",
            2358 to "Elekon AG",
            2357 to "Reconnect, Inc.",
            2356 to "KiteSpring Inc.",
            2355 to "SRAM",
            2354 to "BarVision, LLC",
            2353 to "BREATHINGS Co., Ltd.",
            2352 to "James Walker RotaBolt Limited",
            2351 to "C.O.B.O. SpA",
            2350 to "PS GmbH",
            2349 to "Leggett & Platt, Incorporated",
            2348 to "PCI Private Limited",
            2347 to "TekHome",
            2346 to "Sappl Verwaltungs- und Betriebs GmbH",
            2345 to "Qingdao Haier Technology Co., Ltd.",
            2344 to "AiRISTA",
            2343 to "ROOQ GmbH",
            2342 to "Gooligum Technologies Pty Ltd",
            2341 to "Yukai Engineering Inc.",
            2340 to "Fundacion Tecnalia Research and Innovation",
            2339 to "JSB TECH PTE LTD",
            2338 to "Shanghai MXCHIP Information Technology Co., Ltd.",
            2337 to "KAHA PTE. LTD.",
            2336 to "Omnisense Limited",
            2335 to "Myzee Technology",
            2334 to "Melbot Studios, Sociedad Limitada",
            2333 to "Innokind, Inc.",
            2332 to "Oblamatik AG",
            2331 to "Luminostics, Inc.",
            2330 to "Albertronic BV",
            2329 to "NO SMD LIMITED",
            2328 to "Technosphere Labs Pvt. Ltd.",
            2327 to "ASR Microelectronics(ShenZhen)Co., Ltd.",
            2326 to "Ambient Sensors LLC",
            2325 to "Honda Motor Co., Ltd.",
            2324 to "INEO-SENSE",
            2323 to "Braveheart Wireless, Inc.",
            2322 to "Nerbio Medical Software Platforms Inc",
            2321 to "Douglas Lighting Controls Inc.",
            2320 to "ASR Microelectronics (Shanghai) Co., Ltd.",
            2319 to "VC Inc.",
            2318 to "OPTIMUSIOT TECH LLP",
            2317 to "IOT Invent GmbH",
            2316 to "Radiawave Technologies Co.,Ltd.",
            2315 to "EMBR labs, INC",
            2314 to "Zhuhai Hoksi Technology CO.,LTD",
            2313 to "70mai Co.,Ltd.",
            2312 to "Pinpoint Innovations Limited",
            2311 to "User Hello, LLC",
            2310 to "Scope Logistical Solutions",
            2309 to "Yandex Services AG",
            2308 to "SUNCORPORATION",
            2307 to "DATAMARS, Inc.",
            2306 to "TSC Auto-ID Technology Co., Ltd.",
            2305 to "Lucimed",
            2304 to "Beijing Zizai Technology Co., LTD.",
            2303 to "Plastimold Products, Inc",
            2302 to "Ketronixs Sdn Bhd",
            2301 to "BioIntelliSense, Inc.",
            2300 to "Hill-Rom",
            2299 to "Darkglass Electronics Oy",
            2298 to "Troo Corporation",
            2297 to "Spacelabs Medical Inc.",
            2296 to "instagrid GmbH",
            2295 to "MTD Products Inc & Affiliates",
            2294 to "Dermal Photonics Corporation",
            2293 to "Tymtix Technologies Private Limited",
            2292 to "Kodimo Technologies Company Limited",
            2291 to "PSP - Pauli Services & Products GmbH",
            2290 to "Microoled",
            2289 to "The L.S. Starrett Company",
            2288 to "Joovv, Inc.",
            2287 to "Cumulus Digital Systems, Inc",
            2286 to "Askey Computer Corp.",
            2285 to "IMI Hydronic Engineering International SA",
            2284 to "Denso Corporation",
            2283 to "Beijing Big Moment Technology Co., Ltd.",
            2282 to "COWBELL ENGINEERING CO.,LTD.",
            2281 to "Taiwan Intelligent Home Corp.",
            2280 to "Naonext",
            2279 to "Barrot Technology Co.,Ltd.",
            2278 to "Eneso Tecnologia de Adaptacion S.L.",
            2277 to "Crowd Connected Ltd",
            2276 to "Rashidov ltd",
            2275 to "Republic Wireless, Inc.",
            2274 to "Shenzhen Simo Technology co. LTD",
            2273 to "KOZO KEIKAKU ENGINEERING Inc.",
            2272 to "Philia Technology",
            2271 to "IRIS OHYAMA CO.,LTD.",
            2270 to "TE Connectivity Corporation",
            2269 to "code-Q",
            2268 to "SHENZHEN AUKEY E BUSINESS CO., LTD",
            2267 to "Tertium Technology",
            2266 to "Miridia Technology Incorporated",
            2265 to "Pointr Labs Limited",
            2264 to "WARES",
            2263 to "Inovonics Corp",
            2262 to "Nome Oy",
            2261 to "KEYes",
            2260 to "ADATA Technology Co., LTD.",
            2259 to "Novel Bits, LLC",
            2258 to "Virscient Limited",
            2257 to "Sensovium Inc.",
            2256 to "ESTOM Infotech Kft.",
            2255 to "betternotstealmybike UG (with limited liability)",
            2254 to "ZIMI CORPORATION",
            2253 to "ifly",
            2252 to "TGM TECHNOLOGY CO., LTD.",
            2251 to "JT INNOVATIONS LIMITED",
            2250 to "Nubia Technology Co.,Ltd.",
            2249 to "Noventa AG",
            2248 to "Liteboxer Technologies Inc.",
            2247 to "Monadnock Systems Ltd.",
            2246 to "Integra Optics Inc",
            2245 to "J. Wagner GmbH",
            2244 to "CellAssist, LLC",
            2243 to "CHIPOLO d.o.o.",
            2242 to "Lindinvent AB",
            2241 to "Rayden.Earth LTD",
            2240 to "Accent Advanced Systems SLU",
            2239 to "SIRC Co., Ltd.",
            2238 to "ubisys technologies GmbH",
            2237 to "bf1systems limited",
            2236 to "Prevayl Limited",
            2235 to "Tokai-rika co.,ltd.",
            2234 to "HYPER ICE, INC.",
            2233 to "U-Shin Ltd.",
            2232 to "Check Technology Solutions LLC",
            2231 to "ABB Inc",
            2230 to "Boehringer Ingelheim Vetmedica GmbH",
            2229 to "TransferFi",
            2228 to "Sengled Co., Ltd.",
            2227 to "IONIQ Skincare GmbH & Co. KG",
            2226 to "PF SCHWEISSTECHNOLOGIE GMBH",
            2225 to "CORE|vision BV",
            2224 to "Trivedi Advanced Technologies LLC",
            2223 to "Polidea Sp. z o.o.",
            2222 to "Moticon ReGo AG",
            2221 to "Kayamatics Limited",
            2220 to "Topre Corporation",
            2219 to "Coburn Technology, LLC",
            2218 to "SZ DJI TECHNOLOGY CO.,LTD",
            2217 to "Fraunhofer IIS",
            2216 to "Shanghai Kfcube Inc",
            2215 to "TGR 1.618 Limited",
            2214 to "Intelligenceworks Inc.",
            2213 to "UMEHEAL Ltd",
            2212 to "Realme Chongqing Mobile Telecommunications Corp., Ltd.",
            2211 to "Hoffmann SE",
            2210 to "Epic Systems Co., Ltd.",
            2209 to "EXEO TECH CORPORATION",
            2208 to "Aclara Technologies LLC",
            2207 to "Witschi Electronic Ltd",
            2206 to "i-SENS, inc.",
            2205 to "J-J.A.D.E. Enterprise LLC",
            2204 to "Embedded Devices Co. Company",
            2203 to "Saucon Technologies",
            2202 to "Private limited company \"Teltonika\"",
            2201 to "SFS unimarket AG",
            2200 to "Sensibo, Inc.",
            2199 to "Current Lighting Solutions LLC",
            2198 to "Nokian Renkaat Oyj",
            2197 to "Gimer medical",
            2196 to "EPIFIT",
            2195 to "Maytronics Ltd",
            2194 to "Ingenieurbuero Birnfeld UG (haftungsbeschraenkt)",
            2193 to "SmartWireless GmbH & Co. KG",
            2192 to "NICHIEI INTEC CO., LTD.",
            2191 to "Tait International Limited",
            2190 to "GIGA-TMS INC",
            2189 to "Soliton Systems K.K.",
            2188 to "GB Solution co.,Ltd",
            2187 to "Tricorder Arraay Technologies LLC",
            2186 to "sclak s.r.l.",
            2185 to "XANTHIO",
            2184 to "EnPointe Fencing Pty Ltd",
            2183 to "Hydro-Gear Limited Partnership",
            2182 to "Movella Technologies B.V.",
            2181 to "LEVOLOR INC",
            2180 to "Controlid Industria, Comercio de Hardware e Servicos de Tecnologia Ltda",
            2179 to "Wintersteiger AG",
            2178 to "PSYONIC, Inc.",
            2177 to "Optalert",
            2176 to "imagiLabs AB",
            2175 to "Phillips Connect Technologies LLC",
            2174 to "1bar.net Limited",
            2173 to "Konftel AB",
            2172 to "Crosscan GmbH",
            2171 to "BYSTAMP",
            2170 to "ZRF, LLC",
            2169 to "MIZUNO Corporation",
            2168 to "The Chamberlain Group, Inc.",
            2167 to "Tome, Inc.",
            2166 to "SmartResQ ApS",
            2165 to "Berner International LLC",
            2164 to "Treegreen Limited",
            2163 to "Innophase Incorporated",
            2162 to "11 Health & Technologies Limited",
            2161 to "Dension Elektronikai Kft.",
            2160 to "Wyze Labs, Inc",
            2159 to "Trackunit A/S",
            2158 to "Vorwerk Elektrowerke GmbH & Co. KG",
            2157 to "Biometrika d.o.o.",
            2156 to "Revvo Technologies, Inc.",
            2155 to "Pacific Track, LLC",
            2154 to "Odic Incorporated",
            2153 to "EVVA Sicherheitstechnologie GmbH",
            2152 to "WIOsense GmbH & Co. KG",
            2151 to "Western Digital Techologies, Inc.",
            2150 to "LAONZ Co.,Ltd",
            2149 to "Emergency Lighting Products Limited",
            2148 to "Rafaelmicro",
            2147 to "Yo-tronics Technology Co., Ltd.",
            2146 to "SmartDrive",
            2145 to "SmartSensor Labs Ltd",
            2144 to "Alflex Products B.V.",
            2143 to "COMPEGPS TEAM,SOCIEDAD LIMITADA",
            2142 to "Krog Systems LLC",
            2141 to "Guilin Zhishen Information Technology Co.,Ltd.",
            2140 to "ACOS CO.,LTD.",
            2139 to "Nisshinbo Micro Devices Inc.",
            2138 to "DAKATECH",
            2137 to "BlueUp",
            2136 to "SOUNDBOKS",
            2135 to "Parsyl Inc",
            2134 to "Canopy Growth Corporation",
            2133 to "Helios Sports, Inc.",
            2132 to "Tap Sound System",
            2131 to "Pektron Group Limited",
            2130 to "Cognosos, Inc.",
            2129 to "Subeca, Inc.",
            2128 to "Yealink (Xiamen) Network Technology Co.,LTD",
            2127 to "Embedded Fitness B.V.",
            2126 to "Carol Cole Company",
            2125 to "SafePort",
            2124 to "ORSO Inc.",
            2123 to "Biotechware SRL",
            2122 to "ARCOM",
            2121 to "Dopple Technologies B.V.",
            2120 to "JUJU JOINTS CANADA CORP.",
            2119 to "DNANUDGE LIMITED",
            2118 to "USound GmbH",
            2117 to "Dometic Corporation",
            2116 to "Pepperl + Fuchs GmbH",
            2115 to "FRAGRANCE DELIVERY TECHNOLOGIES LTD",
            2114 to "Tangshan HongJia electronic technology co., LTD.",
            2113 to "General Luminaire (Shanghai) Co., Ltd.",
            2112 to "Down Range Systems LLC",
            2111 to "D-Link Corp.",
            2110 to "Zorachka LTD",
            2109 to "Tokenize, Inc.",
            2108 to "BeerTech LTD",
            2107 to "Piaggio Fast Forward",
            2106 to "BPW Bergische Achsen Kommanditgesellschaft",
            2105 to "A puissance 3",
            2104 to "Etymotic Research, Inc.",
            2103 to "vivo Mobile Communication Co., Ltd.",
            2102 to "Bitwards Oy",
            2101 to "Canopy Growth Corporation",
            2100 to "RIKEN KEIKI CO., LTD.,",
            2099 to "Conneqtech B.V.",
            2098 to "Intermotive,Inc.",
            2097 to "Foxble, LLC",
            2096 to "Core Health and Fitness LLC",
            2095 to "Blippit AB",
            2094 to "ABB S.p.A.",
            2093 to "INCUS PERFORMANCE LTD.",
            2092 to "INGICS TECHNOLOGY CO., LTD.",
            2091 to "shenzhen fitcare electronics Co.,Ltd",
            2090 to "Mitutoyo Corporation",
            2089 to "HEXAGON METROLOGY DIVISION ROMER",
            2088 to "Shanghai Suisheng Information Technology Co., Ltd.",
            2087 to "Kickmaker",
            2086 to "Hyundai Motor Company",
            2085 to "CME PTE. LTD.",
            2084 to "8Power Limited",
            2083 to "Nexite Ltd",
            2082 to "adafruit industries",
            2081 to "INOVA Geophysical, Inc.",
            2080 to "Brilliant Home Technology, Inc.",
            2079 to "eSenseLab LTD",
            2078 to "iNFORM Technology GmbH",
            2077 to "Potrykus Holdings and Development LLC",
            2076 to "Bobrick Washroom Equipment, Inc.",
            2075 to "DIM3",
            2074 to "Shenzhen Conex",
            2073 to "Hunter Douglas Inc",
            2072 to "tatwah SA",
            2071 to "Wangs Alliance Corporation",
            2070 to "SPICA SYSTEMS LLC",
            2069 to "SKC Inc",
            2068 to "Ossur hf.",
            2067 to "Flextronics International USA Inc.",
            2066 to "Mstream Technologies., Inc.",
            2065 to "Becker Antriebe GmbH",
            2064 to "LECO Corporation",
            2063 to "Paradox Engineering SA",
            2062 to "TATTCOM LLC",
            2061 to "Azbil Co.",
            2060 to "Ingy B.V.",
            2059 to "Nanoleaf Canada Limited",
            2058 to "Altaneos",
            2057 to "Trulli Audio",
            2056 to "SISTEMAS KERN, SOCIEDAD ANÃ“MINA",
            2055 to "ECD Electronic Components GmbH Dresden",
            2054 to "TYRI Sweden AB",
            2053 to "Urbanminded Ltd",
            2052 to "Andon Health Co.,Ltd",
            2051 to "Domintell s.a.",
            2050 to "NantSound, Inc.",
            2049 to "CRONUS ELECTRONICS LTD",
            2048 to "Optek",
            2047 to "maxon motor ltd.",
            2046 to "BIROTA",
            2045 to "JSK CO., LTD.",
            2044 to "Renault SA",
            2043 to "Access Co., Ltd",
            2042 to "Klipsch Group, Inc.",
            2041 to "Direct Communication Solutions, Inc.",
            2040 to "quip NYC Inc.",
            2039 to "Cesar Systems Ltd.",
            2038 to "Shenzhen TonliScience and Technology Development Co.,Ltd",
            2037 to "Byton North America Corporation",
            2036 to "MEDIRLAB Orvosbiologiai Fejleszto Korlatolt Felelossegu Tarsasag",
            2035 to "DIGISINE ENERGYTECH CO. LTD.",
            2034 to "SERENE GROUP, INC",
            2033 to "Zimi Innovations Pty Ltd",
            2032 to "e-moola.com Pty Ltd",
            2031 to "Aktiebolaget Sandvik Coromant",
            2030 to "KidzTek LLC",
            2029 to "Joule IQ, INC.",
            2028 to "Frecce LLC",
            2027 to "NOVABASE S.R.L.",
            2026 to "ShapeLog, Inc.",
            2025 to "HÃ¤fele GmbH & Co KG",
            2024 to "Packetcraft, Inc.",
            2023 to "Komfort IQ, Inc.",
            2022 to "Waybeyond Limited",
            2021 to "Minut, Inc.",
            2020 to "Geeksme S.L.",
            2019 to "Airoha Technology Corp.",
            2018 to "Alfred Kaercher SE & Co. KG",
            2017 to "Lucie Labs",
            2016 to "Edifier International Limited",
            2015 to "Snap-on Incorporated",
            2014 to "Unlimited Engineering SL",
            2013 to "Linear Circuits",
            2012 to "ThingOS GmbH & Co KG",
            2011 to "Remedee Labs",
            2010 to "STARLITE Co., Ltd.",
            2009 to "Micro-Design, Inc.",
            2008 to "SOLUTIONS AMBRA INC.",
            2007 to "Nanjing Qinheng Microelectronics Co., Ltd",
            2006 to "ecobee Inc.",
            2005 to "hoots classic GmbH",
            2004 to "Kano Computing Limited",
            2003 to "LIVNEX Co.,Ltd.",
            2002 to "React Accessibility Limited",
            2001 to "Shanghai Panchip Microelectronics Co., Ltd",
            2000 to "Hangzhou Tuya Information  Technology Co., Ltd",
            1999 to "NeoSensory, Inc.",
            1998 to "Shanghai Top-Chip Microelectronics Tech. Co., LTD",
            1997 to "Smart Wave Technologies Canada Inc",
            1996 to "Barnacle Systems Inc.",
            1995 to "West Pharmaceutical Services, Inc.",
            1994 to "Modul-System HH AB",
            1993 to "Skullcandy, Inc.",
            1992 to "WRLDS Creations AB",
            1991 to "iaconicDesign Inc.",
            1990 to "Bluenetics GmbH",
            1989 to "June Life, Inc.",
            1988 to "Johnson Health Tech NA",
            1987 to "CIMTechniques, Inc.",
            1986 to "Radinn AB",
            1985 to "A.W. Chesterton Company",
            1984 to "Biral AG",
            1983 to "REGULA Ltd.",
            1982 to "Axentia Technologies AB",
            1981 to "Genedrive Diagnostics Ltd",
            1980 to "KD CIRCUITS LLC",
            1979 to "EPIC S.R.L.",
            1978 to "Battery-Biz Inc.",
            1977 to "Epona Biotec Limited",
            1976 to "iSwip",
            1975 to "ETABLISSEMENTS GEORGES RENAULT",
            1974 to "Soundbrenner Limited",
            1973 to "CRONO CHIP, S.L.",
            1972 to "Hormann KG Antriebstechnik",
            1971 to "2N TELEKOMUNIKACE a.s.",
            1970 to "Moeco IOT Inc.",
            1969 to "Thomas Dynamics, LLC",
            1968 to "GV Concepts Inc.",
            1967 to "Hong Kong Bouffalo Lab Limited",
            1966 to "Aurea Solucoes Tecnologicas Ltda.",
            1965 to "New H3C Technologies Co.,Ltd",
            1964 to "LoupeDeck Oy",
            1963 to "Granite River Solutions, Inc.",
            1962 to "The Kroger Co.",
            1961 to "Bruel & Kjaer Sound & Vibration",
            1960 to "conbee GmbH",
            1959 to "Zume, Inc.",
            1958 to "Musen Connect, Inc.",
            1957 to "RAB Lighting, Inc.",
            1956 to "Xiamen Mage Information Technology Co., Ltd.",
            1955 to "Comcast Cable",
            1954 to "Roku, Inc.",
            1953 to "Apollo Neuroscience, Inc.",
            1952 to "Regent Beleuchtungskorper AG",
            1951 to "Pune Scientific LLP",
            1950 to "Smartloxx GmbH",
            1949 to "Digibale Pty Ltd",
            1948 to "Sky UK Limited",
            1947 to "CST ELECTRONICS (PROPRIETARY) LIMITED",
            1946 to "GuangDong Oppo Mobile Telecommunications Corp., Ltd.",
            1945 to "PlantChoir Inc.",
            1944 to "HoloKit, Inc.",
            1943 to "Water-i.d. GmbH",
            1942 to "StarLeaf Ltd",
            1941 to "GASTEC CORPORATION",
            1940 to "The Coca-Cola Company",
            1939 to "AEV spol. s r.o.",
            1938 to "Cricut, Inc.",
            1937 to "Scosche Industries, Inc.",
            1936 to "KOMPAN A/S",
            1935 to "Hanna Instruments, Inc.",
            1934 to "FUJIMIC NIIGATA, INC.",
            1933 to "Cybex GmbH",
            1932 to "MINIBREW HOLDING B.V",
            1931 to "Optikam Tech Inc.",
            1930 to "The Wildflower Foundation",
            1929 to "PCB Piezotronics, Inc.",
            1928 to "BubblyNet, LLC",
            1927 to "Pangaea Solution",
            1926 to "HLP Controls Pty Limited",
            1925 to "O2 Micro, Inc.",
            1924 to "audifon GmbH & Co. KG",
            1923 to "ESEMBER LIMITED LIABILITY COMPANY",
            1922 to "DeviceDrive AS",
            1921 to "Qingping Technology (Beijing) Co., Ltd.",
            1920 to "Finch Technologies Ltd.",
            1919 to "Glenview Software Corporation",
            1918 to "Sparkage Inc.",
            1917 to "Sensority, s.r.o.",
            1916 to "radius co., ltd.",
            1915 to "AmaterZ, Inc.",
            1914 to "Niruha Systems Private Limited",
            1913 to "Loopshore Oy",
            1912 to "KOAMTAC INC.",
            1911 to "Cue",
            1910 to "Cyber Transport Control GmbH",
            1909 to "4eBusiness GmbH",
            1908 to "C-MAX Asia Limited",
            1907 to "Echoflex Solutions Inc.",
            1906 to "Thirdwayv Inc.",
            1905 to "Corvex Connected Safety",
            1904 to "InnoCon Medical ApS",
            1903 to "Successful Endeavours Pty Ltd",
            1902 to "WuQi technologies, Inc.",
            1901 to "Graesslin GmbH",
            1900 to "Noodle Technology inc",
            1899 to "Engineered Medical Technologies",
            1898 to "Dmac Mobile Developments, LLC",
            1897 to "Force Impact Technologies",
            1896 to "Peloton Interactive Inc.",
            1895 to "NITTO DENKO ASIA TECHNICAL CENTRE PTE. LTD.",
            1894 to "ART AND PROGRAM, INC.",
            1893 to "Voxx International",
            1892 to "WWZN Information Technology Company Limited",
            1891 to "PIKOLIN S.L.",
            1890 to "TerOpta Ltd",
            1889 to "Mantis Tech LLC",
            1888 to "Vimar SpA",
            1887 to "Remote Solution Co., LTD.",
            1886 to "Katerra Inc.",
            1885 to "RHOMBUS SYSTEMS, INC.",
            1884 to "Antitronics Inc.",
            1883 to "Smart Sensor Devices AB",
            1882 to "HARMAN CO.,LTD.",
            1881 to "Shanghai InGeek Cyber Security Co., Ltd.",
            1880 to "umanSense AB",
            1879 to "ELA Innovation",
            1878 to "Lumens For Less, Inc",
            1877 to "Brother Industries, Ltd",
            1876 to "Michael Parkin",
            1875 to "JLG Industries, Inc.",
            1874 to "Elatec GmbH",
            1873 to "Changsha JEMO IC Design Co.,Ltd",
            1872 to "Hamilton Professional Services of Canada Incorporated",
            1871 to "MEDIATECH S.R.L.",
            1870 to "EAGLE DETECTION SA",
            1869 to "Amtech Systems, LLC",
            1868 to "iopool s.a.",
            1867 to "Sarvavid Software Solutions LLP",
            1866 to "Illusory Studios LLC",
            1865 to "DIAODIAO (Beijing) Technology Co., Ltd.",
            1864 to "GuangZhou KuGou Computer Technology Co.Ltd",
            1863 to "OR Technologies Pty Ltd",
            1862 to "Seitec Elektronik GmbH",
            1861 to "WIZNOVA, Inc.",
            1860 to "SOCOMEC",
            1859 to "Sanofi",
            1858 to "DML LLC",
            1857 to "MAC SRL",
            1856 to "HITIQ LIMITED",
            1855 to "Beijing Unisoc Technologies Co., Ltd.",
            1854 to "Bluepack S.R.L.",
            1853 to "Beijing Hao Heng Tian Tech Co., Ltd.",
            1852 to "Acubit ApS",
            1851 to "Fantini Cosmi s.p.a.",
            1850 to "Chandler Systems Inc.",
            1849 to "Jiangsu Qinheng Co., Ltd.",
            1848 to "Glass Security Pte Ltd",
            1847 to "LLC Navitek",
            1846 to "Luna XIO, Inc.",
            1845 to "UpRight Technologies LTD",
            1844 to "DiUS Computing Pty Ltd",
            1843 to "Iguanavation, Inc.",
            1842 to "Dairy Tech, Inc.",
            1841 to "ABLIC Inc.",
            1840 to "Wildlife Acoustics, Inc.",
            1839 to "OnePlus Electronics (Shenzhen) Co., Ltd.",
            1838 to "Open Platform Systems LLC",
            1837 to "Safera Oy",
            1836 to "GWA Hygiene GmbH",
            1835 to "Bitkey Inc.",
            1834 to "JMR embedded systems GmbH",
            1833 to "SwaraLink Technologies",
            1832 to "Eli Lilly and Company",
            1831 to "STALKIT AS",
            1830 to "PHC Corporation",
            1829 to "Tedee Sp. z o.o.",
            1828 to "Guangzhou SuperSound Information Technology Co.,Ltd",
            1827 to "Ford Motor Company",
            1826 to "Xiamen Eholder Electronics Co.Ltd",
            1825 to "Clover Network, Inc.",
            1824 to "Oculeve, Inc.",
            1823 to "Dongguan Liesheng Electronic Co.Ltd",
            1822 to "DONGGUAN HELE ELECTRONICS CO., LTD",
            1821 to "exoTIC Systems",
            1820 to "F5 Sports, Inc",
            1819 to "Precor",
            1818 to "REVSMART WEARABLE HK CO LTD",
            1817 to "COREIOT PTY LTD",
            1816 to "IDIBAIX enginneering",
            1815 to "iQsquare BV",
            1814 to "Altonics",
            1813 to "MBARC LABS Inc",
            1812 to "MindPeace Safety LLC",
            1811 to "Respiri Limited",
            1810 to "Bull Group Company Limited",
            1809 to "ABAX AS",
            1808 to "Audiodo AB",
            1807 to "California Things Inc.",
            1806 to "FiveCo Sarl",
            1805 to "SmartSnugg Pty Ltd",
            1804 to "Beijing Winner Microelectronics Co.,Ltd",
            1803 to "Element Products, Inc.",
            1802 to "Huf HÃ¼lsbeck & FÃ¼rst GmbH & Co. KG",
            1801 to "Carewear Corp.",
            1800 to "Be Interactive Co., Ltd",
            1799 to "Essity Hygiene and Health Aktiebolag",
            1798 to "Wernher von Braun Center for ASdvanced Research",
            1797 to "AB Electrolux",
            1796 to "JBX Designs Inc.",
            1795 to "Beijing Jingdong Century Trading Co., Ltd.",
            1794 to "Akciju sabiedriba \"SAF TEHNIKA\"",
            1793 to "PAFERS TECH",
            1792 to "TraqFreq LLC",
            1791 to "Redpine Signals Inc",
            1790 to "Mahr GmbH",
            1789 to "ESS Embedded System Solutions Inc.",
            1788 to "Tom Communication Industrial Co.,Ltd.",
            1787 to "Sartorius AG",
            1786 to "Enequi AB",
            1785 to "happybrush GmbH",
            1784 to "BodyPlus Technology Co.,Ltd",
            1783 to "WILKA Schliesstechnik GmbH",
            1782 to "Vitulo Plus BV",
            1781 to "Vigil Technologies Inc.",
            1780 to "TouchÃ© Technology Ltd",
            1779 to "Alfred International Inc.",
            1778 to "Trapper Data AB",
            1777 to "Shibutani Co., Ltd.",
            1776 to "Chargy Technologies, SL",
            1775 to "ALCARE Co., Ltd.",
            1774 to "Avantis Systems Limited",
            1773 to "J Neades Ltd",
            1772 to "Sigur",
            1771 to "Houston Radar LLC",
            1770 to "SafeLine Sweden AB",
            1769 to "Zmartfun Electronics, Inc.",
            1768 to "Almendo Technologies GmbH",
            1767 to "VELUX A/S",
            1766 to "NIHON DENGYO KOUSAKU",
            1765 to "OPTEX CO.,LTD.",
            1764 to "Aluna",
            1763 to "Spinlock Ltd",
            1762 to "Alango Technologies Ltd",
            1761 to "Milestone AV Technologies LLC",
            1760 to "Avaya Inc.",
            1759 to "HLI Solutions Inc.",
            1758 to "Navcast, Inc.",
            1757 to "Intellithings Ltd.",
            1756 to "Industrial Network Controls, LLC",
            1755 to "Automatic Labs, Inc.",
            1754 to "Zliide Technologies ApS",
            1753 to "Shanghai Mountain View Silicon Co.,Ltd.",
            1752 to "AW Company",
            1751 to "FUBA Automotive Electronics GmbH",
            1750 to "JCT Healthcare Pty Ltd",
            1749 to "Sensirion AG",
            1748 to "DYNAKODE TECHNOLOGY PRIVATE LIMITED",
            1747 to "TriTeq Lock and Security, LLC",
            1746 to "CeoTronics AG",
            1745 to "Meyer Sound Laboratories, Incorporated",
            1744 to "Etekcity Corporation",
            1743 to "Belparts N.V.",
            1742 to "FIOR & GENTZ",
            1741 to "DIG Corporation",
            1740 to "Dongguan SmartAction Technology Co.,Ltd.",
            1739 to "Dyeware, LLC",
            1738 to "Shenzhen Zhongguang Infotech Technology Development Co., Ltd",
            1737 to "MYLAPS B.V.",
            1736 to "Storz & Bickel GmbH & Co. KG",
            1735 to "Somatix Inc",
            1734 to "Simm Tronic Limited",
            1733 to "Urban Compass, Inc",
            1732 to "Dream Labs GmbH",
            1731 to "King I Electronics.Co.,Ltd",
            1730 to "Measurlogic Inc.",
            1729 to "Alarm.com Holdings, Inc",
            1728 to "CAME S.p.A.",
            1727 to "Delcom Products Inc.",
            1726 to "HitSeed Oy",
            1725 to "ABB Oy",
            1724 to "TWS Srl",
            1723 to "Leaftronix Analogic Solutions Private Limited",
            1722 to "Beaconzone Ltd",
            1721 to "Beflex Inc.",
            1720 to "ShadeCraft, Inc",
            1719 to "iCOGNIZE GmbH",
            1718 to "Sociometric Solutions, Inc.",
            1717 to "Wabilogic Ltd.",
            1716 to "Sencilion Oy",
            1715 to "The Hablab ApS",
            1714 to "Tussock Innovation 2013 Limited",
            1713 to "SimpliSafe, Inc.",
            1712 to "BRK Brands, Inc.",
            1711 to "Shoof Technologies",
            1710 to "SenseQ Inc.",
            1709 to "InnovaSea Systems Inc.",
            1708 to "Ingchips Technology Co., Ltd.",
            1707 to "HMS Industrial Networks AB",
            1706 to "Produal Oy",
            1705 to "Soundmax Electronics Limited",
            1704 to "GD Midea Air-Conditioning Equipment Co., Ltd.",
            1703 to "Chipsea Technologies (ShenZhen) Corp.",
            1702 to "Roambee Corporation",
            1701 to "TEKZITEL PTY LTD",
            1700 to "LIMNO Co. Ltd.",
            1699 to "Nymbus, LLC",
            1698 to "Globalworx GmbH",
            1697 to "Cardo Systems, Ltd",
            1696 to "OBIQ Location Technology Inc.",
            1695 to "FlowMotion Technologies AS",
            1694 to "Delta Electronics, Inc.",
            1693 to "Vakaros LLC",
            1692 to "Noomi AB",
            1691 to "Dragonchip Limited",
            1690 to "Adero, Inc.",
            1689 to "RandomLab SAS",
            1688 to "Wood IT Security, LLC",
            1687 to "Stemco Products Inc",
            1686 to "Gunakar Private Limited",
            1685 to "Koki Holdings Co., Ltd.",
            1684 to "T&A Laboratories LLC",
            1683 to "Hach - Danaher",
            1682 to "Georg Fischer AG",
            1681 to "Curie Point AB",
            1680 to "Eccrine Systems, Inc.",
            1679 to "JRM Group Limited",
            1678 to "Razer Inc.",
            1677 to "JetBeep Inc.",
            1676 to "Changzhou Sound Dragon Electronics and Acoustics Co., Ltd",
            1675 to "Jiangsu Teranovo Tech Co., Ltd.",
            1674 to "Raytac Corporation",
            1673 to "Tacx b.v.",
            1672 to "Amsted Digital Solutions Inc.",
            1671 to "Cherry GmbH",
            1670 to "inQs Co., Ltd.",
            1669 to "Greenwald Industries",
            1668 to "Dermalapps, LLC",
            1667 to "Eltako GmbH",
            1666 to "Photron Limited",
            1665 to "Trade FIDES a.s.",
            1664 to "Mannkind Corporation",
            1663 to "NETGRID S.N.C. DI BISSOLI MATTEO, CAMPOREALE SIMONE, TOGNETTI FEDERICO",
            1662 to "MbientLab Inc",
            1661 to "Form Athletica Inc.",
            1660 to "Tile, Inc.",
            1659 to "I.FARM, INC.",
            1658 to "The Energy Conservatory, Inc.",
            1657 to "4iiii Innovations Inc.",
            1656 to "SABIK Offshore GmbH",
            1655 to "Innovation First, Inc.",
            1654 to "Expai Solutions Private Limited",
            1653 to "Deco Enterprises, Inc.",
            1652 to "BeSpoon",
            1651 to "Innova Ideas Limited",
            1650 to "Kopi",
            1649 to "Buzz Products Ltd.",
            1648 to "Gema Switzerland GmbH",
            1647 to "Hug Technology Ltd",
            1646 to "Eurotronik Kranj d.o.o.",
            1645 to "Venso EcoSolutions AB",
            1644 to "Ztove ApS",
            1643 to "DewertOkin GmbH",
            1642 to "Brady Worldwide Inc.",
            1641 to "Livanova USA, Inc.",
            1640 to "BBb Technology srl",
            1639 to "Spark Technology Labs Inc.",
            1638 to "WTO Werkzeug-Einrichtungen GmbH",
            1637 to "Pure International Limited",
            1636 to "RHA TECHNOLOGIES LTD",
            1635 to "Advanced Telemetry Systems, Inc.",
            1634 to "Particle Industries, Inc.",
            1633 to "Mode Lighting Limited",
            1632 to "RTC Industries, Inc.",
            1631 to "Ricoh Company Ltd",
            1630 to "Alo AB",
            1629 to "Panduit Corp.",
            1628 to "PixArt Imaging Inc.",
            1627 to "Sesam Solutions BV",
            1626 to "Zound Industries International AB",
            1625 to "UnSeen Technologies Oy",
            1624 to "Payex Norge AS",
            1623 to "Meshtronix Limited",
            1622 to "ZhuHai AdvanPro Technology Company Limited",
            1621 to "Renishaw PLC",
            1620 to "Ledlenser GmbH & Co. KG",
            1619 to "Meggitt SA",
            1618 to "ITZ Innovations- und Technologiezentrum GmbH",
            1617 to "Stasis Labs, Inc.",
            1616 to "Coravin, Inc.",
            1615 to "Digital Matter Pty Ltd",
            1614 to "KRUXWorks Technologies Private Limited",
            1613 to "iLOQ Oy",
            1612 to "Zumtobel Group AG",
            1611 to "Scale-Tec, Ltd",
            1610 to "Open Research Institute, Inc.",
            1609 to "Ryeex Technology Co.,Ltd.",
            1608 to "Ultune Technologies",
            1607 to "MED-EL",
            1606 to "SGV Group Holding GmbH & Co. KG",
            1605 to "BM3",
            1604 to "Apogee Instruments",
            1603 to "makita corporation",
            1602 to "Bluetrum Technology Co.,Ltd",
            1601 to "Revenue Collection Systems FRANCE SAS",
            1600 to "Dish Network LLC",
            1599 to "LDL TECHNOLOGY",
            1598 to "The Indoor Lab, LLC",
            1597 to "Xradio Technology Co.,Ltd.",
            1596 to "Contec Medical Systems Co., Ltd.",
            1595 to "Kromek Group Plc",
            1594 to "Prolojik Limited",
            1593 to "Shenzhen Minew Technologies Co., Ltd.",
            1592 to "LX SOLUTIONS PTY LIMITED",
            1591 to "GiP Innovation Tools GmbH",
            1590 to "ELECTRONICA INTEGRAL DE SONIDO S.A.",
            1589 to "Crookwood",
            1588 to "Fanstel Corp",
            1587 to "Wangi Lai PLT",
            1586 to "Hugo Muller GmbH & Co KG",
            1585 to "Fortiori Design LLC",
            1584 to "Asthrea D.O.O.",
            1583 to "ONKYO Corporation",
            1582 to "Procept",
            1581 to "Vossloh-Schwabe Deutschland GmbH",
            1580 to "ASPion GmbH",
            1579 to "MinebeaMitsumi Inc.",
            1578 to "Lunatico Astronomia SL",
            1577 to "PHONEPE PVT LTD",
            1576 to "Ensto Oy",
            1575 to "WEG S.A.",
            1574 to "Amplifico",
            1573 to "Square Panda, Inc.",
            1572 to "Biovotion AG",
            1571 to "Philadelphia Scientific (U.K.) Limited",
            1570 to "Beam Labs, LLC",
            1569 to "Noordung d.o.o.",
            1568 to "Forciot Oy",
            1567 to "Phrame Inc.",
            1566 to "Diamond Kinetics, Inc.",
            1565 to "SENS Innovation ApS",
            1564 to "Univations Limited",
            1563 to "silex technology, inc.",
            1562 to "R.W. Beckett Corporation",
            1561 to "Six Guys Labs, s.r.o.",
            1560 to "Audio-Technica Corporation",
            1559 to "WIZCONNECTED COMPANY LIMITED",
            1558 to "OS42 UG (haftungsbeschraenkt)",
            1557 to "INTER ACTION Corporation",
            1556 to "OnAsset Intelligence, Inc.",
            1555 to "Hans Dinslage GmbH",
            1554 to "Playfinity AS",
            1553 to "Beurer GmbH",
            1552 to "ADH GUARDIAN USA LLC",
            1551 to "Signify Netherlands B.V.",
            1550 to "Blueair AB",
            1549 to "TDK Corporation",
            1548 to "Vuzix Corporation",
            1547 to "Triax Technologies Inc",
            1546 to "IQAir AG",
            1545 to "BUCHI Labortechnik AG",
            1544 to "KeySafe-Cloud",
            1543 to "Rookery Technology Ltd",
            1542 to "John Deere",
            1541 to "FMW electronic Futterer u. Maier-Wolf OHG",
            1540 to "Cell2Jack LLC",
            1539 to "Fourth Evolution Inc",
            1538 to "Geberit International AG",
            1537 to "Schrader Electronics",
            1536 to "iRobot Corporation",
            1535 to "Wellnomics Ltd",
            1534 to "Niko nv",
            1533 to "Innoseis",
            1532 to "Masbando GmbH",
            1531 to "Arblet Inc.",
            1530 to "Konami Sports Life Co., Ltd.",
            1529 to "Hagleitner Hygiene International GmbH",
            1528 to "Anki Inc.",
            1527 to "TRACMO, INC.",
            1526 to "DPTechnics",
            1525 to "GS TAG",
            1524 to "Clearity, LLC",
            1523 to "SeeScan",
            1522 to "Try and E CO.,LTD.",
            1521 to "The Linux Foundation",
            1520 to "beken",
            1519 to "SIKOM AS",
            1518 to "Wristcam Inc.",
            1517 to "Fuji Xerox Co., Ltd",
            1516 to "Gycom Svenska AB",
            1515 to "Bayerische Motoren Werke AG",
            1514 to "ACS-Control-System GmbH",
            1513 to "iconmobile GmbH",
            1512 to "COWBOY",
            1511 to "PressurePro",
            1510 to "Motion Instruments Inc.",
            1509 to "INEO ENERGY& SYSTEMS",
            1508 to "Taiyo Yuden Co., Ltd",
            1507 to "Elemental Machines, Inc.",
            1506 to "stAPPtronics GmbH",
            1505 to "Human, Incorporated",
            1504 to "Viper Design LLC",
            1503 to "VIRTUALCLINIC.DIRECT LIMITED",
            1502 to "QT Medical INC.",
            1501 to "essentim GmbH",
            1500 to "Petronics Inc.",
            1499 to "Avid Identification Systems, Inc.",
            1498 to "Applied Neural Research Corp",
            1497 to "Toyo Electronics Corporation",
            1496 to "Farm Jenny LLC",
            1495 to "modum.io AG",
            1494 to "Zhuhai Jieli technology Co.,Ltd",
            1493 to "TEGAM, Inc.",
            1492 to "LAMPLIGHT Co., Ltd.",
            1491 to "Acurable Limited",
            1490 to "frogblue TECHNOLOGY GmbH",
            1489 to "Lindab AB",
            1488 to "Anova Applied Electronics",
            1487 to "Biowatch SA",
            1486 to "V-ZUG Ltd",
            1485 to "RJ Brands LLC",
            1484 to "WATTS ELECTRONICS",
            1483 to "LucentWear LLC",
            1482 to "MHL Custom Inc",
            1481 to "TBS Electronics B.V.",
            1480 to "SOMFY SAS",
            1479 to "Lippert Components, INC",
            1478 to "Smart Animal Training Systems, LLC",
            1477 to "SELVE GmbH & Co. KG",
            1476 to "Codecoup sp. z o.o. sp. k.",
            1475 to "Runtime, Inc.",
            1474 to "Grote Industries",
            1473 to "P.I.Engineering",
            1472 to "Nalu Medical, Inc.",
            1471 to "Real-World-Systems Corporation",
            1470 to "RFID Global by Softwork SrL",
            1469 to "ULC Robotics Inc.",
            1468 to "Leviton Mfg. Co., Inc.",
            1467 to "Oxford Metrics plc",
            1466 to "igloohome",
            1465 to "Suzhou Pairlink Network Technology",
            1464 to "Ambystoma Labs Inc.",
            1463 to "Beijing Pinecone Electronics Co.,Ltd.",
            1462 to "Elecs Industry Co.,Ltd.",
            1461 to "verisilicon",
            1460 to "White Horse Scientific ltd",
            1459 to "Parabit Systems, Inc.",
            1458 to "CAREL INDUSTRIES S.P.A.",
            1457 to "Medallion Instrumentation Systems",
            1456 to "NewTec GmbH",
            1455 to "OV LOOP, INC.",
            1454 to "CARMATE MFG.CO.,LTD",
            1453 to "INIA",
            1452 to "GoerTek Dynaudio Co., Ltd.",
            1451 to "Nofence AS",
            1450 to "Tramex Limited",
            1449 to "Monidor",
            1448 to "Tom Allebrandi Consulting",
            1447 to "Sonos Inc",
            1446 to "Telecon Mobile Limited",
            1445 to "Kiiroo BV",
            1444 to "O. E. M. Controls, Inc.",
            1443 to "Axiomware Systems Incorporated",
            1442 to "ADHERIUM(NZ) LIMITED",
            1441 to "Shanghai Xiaoyi Technology Co.,Ltd.",
            1440 to "RCP Software Oy",
            1439 to "Fisher & Paykel Healthcare",
            1438 to "Polycom, Inc.",
            1437 to "Tandem Diabetes Care",
            1436 to "Macrogiga Electronics",
            1435 to "Dataflow Systems Limited",
            1434 to "Teledyne Lecroy, Inc.",
            1433 to "Lazlo326, LLC.",
            1432 to "rapitag GmbH",
            1431 to "RadioPulse Inc",
            1430 to "My Smart Blinds",
            1429 to "Inor Process AB",
            1428 to "Kohler Company",
            1427 to "Spaulding Clinical Research",
            1426 to "IZITHERM",
            1425 to "Viasat Group S.p.A.",
            1424 to "Pur3 Ltd",
            1423 to "HENDON SEMICONDUCTORS PTY LTD",
            1422 to "Meta Platforms Technologies, LLC",
            1421 to "Jungheinrich Aktiengesellschaft",
            1420 to "Fracarro Radioindustrie SRL",
            1419 to "Maxim Integrated Products",
            1418 to "START TODAY CO.,LTD.",
            1417 to "Star Technologies",
            1416 to "ALT-TEKNIK LLC",
            1415 to "Derichs GmbH",
            1414 to "LEGRAND",
            1413 to "Hearing Lab Technology",
            1412 to "Gira Giersiepen GmbH & Co. KG",
            1411 to "Code Blue Communications",
            1410 to "Breakwall Analytics, LLC",
            1409 to "LYS TECHNOLOGIES LTD",
            1408 to "ARANZ Medical Limited",
            1407 to "Scuf Gaming International, LLC",
            1406 to "Beco, Inc",
            1405 to "Instinct Performance",
            1404 to "Toor Technologies LLC",
            1403 to "Duracell U.S. Operations Inc.",
            1402 to "OMNI Remotes",
            1401 to "Ensemble Tech Private Limited",
            1400 to "Wellington Drive Technologies Ltd",
            1399 to "True Wearables, Inc.",
            1398 to "Globalstar, Inc.",
            1397 to "Integral Memroy Plc",
            1396 to "AFFORDABLE ELECTRONICS INC",
            1395 to "Lighting Science Group Corp.",
            1394 to "AntTail.com",
            1393 to "PSIKICK, INC.",
            1392 to "Consumer Sleep Solutions LLC",
            1391 to "BikeFinder AS",
            1390 to "VIZPIN INC.",
            1389 to "Redmond Industrial Group LLC",
            1388 to "Long Range Systems, LLC",
            1387 to "Rion Co., Ltd.",
            1386 to "Flipnavi Co.,Ltd.",
            1385 to "Audionics System, INC.",
            1384 to "Bodyport Inc.",
            1383 to "Xiamen Everesports Goods Co., Ltd",
            1382 to "CORE TRANSPORT TECHNOLOGIES NZ LIMITED",
            1381 to "Beijing Smartspace Technologies Inc.",
            1380 to "Beghelli Spa",
            1379 to "Steinel Vertrieb GmbH",
            1378 to "Thalmic Labs Inc.",
            1377 to "Finder S.p.A.",
            1376 to "Sarita CareTech APS",
            1375 to "PROTECH S.A.S. DI GIRARDI ANDREA & C.",
            1374 to "Hekatron Vertriebs GmbH",
            1373 to "Valve Corporation",
            1372 to "Lely",
            1371 to "FRANKLIN TECHNOLOGY INC",
            1370 to "CANDY HOUSE, Inc.",
            1369 to "Newcon Optik",
            1368 to "benegear, inc.",
            1367 to "Arwin Technology Limited",
            1366 to "Otodynamics Ltd",
            1365 to "KROHNE Messtechnik GmbH",
            1364 to "National Instruments",
            1363 to "Nintendo Co., Ltd.",
            1362 to "Avempace SARL",
            1361 to "Sylero",
            1360 to "Versa Networks, Inc.",
            1359 to "Sinnoz",
            1358 to "FORTRONIK storitve d.o.o.",
            1357 to "Sensome",
            1356 to "Carefree Scott Fetzer Co Inc",
            1355 to "Advanced Electronic Designs, Inc.",
            1354 to "Linough Inc.",
            1353 to "Smart Technologies and Investment Limited",
            1352 to "Knick Elektronische Messgeraete GmbH & Co. KG",
            1351 to "LOGICDATA Electronic & Software Entwicklungs GmbH",
            1350 to "Apexar Technologies S.A.",
            1349 to "Candy Hoover Group s.r.l",
            1348 to "OrthoSensor, Inc.",
            1347 to "MIWA LOCK CO.,Ltd",
            1346 to "Mist Systems, Inc.",
            1345 to "Sharknet srl",
            1344 to "SilverPlus, Inc",
            1343 to "Silergy Corp",
            1342 to "CLIM8 LIMITED",
            1341 to "TESA SA",
            1340 to "Screenovate Technologies Ltd",
            1339 to "prodigy",
            1338 to "Savitech Corp.,",
            1337 to "OPPLE Lighting Co., Ltd",
            1336 to "Medela AG",
            1335 to "MetaLogics Corporation",
            1334 to "ZTR Control Systems LLC",
            1333 to "Smart Component Technologies Limited",
            1332 to "Frontiergadget, Inc.",
            1331 to "Nura Operations Pty Ltd",
            1330 to "CRESCO Wireless, Inc.",
            1329 to "D&M Holdings Inc.",
            1328 to "Adolene, Inc.",
            1327 to "Center ID Corp.",
            1326 to "LEDVANCE GmbH",
            1325 to "EXFO, Inc.",
            1324 to "Geosatis SA",
            1323 to "Novartis AG",
            1322 to "Keynes Controls Ltd",
            1321 to "Lumen UAB",
            1320 to "Lunera Lighting Inc.",
            1319 to "Albrecht JUNG",
            1318 to "Honeywell International Inc.",
            1317 to "HONGKONG NANO IC TECHNOLOGIES  CO., LIMITED",
            1316 to "Hangzhou iMagic Technology Co., Ltd",
            1315 to "MTG Co., Ltd.",
            1314 to "NS Tech, Inc.",
            1313 to "IAI Corporation",
            1312 to "Target Corporation",
            1311 to "Setec Pty Ltd",
            1310 to "Detect Blue Limited",
            1309 to "OFF Line Co., Ltd.",
            1308 to "EDPS",
            1307 to "Angee Technologies Ltd.",
            1306 to "Leica Camera AG",
            1305 to "Tyto Life LLC",
            1304 to "MAMORIO.inc",
            1303 to "Amtronic Sverige AB",
            1302 to "Footmarks",
            1301 to "RB Controls Co., Ltd.",
            1300 to "FIBRO GmbH",
            1299 to "9974091 Canada Inc.",
            1298 to "Soprod SA",
            1297 to "Brookfield Equinox LLC",
            1296 to "UNI-ELECTRONICS, INC.",
            1295 to "Foundation Engineering LLC",
            1294 to "Yichip Microelectronics (Hangzhou) Co.,Ltd.",
            1293 to "TRSystems GmbH",
            1292 to "OSRAM GmbH",
            1291 to "Vibrissa Inc.",
            1290 to "Shake-on B.V.",
            1289 to "Garage Smart, Inc.",
            1288 to "Axes System sp. z o. o.",
            1287 to "Yellowcog",
            1286 to "Hager",
            1285 to "InPlay, Inc.",
            1284 to "PHYPLUS Inc",
            1283 to "Locoroll, Inc",
            1282 to "Specifi-Kali LLC",
            1281 to "Polaris IND",
            1280 to "Wiliot LTD.",
            1279 to "Microsemi Corporation",
            1278 to "Woosim Systems Inc.",
            1277 to "Tapkey GmbH",
            1276 to "SwingLync L. L. C.",
            1275 to "Benchmark Drives GmbH & Co. KG",
            1274 to "Androtec GmbH",
            1273 to "Interactio",
            1272 to "Convergence Systems Limited",
            1271 to "Shenzhen Goodix Technology Co., Ltd",
            1270 to "McLear Limited",
            1269 to "Pirelli Tyre S.P.A.",
            1268 to "ZanCompute Inc.",
            1267 to "Cerevast Medical",
            1266 to "InDreamer Techsol Private Limited",
            1265 to "Theben AG",
            1264 to "Kosi Limited",
            1263 to "DaisyWorks, Inc",
            1262 to "Auxivia",
            1261 to "R9 Technology, Inc.",
            1260 to "Motorola Solutions",
            1259 to "Bird Home Automation GmbH",
            1258 to "Pacific Bioscience Laboratories, Inc",
            1257 to "Busch Jaeger Elektro GmbH",
            1256 to "STABILO International",
            1255 to "REHABTRONICS INC.",
            1254 to "Smart Solution Technology, Inc.",
            1253 to "Avack Oy",
            1252 to "Woodenshark",
            1251 to "Under Armour",
            1250 to "EllieGrid",
            1249 to "REACTEC LIMITED",
            1248 to "Guardtec, Inc.",
            1247 to "Emerson Electric Co.",
            1246 to "Lutron Electronics Co., Inc.",
            1245 to "4MOD Technology",
            1244 to "IOTTIVE (OPC) PRIVATE LIMITED",
            1243 to "Engineered Audio, LLC.",
            1242 to "Franceschi Marina snc",
            1241 to "RM Acquisition LLC",
            1240 to "FUJIFILM Corporation",
            1239 to "Blincam, Inc.",
            1238 to "LUGLOC LLC",
            1237 to "Gooee Limited",
            1236 to "5th Element Ltd",
            1235 to "Queercon, Inc",
            1234 to "Anloq Technologies Inc.",
            1233 to "KTS GmbH",
            1232 to "Olympus Corporation",
            1231 to "DOM Sicherheitstechnik GmbH & Co. KG",
            1230 to "GOOOLED S.R.L.",
            1229 to "Safetech Products LLC",
            1228 to "Enflux Inc.",
            1227 to "Novo Nordisk A/S",
            1226 to "Steiner-Optik GmbH",
            1225 to "Thornwave Labs Inc",
            1224 to "Shanghai Flyco Electrical Appliance Co., Ltd.",
            1223 to "Svantek Sp. z o.o.",
            1222 to "Insta GmbH",
            1221 to "Seibert Williams Glass, LLC",
            1220 to "TeAM Hutchins AB",
            1219 to "Mantracourt Electronics Limited",
            1218 to "Dmet Products Corp.",
            1217 to "Sospitas, s.r.o.",
            1216 to "Statsports International",
            1215 to "VIT Initiative, LLC",
            1214 to "Averos FZCO",
            1213 to "AlbynMedical",
            1212 to "Draegerwerk AG & Co. KGaA",
            1211 to "Neatebox Ltd",
            1210 to "Crestron Electronics, Inc.",
            1209 to "CSR Building Products Limited",
            1208 to "Soraa Inc.",
            1207 to "Analog Devices, Inc.",
            1206 to "Diagnoptics Technologies",
            1205 to "Swiftronix AB",
            1204 to "Inuheat Group AB",
            1203 to "mobike (Hong Kong) Limited",
            1202 to "The Shadow on the Moon",
            1201 to "Kartographers Technologies Pvt. Ltd.",
            1200 to "Weba Sport und Med. Artikel GmbH",
            1199 to "BIOROWER Handelsagentur GmbH",
            1198 to "ERM Electronic Systems LTD",
            1197 to "Shure Inc",
            1196 to "Undagrid B.V.",
            1195 to "Harbortronics, Inc.",
            1194 to "LINKIO SAS",
            1193 to "DISCOVERY SOUND TECHNOLOGY, LLC",
            1192 to "BioTex, Inc.",
            1191 to "Dallas Logic Corporation",
            1190 to "Vinetech Co., Ltd",
            1189 to "Guangzhou FiiO Electronics Technology Co.,Ltd",
            1188 to "Herbert Waldmann GmbH & Co. KG",
            1187 to "GT-tronics HK Ltd",
            1186 to "ovrEngineered, LLC",
            1185 to "PNI Sensor Corporation",
            1184 to "Vypin, LLC",
            1183 to "Popper Pay AB",
            1182 to "AND!XOR LLC",
            1181 to "Uhlmann & Zacher GmbH",
            1180 to "DyOcean",
            1179 to "nVisti, LLC",
            1178 to "Situne AS",
            1177 to "Ruuvi Innovations Ltd.",
            1176 to "METER Group, Inc. USA",
            1175 to "Cochlear Limited",
            1174 to "Polymorphic Labs LLC",
            1173 to "LMT Mercer Group, Inc",
            1172 to "SENNHEISER electronic GmbH & Co. KG",
            1171 to "Lynxemi Pte Ltd",
            1170 to "ADC Technology, Inc.",
            1169 to "SOREX - Wireless Solutions GmbH",
            1168 to "Matting AB",
            1167 to "BlueKitchen GmbH",
            1166 to "Companion Medical, Inc.",
            1165 to "S-Labs Sp. z o.o.",
            1164 to "Vectronix AG",
            1163 to "CP Electronics Limited",
            1162 to "Taelek Oy",
            1161 to "Igarashi Engineering",
            1160 to "Automotive Data Solutions Inc",
            1159 to "Centrica Connected Home",
            1158 to "DEV TECNOLOGIA INDUSTRIA, COMERCIO E MANUTENCAO DE EQUIPAMENTOS LTDA. - ME",
            1157 to "SKIDATA AG",
            1156 to "Revol Technologies Inc",
            1155 to "Multi Care Systems B.V.",
            1154 to "POS Tuning Udo Vosshenrich GmbH & Co. KG",
            1153 to "Quintrax Limited",
            1152 to "Dynometrics Inc.",
            1151 to "Pro-Mark, Inc.",
            1150 to "OurHub Dev IvS",
            1149 to "Occly LLC",
            1148 to "POWERMAT LTD",
            1147 to "MIYOSHI ELECTRONICS CORPORATION",
            1146 to "Sinosun Technology Co., Ltd.",
            1145 to "mywerk system GmbH",
            1144 to "FarSite Communications Limited",
            1143 to "Blue Spark Technologies",
            1142 to "Oxstren Wearable Technologies Private Limited",
            1141 to "Icom inc.",
            1140 to "iApartment co., ltd.",
            1139 to "Steelcase, Inc.",
            1138 to "Control-J Pty Ltd",
            1137 to "TiVo Corp",
            1136 to "iDesign s.r.l.",
            1135 to "Develco Products A/S",
            1134 to "Pambor Ltd.",
            1133 to "BEGA Gantenbrink-Leuchten KG",
            1132 to "Qingdao Realtime Technology Co., Ltd.",
            1131 to "PMD Solutions",
            1130 to "INSIGMA INC.",
            1129 to "Palago AB",
            1128 to "Kynesim Ltd",
            1127 to "Codenex Oy",
            1126 to "CycleLabs Solutions inc.",
            1125 to "International Forte Group LLC",
            1124 to "Bellman & Symfon Group AB",
            1123 to "Fathom Systems Inc.",
            1122 to "Bonsai Systems GmbH",
            1121 to "vhf elektronik GmbH",
            1120 to "Kolibree",
            1119 to "Real Time Automation, Inc.",
            1118 to "Nuviz, Inc.",
            1117 to "Boston Scientific Corporation",
            1116 to "Delta T Corporation",
            1115 to "SPACEEK LTD",
            1114 to "2048450 Ontario Inc",
            1113 to "Lumenetix, Inc",
            1112 to "Mini Solution Co., Ltd.",
            1111 to "RF INNOVATION",
            1110 to "Nemik Consulting Inc",
            1109 to "Atomation",
            1108 to "Sphinx Electronics GmbH & Co KG",
            1107 to "Qorvo Utrecht B.V.",
            1106 to "Svep Design Center AB",
            1105 to "Tunstall Nordic AB",
            1104 to "Teenage Engineering AB",
            1103 to "TTS Tooltechnic Systems AG & Co. KG",
            1102 to "Xtrava Inc.",
            1101 to "VEGA Grieshaber KG",
            1100 to "LifeStyle Lock, LLC",
            1099 to "Nain Inc.",
            1098 to "SHIMANO INC.",
            1097 to "1UP USA.com llc",
            1096 to "Grand Centrix GmbH",
            1095 to "Fabtronics Australia Pty Ltd",
            1094 to "NETGEAR, Inc.",
            1093 to "Kobian Canada Inc.",
            1092 to "Metanate Limited",
            1091 to "Tucker International LLC",
            1090 to "SECOM CO., LTD.",
            1089 to "iProtoXi Oy",
            1088 to "Valencell, Inc.",
            1087 to "Tentacle Sync GmbH",
            1086 to "Thermomedics, Inc.",
            1085 to "Coiler Corporation",
            1084 to "DeLaval",
            1083 to "Appside co., ltd.",
            1082 to "Nuheara Limited",
            1081 to "Radiance Technologies",
            1080 to "Helvar Ltd",
            1079 to "eBest IOT Inc.",
            1078 to "Drayson Technologies (Europe) Limited",
            1077 to "Blocks Wearables Ltd.",
            1076 to "Hatch Baby, Inc.",
            1075 to "Pillsy Inc.",
            1074 to "Silk Labs, Inc.",
            1073 to "Alticor Inc.",
            1072 to "SnapStyk Inc.",
            1071 to "Danfoss A/S",
            1070 to "MemCachier Inc.",
            1069 to "Meshtech AS",
            1068 to "Ticto N.V.",
            1067 to "iMicroMed Incorporated",
            1066 to "BD Medical",
            1065 to "Prolon Inc.",
            1064 to "SmallLoop, LLC",
            1063 to "Focus fleet and fuel management inc",
            1062 to "Husqvarna AB",
            1061 to "Unify Software and Solutions GmbH & Co. KG",
            1060 to "Trainesense Ltd.",
            1059 to "Chargifi Limited",
            1058 to "DELSEY SA",
            1057 to "Backbone Labs, Inc.",
            1056 to "TecBakery GmbH",
            1055 to "Kopin Corporation",
            1054 to "Dell Computer Corporation",
            1053 to "Benning Elektrotechnik und Elektronik GmbH & Co. KG",
            1052 to "WaterGuru, Inc.",
            1051 to "OrthoAccel Technologies",
            1050 to "Friday Labs Limited",
            1049 to "Novalogy LTD",
            1047 to "Fatigue Science",
            1046 to "SODA GmbH",
            1045 to "Uber Technologies Inc",
            1044 to "Lightning Protection International Pty Ltd",
            1043 to "Wireless Cables Inc",
            1042 to "SEFAM",
            1041 to "Luidia Inc",
            1040 to "Fender Musical Instruments",
            1039 to "CO-AX Technology, Inc.",
            1038 to "SKF (U.K.) Limited",
            1037 to "NorthStar Battery Company, LLC",
            1036 to "Senix Corporation",
            1035 to "Jana Care Inc.",
            1034 to "ZF OPENMATICS s.r.o.",
            1033 to "RYSE INC.",
            1032 to "ToGetHome Inc.",
            1031 to "Swiss Audio SA",
            1030 to "Airtago",
            1029 to "Vertex International, Inc.",
            1028 to "Authomate Inc",
            1027 to "Gantner Electronic GmbH",
            1026 to "Sears Holdings Corporation",
            1025 to "Relations Inc.",
            1024 to "i-developer IT Beratung UG",
            1023 to "Withings",
            1022 to "Littelfuse",
            1021 to "Trimble Inc.",
            1020 to "Kimberly-Clark",
            1019 to "Nox Medical",
            1018 to "Vyassoft Technologies Inc",
            1017 to "Becon Technologies Co.,Ltd.",
            1016 to "Rockford Corp.",
            1015 to "Owl Labs Inc.",
            1014 to "Iton Technology Corp.",
            1013 to "WHERE, Inc.",
            1012 to "PAL Technologies Ltd",
            1011 to "Flowscape AB",
            1010 to "WindowMaster A/S",
            1009 to "Hestan Smart Cooking Inc.",
            1008 to "CLINK",
            1007 to "foolography GmbH",
            1006 to "CUBE TECHNOLOGIES",
            1005 to "BASIC MICRO.COM,INC.",
            1004 to "Jigowatts Inc.",
            1003 to "Ozo Edu, Inc.",
            1002 to "Hello Inc.",
            1001 to "SHENZHEN LEMONJOY TECHNOLOGY CO., LTD.",
            1000 to "Reiner Kartengeraete GmbH & Co. KG.",
            999 to "TRUE Fitness Technology",
            998 to "IoT Instruments Oy",
            997 to "ffly4u",
            996 to "Chip-ing AG",
            995 to "Qualcomm Life Inc",
            994 to "Sensoan Oy",
            993 to "SPD Development Company Ltd",
            992 to "Actions (Zhuhai) Technology Co., Limited",
            991 to "Grob Technologies, LLC",
            990 to "Nathan Rhoades LLC",
            989 to "Andreas Stihl AG & Co. KG",
            988 to "Nima Labs",
            987 to "Instabeat, Inc",
            986 to "EnOcean GmbH",
            985 to "3IWare Co., Ltd.",
            984 to "Zen-Me Labs Ltd",
            983 to "FINSECUR",
            982 to "Yota Devices LTD",
            981 to "Wyzelink Systems Inc.",
            980 to "PEG PEREGO SPA",
            979 to "Sigma Connectivity AB",
            978 to "IOT Pot India Private Limited",
            977 to "Density Inc.",
            976 to "Watteam Ltd",
            975 to "MIRA, Inc.",
            974 to "CONTRINEX S.A.",
            973 to "Wynd Technologies, Inc.",
            972 to "Vonkil Technologies Ltd",
            971 to "SYSDEV Srl",
            970 to "In2things Automation Pvt. Ltd.",
            969 to "Gallagher Group",
            968 to "Avvel International",
            967 to "Structural Health Systems, Inc.",
            966 to "Intricon",
            965 to "St. Jude Medical, Inc.",
            964 to "Pico Technology Inc.",
            963 to "Casambi Technologies Oy",
            962 to "Snapchat Inc",
            961 to "Ember Technologies, Inc.",
            960 to "Arch Systems Inc.",
            959 to "iLumi Solutions Inc.",
            958 to "Applied Science, Inc.",
            957 to "amadas",
            956 to "ASB Bank Ltd",
            955 to "Abbott",
            954 to "Maxscend Microelectronics Company Limited",
            953 to "FREDERIQUE CONSTANT SA",
            952 to "A-Safe Limited",
            951 to "Airbly Inc.",
            950 to "Mattel",
            949 to "petPOMM, Inc",
            948 to "Alpha Nodus, inc.",
            947 to "Midwest Instruments & Controls",
            946 to "Propagation Systems Limited",
            945 to "Otodata Wireless Network Inc.",
            944 to "VIBRADORM GmbH",
            943 to "Comm-N-Sense Corp DBA Verigo",
            942 to "Allswell Inc.",
            941 to "XiQ",
            940 to "Smablo LTD",
            939 to "Meizu Technology Co., Ltd.",
            938 to "Exon Sp. z o.o.",
            937 to "THINKERLY SRL",
            936 to "Esrille Inc.",
            935 to "AeroScout",
            934 to "Medela, Inc",
            933 to "ACE CAD Enterprise Co., Ltd. (ACECAD)",
            932 to "Token Zero Ltd",
            931 to "SmartMovt Technology Co., Ltd",
            930 to "Candura Instruments",
            929 to "Alpine Labs LLC",
            928 to "IVT Wireless Limited",
            927 to "Molex Corporation",
            926 to "SchoolBoard Limited",
            925 to "CareView Communications, Inc.",
            924 to "ALE International",
            923 to "South Silicon Valley Microelectronics",
            922 to "NeST",
            921 to "Nikon Corporation",
            920 to "Thetatronics Ltd",
            919 to "LEGO System A/S",
            918 to "BLOKS GmbH",
            917 to "SDATAWAY",
            916 to "Netclearance Systems, Inc.",
            915 to "LAVAZZA S.p.A.",
            914 to "T&D",
            913 to "Thingsquare AB",
            912 to "INFOTECH s.r.o.",
            911 to "Xiaomi Inc.",
            910 to "Crownstone B.V.",
            909 to "Resmed Ltd",
            908 to "Appion Inc.",
            907 to "Noke",
            906 to "Kohler Mira Limited",
            905 to "ActiveBlu Corporation",
            904 to "Kapsch TrafficCom AB",
            903 to "BluStor PMC, Inc.",
            902 to "Aterica Inc.",
            901 to "Embedded Electronic Solutions Ltd. dba e2Solutions",
            900 to "OCOSMOS Co., Ltd.",
            899 to "Kronos Incorporated",
            898 to "Precision Outcomes Ltd",
            897 to "Sharp Corporation",
            896 to "LLC \"MEGA-F service\"",
            895 to "SociÃ©tÃ© des Produits NestlÃ© S.A.",
            894 to "lulabytes S.L.",
            893 to "MICRODIA Ltd.",
            892 to "Cronologics Corporation",
            891 to "Apption Labs Inc.",
            890 to "Algoria",
            889 to "Shenzhen iMCO Electronic Technology Co.,Ltd",
            888 to "Propeller Health",
            887 to "Plejd AB",
            886 to "Electronic Temperature Instruments Ltd",
            885 to "Expain AS",
            884 to "Holman Industries",
            883 to "AppNearMe Ltd",
            882 to "Nixie Labs, Inc.",
            881 to "ORBCOMM",
            880 to "Wazombi Labs OÃœ",
            879 to "Motiv, Inc.",
            878 to "MOTIVE TECHNOLOGIES, INC.",
            877 to "AirBolt Pty Ltd",
            876 to "Zipcar",
            875 to "BRControls Products BV",
            874 to "SetPoint Medical",
            873 to "littleBits",
            872 to "Metormote AB",
            871 to "Saphe International",
            870 to "BOLTT Sports technologies Private limited",
            869 to "BioMech Sensor LLC",
            868 to "Favero Electronics Srl",
            867 to "FREELAP SA",
            866 to "ON Semiconductor",
            865 to "Wellinks Inc.",
            864 to "Insulet Corporation",
            863 to "Acromag",
            862 to "Naya Health, Inc.",
            861 to "KYS",
            860 to "Eaton Corporation",
            859 to "Matrix Inc.",
            858 to "Phillips-Medisize A/S",
            857 to "Novotec Medical GmbH",
            856 to "MagniWare Ltd.",
            855 to "Polymap Wireless",
            854 to "Spectrum Brands, Inc.",
            853 to "Sigma Designs, Inc.",
            852 to "TOPPAN FORMS CO.,LTD.",
            851 to "Alpha Audiotronics, Inc.",
            850 to "iRiding(Xiamen)Technology Co.,Ltd.",
            849 to "Pieps GmbH",
            848 to "Bitstrata Systems Inc.",
            847 to "Heartland Payment Systems",
            846 to "SafeTrust Inc.",
            845 to "TASER International, Inc.",
            844 to "HM Electronics, Inc.",
            843 to "Libratone A/S",
            842 to "Vaddio",
            841 to "VersaMe",
            840 to "Arioneo",
            839 to "Prevent Biometrics",
            838 to "Acuity Brands Lighting, Inc",
            837 to "Locus Positioning",
            836 to "Whirl Inc",
            835 to "Drekker Development Pty. Ltd.",
            834 to "GERTEC BRASIL LTDA.",
            833 to "Etesian Technologies LLC",
            832 to "Letsense s.r.l.",
            831 to "Automation Components, Inc.",
            830 to "Monitra SA",
            829 to "TPV Technology Limited",
            828 to "Virtuosys",
            827 to "Courtney Thorne Limited",
            826 to "Appception, Inc.",
            825 to "Blue Sky Scientific, LLC",
            824 to "COBI GmbH",
            823 to "AJP2 Holdings, LLC",
            822 to "GISTIC",
            821 to "Enlighted Inc",
            820 to "Airthings ASA",
            819 to "Mul-T-Lock",
            818 to "Electrocompaniet A.S.",
            817 to "3flares Technologies Inc.",
            816 to "North Pole Engineering",
            815 to "OttoQ Inc",
            814 to "indoormap",
            813 to "BM innovations GmbH",
            812 to "NIPPON SMT.CO.,Ltd",
            811 to "ESYLUX",
            810 to "Electronic Design Lab",
            809 to "Eargo, Inc.",
            808 to "Grundfos A/S",
            807 to "Essex Electronics",
            806 to "Healthwear Technologies (Changzhou)Ltd",
            805 to "Amotus Solutions",
            804 to "Astro, Inc.",
            803 to "Rotor Bike Components",
            802 to "Compumedics Limited",
            801 to "Jewelbots",
            800 to "SONO ELECTRONICS. CO., LTD",
            799 to "MetaSystem S.p.A.",
            798 to "Eyefi, Inc.",
            797 to "Enterlab ApS",
            796 to "Lab Sensor Solutions",
            795 to "HQ Inc",
            794 to "Wurth Elektronik eiSos GmbH & Co. KG",
            793 to "Eugster Frismag AG",
            792 to "Aspenta International",
            791 to "CHUO Electronics CO., LTD.",
            790 to "AG Measurematics Pvt. Ltd.",
            789 to "Thermo Fisher Scientific",
            788 to "RIIG AI Sp. z o.o.",
            787 to "DiveNav, Inc.",
            786 to "Ducere Technologies Pvt Ltd",
            785 to "PEEQ DATA",
            784 to "SGL Italia S.r.l.",
            783 to "Shortcut Labs",
            782 to "Deviceworx",
            781 to "Devdata S.r.l.",
            780 to "Hilti AG",
            779 to "Magnitude Lighting Converters",
            778 to "Ellisys",
            777 to "Dolby Labs",
            776 to "Surefire, LLC",
            775 to "FUJI INDUSTRIAL CO.,LTD.",
            774 to "Life Laboratory Inc.",
            773 to "Swipp ApS",
            772 to "Oura Health Ltd",
            771 to "IACA electronique",
            770 to "Loop Devices, Inc",
            769 to "Giatec Scientific Inc.",
            768 to "World Moto Inc.",
            767 to "Silicon Laboratories",
            766 to "Lierda Science & Technology Group Co., Ltd.",
            765 to "Uwanna, Inc.",
            764 to "Shanghai Frequen Microelectronics Co., Ltd.",
            763 to "Clarius Mobile Health Corp.",
            762 to "CoSTAR TEchnologies",
            761 to "IMAGINATION TECHNOLOGIES LTD",
            760 to "Runteq Oy Ltd",
            759 to "DreamVisions co., Ltd.",
            758 to "Intemo Technologies",
            757 to "Indagem Tech LLC",
            756 to "Vensi, Inc.",
            755 to "AuthAir, Inc",
            754 to "GoPro, Inc.",
            753 to "The Idea Cave, LLC",
            752 to "Blackrat Software",
            751 to "SMART-INNOVATION.inc",
            750 to "Citizen Holdings Co., Ltd.",
            749 to "HTC Corporation",
            748 to "Delta Systems, Inc",
            747 to "Ardic Technology",
            746 to "Fujitsu Limited",
            745 to "Sensogram Technologies, Inc.",
            744 to "American Music Environments",
            743 to "Connected Yard, Inc.",
            742 to "Unwire",
            741 to "Espressif Systems (Shanghai) Co., Ltd.",
            740 to "Bytestorm Ltd.",
            739 to "Carmanah Technologies Corp.",
            738 to "NTT docomo",
            737 to "Victron Energy BV",
            736 to "University of Michigan",
            735 to "Blur Product Development",
            734 to "Samsung SDS Co., Ltd.",
            733 to "Flint Rehabilitation Devices, LLC",
            732 to "DeWalch Technologies, Inc.",
            731 to "Digi International Inc (R)",
            730 to "Gilvader",
            729 to "Fliegl Agrartechnik GmbH",
            728 to "Neosfar",
            727 to "NIPPON SYSTEMWARE CO.,LTD.",
            726 to "Send Solutions",
            725 to "OMRON Corporation",
            724 to "Secuyou ApS",
            723 to "Powercast Corporation",
            722 to "Afero, Inc.",
            721 to "Empatica Srl",
            720 to "3M",
            719 to "Anima",
            718 to "Teva Branded Pharmaceutical Products R&D, Inc.",
            717 to "BMA ergonomics b.v.",
            716 to "Eijkelkamp Soil & Water",
            715 to "AINA-Wireless Inc.",
            714 to "ABOV Semiconductor",
            713 to "PayRange Inc.",
            712 to "OneSpan",
            711 to "Electronics Tomorrow Limited",
            710 to "Ayatan Sensors",
            709 to "Lenovo (Singapore) Pte Ltd.",
            708 to "Wilson Sporting Goods",
            707 to "Techtronic Power Tools Technology Limited",
            706 to "Guillemot Corporation",
            705 to "LINE Corporation",
            704 to "Dash Robotics",
            703 to "Redbird Flight Simulations",
            702 to "Seguro Technology Sp. z o.o.",
            701 to "Chemtronics",
            700 to "Genevac Ltd",
            699 to "Koha.,Co.Ltd",
            698 to "Swissprime Technologies AG",
            697 to "Rinnai Corporation",
            696 to "Chrono Therapeutics",
            695 to "Oort Technologies LLC",
            694 to "Schneider Electric",
            693 to "HANSHIN ELECTRIC RAILWAY CO.,LTD.",
            692 to "Hyginex, Inc.",
            691 to "CLABER S.P.A.",
            690 to "Oura Health Oy",
            689 to "Raden Inc",
            688 to "Bestechnic(Shanghai),Ltd",
            687 to "Technicolor USA Inc.",
            686 to "WeatherFlow, Inc.",
            685 to "Rx Networks, Inc.",
            684 to "RTB Elektronik GmbH & Co. KG",
            683 to "BBPOS Limited",
            682 to "Doppler Lab",
            681 to "Chargelib",
            680 to "miSport Ltd.",
            679 to "Illuxtron international B.V.",
            678 to "Robert Bosch GmbH",
            677 to "Tendyron Corporation",
            676 to "Pacific Lock Company",
            675 to "Itude",
            674 to "Sera4 Ltd.",
            673 to "InventureTrack Systems",
            672 to "Impossible Camera GmbH",
            671 to "Areus Engineering GmbH",
            670 to "Kupson spol. s r.o.",
            669 to "ALOTTAZS LABS, LLC",
            668 to "Blue Sky Scientific, LLC",
            667 to "C2 Development, Inc.",
            666 to "Currant, Inc.",
            665 to "Inexess Technology Simma KG",
            664 to "EISST Ltd",
            663 to "storm power ltd",
            662 to "Petzl",
            661 to "Sivantos GmbH",
            660 to "ELIAS GmbH",
            659 to "Blue Bite",
            658 to "SwiftSensors",
            657 to "CliniCloud Inc",
            656 to "Multibit Oy",
            655 to "Church & Dwight Co., Inc",
            654 to "RF Digital Corp",
            653 to "IF, LLC",
            652 to "NANOLINK APS",
            651 to "Code Gears LTD",
            650 to "Jetro AS",
            649 to "SK Telecom",
            648 to "Willowbank Electronics Ltd",
            647 to "Wally Ventures S.L.",
            646 to "RF Code, Inc.",
            645 to "WOWTech Canada Ltd.",
            644 to "Synapse Electronics",
            643 to "Maven Machines, Inc.",
            642 to "Sonova AG",
            641 to "StoneL",
            640 to "ITEC corporation",
            639 to "ruwido austria gmbh",
            638 to "HabitAware, LLC",
            637 to "HUAWEI Technologies Co., Ltd.",
            636 to "Aseptika Ltd",
            635 to "DEFA AS",
            634 to "Ekomini inc.",
            633 to "steute Schaltgerate GmbH & Co. KG",
            632 to "Johnson Outdoors Inc",
            631 to "bewhere inc",
            630 to "E.G.O. Elektro-Geraetebau GmbH",
            629 to "Geotab",
            628 to "Motsai Research",
            627 to "OCEASOFT",
            626 to "Alps Alpine Co., Ltd.",
            625 to "Animas Corp",
            624 to "LSI ADL Technology",
            623 to "Aptcode Solutions",
            622 to "FLEURBAEY BVBA",
            621 to "Technogym SPA",
            620 to "Domster Tadeusz Szydlowski",
            619 to "DEKA Research & Development Corp.",
            618 to "Gemalto",
            617 to "Torrox GmbH & Co KG",
            616 to "Cerevo",
            615 to "XMI Systems SA",
            614 to "Schawbel Technologies LLC",
            613 to "SMK Corporation",
            612 to "DDS, Inc.",
            611 to "Identiv, Inc.",
            610 to "Glacial Ridge Technologies",
            609 to "SECVRE GmbH",
            608 to "SensaRx",
            607 to "Yardarm Technologies",
            606 to "Fluke Corporation",
            605 to "Lexmark International Inc.",
            604 to "NetEaseï¼ˆHangzhouï¼‰Network co.Ltd.",
            603 to "Five Interactive, LLC dba Zendo",
            602 to "University of Applied Sciences Valais/Haute Ecole Valaisanne",
            601 to "ALTYOR",
            600 to "Devialet SA",
            599 to "AdBabble Local Commerce Inc.",
            598 to "G24 Power Limited",
            597 to "Dai Nippon Printing Co., Ltd.",
            596 to "Playbrush",
            595 to "Xicato Inc.",
            594 to "UKC Technosolution",
            593 to "Lumo Bodytech Inc.",
            592 to "Sapphire Circuits LLC",
            591 to "Schneider SchreibgerÃ¤te GmbH",
            590 to "Microtronics Engineering GmbH",
            589 to "M-Way Solutions GmbH",
            588 to "Blue Clover Devices",
            587 to "Orlan LLC",
            586 to "Uwatec AG",
            585 to "Transcranial Ltd",
            584 to "Parker Hannifin Corp",
            583 to "FiftyThree Inc.",
            582 to "ACKme Networks, Inc.",
            581 to "Endress+Hauser",
            580 to "Iotera Inc",
            579 to "Masimo Corp",
            578 to "16Lab Inc",
            577 to "Bragi GmbH",
            576 to "Argenox Technologies",
            575 to "WaveWare Technologies Inc.",
            574 to "Raven Industries",
            573 to "ViCentra B.V.",
            572 to "Awarepoint",
            571 to "Beijing CarePulse Electronic Technology Co, Ltd",
            570 to "Alatech Tehnology",
            569 to "JIN CO, Ltd",
            568 to "Trakm8 Ltd",
            567 to "MSHeli s.r.l.",
            566 to "Pitpatpet Ltd",
            565 to "Qrio Inc",
            564 to "FengFan (BeiJing) Technology Co, Ltd",
            563 to "Shenzhen SuLong Communication Ltd",
            562 to "x-Senso Solutions Kft",
            561 to "ETA SA",
            560 to "Foster Electric Company, Ltd",
            559 to "Huami (Shanghai) Culture Communication CO., LTD",
            558 to "Siemens AG",
            557 to "Lupine",
            556 to "Pharynks Corporation",
            555 to "Tesla, Inc.",
            554 to "Stamer Musikanlagen GMBH",
            553 to "Muoverti Limited",
            552 to "Twocanoes Labs, LLC",
            551 to "LifeBEAM Technologies",
            550 to "Merlinia A/S",
            549 to "NestlÃ© Nespresso S.A.",
            548 to "Comarch SA",
            547 to "Philip Morris Products S.A.",
            546 to "Praxis Dynamics",
            545 to "Mobiquity Networks Inc",
            544 to "Manus Machina BV",
            543 to "Luster Leaf Products  Inc",
            542 to "Goodnet, Ltd",
            541 to "Edamic",
            540 to "Mobicomm Inc",
            539 to "Cisco Systems, Inc",
            538 to "Blue Speck Labs, LLC",
            537 to "DOTT Limited",
            536 to "Hiotech AB",
            535 to "Tech4home, Lda",
            534 to "MTI Ltd",
            533 to "Lukoton Experience Oy",
            532 to "IK Multimedia Production srl",
            531 to "Wyler AG",
            530 to "Interplan Co., Ltd",
            529 to "Telink Semiconductor Co. Ltd",
            528 to "ikeGPS",
            527 to "Comodule GMBH",
            526 to "Omron Healthcare Co., LTD",
            525 to "Simplo Technology Co., LTD",
            524 to "CoroWare Technologies, Inc",
            523 to "Jaguar Land Rover Limited",
            522 to "Macnica Inc.",
            521 to "InvisionHeart Inc.",
            520 to "LumiGeek LLC",
            519 to "STEMP Inc.",
            518 to "Otter Products, LLC",
            517 to "Smartbotics Inc.",
            516 to "Tapcentive Inc.",
            515 to "Kemppi Oy",
            514 to "Rigado LLC",
            513 to "AR Timing",
            512 to "Verifone Systems Pte Ltd. Taiwan Branch",
            511 to "Freescale Semiconductor, Inc.",
            510 to "Radio Systems Corporation",
            509 to "Kontakt Micro-Location Sp. z o.o.",
            508 to "Wahoo Fitness, LLC",
            507 to "Form Lifting, LLC",
            506 to "Gozio Inc.",
            505 to "Medtronic Inc.",
            504 to "Anyka (Guangzhou) Microelectronics Technology Co, LTD",
            503 to "Gelliner Limited",
            502 to "DJO Global",
            501 to "Cool Webthings Limited",
            500 to "UTC Fire and Security",
            499 to "The University of Tokyo",
            498 to "Itron, Inc.",
            497 to "Zebra Technologies Corporation",
            496 to "KloudNation",
            495 to "Fullpower Technologies, Inc.",
            494 to "Valeo Service",
            493 to "CuteCircuit LTD",
            492 to "Spreadtrum Communications Shanghai Ltd",
            491 to "AutoMap LLC",
            490 to "Advanced Application Design, Inc.",
            489 to "Sano, Inc.",
            488 to "STIR",
            487 to "IPS Group Inc.",
            486 to "Technology Solutions (UK) Ltd",
            485 to "Dynamic Devices Ltd",
            484 to "Freedom Innovations",
            483 to "Caterpillar Inc",
            482 to "Lectronix, Inc.",
            481 to "Jolla Ltd",
            480 to "Widex A/S",
            479 to "Bison Group Ltd.",
            478 to "Minelab Electronics Pty Limited",
            477 to "Koninklijke Philips N.V.",
            476 to "iParking Ltd.",
            475 to "Innblue Consulting",
            474 to "Logitech International SA",
            473 to "Savant Systems LLC",
            472 to "Code Corporation",
            471 to "Squadrone Systems Inc.",
            470 to "G-wearables inc.",
            469 to "ELAD srl",
            468 to "Newlab S.r.l.",
            467 to "Sky Wave Design",
            466 to "Gill Electronics",
            465 to "August Home, Inc",
            464 to "Primus Inter Pares Ltd",
            463 to "BSH",
            462 to "HOUWA SYSTEM DESIGN, k.k.",
            461 to "Chengdu Synwing Technology Ltd",
            460 to "Sam Labs Ltd.",
            459 to "Fetch My Pet",
            458 to "Laerdal Medical AS",
            457 to "Avi-on",
            456 to "Poly-Control ApS",
            455 to "Abiogenix Inc.",
            454 to "HASWARE Inc.",
            453 to "Bitcraze AB",
            452 to "DME Microelectronics",
            451 to "Bunch",
            450 to "Transenergooil AG",
            449 to "BRADATECH Corp.",
            448 to "pironex GmbH",
            447 to "Hongkong OnMicro Electronics Limited",
            446 to "Pulsate Mobile Ltd.",
            445 to "Syszone Co., Ltd",
            444 to "SenionLab AB",
            443 to "Cochlear Bone Anchored Solutions AB",
            442 to "Stages Cycling LLC",
            441 to "HANA Micron",
            440 to "i+D3 S.L.",
            439 to "General Electric Company",
            438 to "LM Technologies Ltd",
            437 to "Nest Labs Inc.",
            436 to "Trineo Sp. z o.o.",
            435 to "Nytec, Inc.",
            434 to "Nymi Inc.",
            433 to "Netizens Sp. z o.o.",
            432 to "Star Micronics Co., Ltd.",
            431 to "Sunrise Micro Devices, Inc.",
            430 to "Earlens Corporation",
            429 to "FlightSafety International",
            428 to "Trividia Health, Inc.",
            427 to "Meta Platforms, Inc.",
            426 to "Geophysical Technology Inc.",
            425 to "Canon Inc.",
            424 to "Taobao",
            423 to "ENERGOUS CORPORATION",
            422 to "Wille Engineering",
            421 to "Icon Health and Fitness",
            420 to "MSA Innovation, LLC",
            419 to "EROAD",
            418 to "GIGALANE.CO.,LTD",
            417 to "FIAMM",
            416 to "Channel Enterprises (HK) Ltd.",
            415 to "Strainstall Ltd",
            414 to "Ceruus",
            413 to "CVS Health",
            412 to "Cokiya Incorporated",
            411 to "CUBETECH s.r.o.",
            410 to "TRON Forum",
            409 to "SALTO SYSTEMS S.L.",
            408 to "VENGIT Korlatolt Felelossegu Tarsasag",
            407 to "WiSilica Inc.",
            406 to "Paxton Access Ltd",
            405 to "Zuli",
            404 to "Acoustic Stream Corporation",
            403 to "Maveric Automation LLC",
            402 to "Cloudleaf, Inc",
            401 to "FDK CORPORATION",
            400 to "Intelletto Technologies Inc.",
            399 to "Fireflies Systems",
            398 to "Google LLC",
            397 to "Extron Design Services",
            396 to "Wilo SE",
            395 to "Konica Minolta, Inc.",
            394 to "Able Trend Technology Limited",
            393 to "Physical Enterprises Inc.",
            392 to "Unico RBC",
            391 to "Seraphim Sense Ltd",
            390 to "CORE Lighting Ltd",
            389 to "bel'apps LLC",
            388 to "Nectar",
            387 to "Walt Disney",
            386 to "HOP Ubiquitous",
            385 to "Gecko Health Innovations, Inc.",
            384 to "Gigaset Communications GmbH",
            383 to "XTel Wireless ApS",
            382 to "BluDotz Ltd",
            381 to "BatAndCat",
            380 to "Mercedes-Benz Group AG",
            379 to "taskit GmbH",
            378 to "Telemonitor, Inc.",
            377 to "LAPIS Semiconductor Co.,Ltd",
            376 to "CASIO COMPUTER CO., LTD.",
            375 to "I-SYST inc.",
            374 to "SentriLock",
            373 to "Dynamic Controls",
            372 to "Everykey Inc.",
            371 to "Kocomojo, LLC",
            370 to "Connovate Technology Private Limited",
            369 to "Amazon.com Services LLC",
            368 to "Roche Diabetes Care AG",
            367 to "Podo Labs, Inc",
            366 to "Volantic AB",
            365 to "LifeScan Inc",
            364 to "MYSPHERA",
            363 to "Qblinks",
            362 to "Copeland Cold Chain LP",
            361 to "emberlight",
            360 to "Spicebox LLC",
            359 to "Ascensia Diabetes Care US Inc.",
            358 to "MISHIK Pte Ltd",
            357 to "Milwaukee Electric Tools",
            356 to "Qingdao Yeelink Information Technology Co., Ltd.",
            355 to "PCH International",
            354 to "MADSGlobalNZ Ltd.",
            353 to "yikes",
            352 to "AwoX",
            351 to "Timer Cap Co.",
            350 to "Unikey Technologies, Inc.",
            349 to "Estimote, Inc.",
            348 to "Pitius Tec S.L.",
            347 to "Biomedical Research Ltd.",
            346 to "micas AG",
            345 to "ChefSteps, Inc.",
            344 to "Inmite s.r.o.",
            343 to "Anhui Huami Information Technology Co., Ltd.",
            342 to "Accumulate AB",
            341 to "NETATMO",
            340 to "Pebble Technology",
            339 to "ROL Ergo",
            338 to "Vernier Software & Technology",
            337 to "OnBeep",
            336 to "Pioneer Corporation",
            335 to "B&W Group Ltd.",
            334 to "Tangerine, Inc.",
            333 to "HUIZHOU DESAY SV AUTOMOTIVE CO., LTD.",
            332 to "Mesh-Net Ltd",
            331 to "Master Lock",
            330 to "Tivoli Audio, LLC",
            329 to "Perytons Ltd.",
            328 to "Ambimat Electronics",
            327 to "Mighty Cast, Inc.",
            326 to "Ciright",
            325 to "Novatel Wireless",
            324 to "Lintech GmbH",
            323 to "Bkon Connect",
            322 to "Grape Systems Inc.",
            321 to "FedEx Services",
            320 to "Alpine Electronics (China) Co., Ltd",
            319 to "B&B Manufacturing Company",
            318 to "Nod, Inc.",
            317 to "WirelessWERX",
            316 to "Murata Manufacturing Co., Ltd.",
            315 to "Allegion",
            314 to "Tencent Holdings Ltd.",
            313 to "Focus Systems Corporation",
            312 to "NTEO Inc.",
            311 to "Prestigio Plaza Ltd.",
            310 to "Silvair, Inc.",
            309 to "Aireware LLC",
            308 to "Resolution Products, Ltd.",
            307 to "Blue Maestro Limited",
            306 to "MADS Inc",
            305 to "Cypress Semiconductor",
            304 to "Warehouse Innovations",
            303 to "Clarion Co. Inc.",
            302 to "ASSA ABLOY",
            301 to "Sony Corporation",
            300 to "TEMEC Instruments B.V.",
            299 to "SportIQ",
            298 to "Changzhou Yongse Infotech  Co., Ltd.",
            297 to "Nimble Devices Oy",
            296 to "GPSI Group Pty Ltd",
            295 to "Salutica Allied Solutions",
            294 to "Promethean Ltd.",
            293 to "SEAT es",
            292 to "HID Global",
            291 to "Kinsa, Inc",
            290 to "AirTurn, Inc.",
            289 to "Sino Wealth Electronic Ltd.",
            288 to "Porsche AG",
            287 to "Volkswagen AG",
            286 to "Skoda Auto a.s.",
            285 to "Arendi AG",
            284 to "Baidu",
            283 to "Hewlett Packard Enterprise",
            282 to "Qualcomm Labs, Inc.",
            281 to "Wize Technology Co., Ltd.",
            280 to "Radius Networks, Inc.",
            279 to "Wimoto Technologies Inc",
            278 to "10AK Technologies",
            277 to "e.solutions",
            276 to "Xensr",
            275 to "Openbrain Technologies, Co., Ltd.",
            274 to "Visybl Inc.",
            273 to "Steelseries ApS",
            272 to "Nippon Seiki Co., Ltd.",
            271 to "HiSilicon Technologies CO., LIMITED",
            270 to "Audi AG",
            269 to "DENSO TEN Limited",
            268 to "Transducers Direct, LLC",
            267 to "ERi, Inc",
            266 to "Codegate Ltd",
            265 to "Atus BV",
            264 to "Chicony Electronics Co., Ltd.",
            263 to "Demant A/S",
            262 to "Innovative Yachtter Solutions",
            261 to "Ubiquitous Computing Technology Corporation",
            260 to "PLUS Location Systems Pty Ltd",
            259 to "Bang & Olufsen A/S",
            258 to "Keiser Corporation",
            257 to "Fugoo, Inc.",
            256 to "TomTom International BV",
            255 to "Typo Products, LLC",
            254 to "Stanley Black and Decker",
            253 to "ValenceTech Limited",
            252 to "Delphi Corporation",
            251 to "KOUKAAM a.s.",
            250 to "Crystal Alarm AB",
            249 to "StickNFind",
            248 to "AceUni Corp., Ltd.",
            247 to "VSN Technologies, Inc.",
            246 to "Elcometer Limited",
            245 to "Smartifier Oy",
            244 to "Nautilus Inc.",
            243 to "Kent Displays Inc.",
            242 to "Morse Project Inc.",
            241 to "Witron Technology Limited",
            240 to "PayPal, Inc.",
            239 to "Bitsplitters GmbH",
            238 to "Above Average Outcomes, Inc.",
            237 to "Jolly Logic, LLC",
            236 to "BioResearch Associates",
            235 to "Server Technology Inc.",
            234 to "www.vtracksystems.com",
            233 to "Vtrack Systems",
            232 to "ACTS Technologies",
            231 to "KS Technologies",
            230 to "Freshtemp",
            229 to "Eden Software Consultants Ltd.",
            228 to "L.S. Research, Inc.",
            227 to "inMusic Brands, Inc",
            226 to "Semilink Inc",
            225 to "Danlers Ltd",
            224 to "Google",
            223 to "Misfit Wearables Corp",
            222 to "Muzik LLC",
            221 to "Hosiden Corporation",
            220 to "Procter & Gamble",
            219 to "Snuza (Pty) Ltd",
            218 to "txtr GmbH",
            217 to "Voyetra Turtle Beach",
            216 to "Qualcomm Connected Experiences, Inc.",
            215 to "Qualcomm Technologies, Inc.",
            214 to "Timex Group USA, Inc.",
            213 to "Austco Communication Systems",
            212 to "Kawantech",
            211 to "Taixingbang Technology (HK) Co,. LTD.",
            210 to "Dialog Semiconductor B.V.",
            209 to "Polar Electro Europe B.V.",
            208 to "Dexcom, Inc.",
            207 to "ARCHOS SA",
            206 to "Eve Systems GmbH",
            205 to "Microchip Technology Inc.",
            204 to "Beats Electronics",
            203 to "Binauric SE",
            202 to "MC10",
            201 to "Evluma",
            200 to "GeLo Inc",
            199 to "Quuppa Oy.",
            198 to "Selfly BV",
            197 to "Onset Computer Corporation",
            196 to "LG Electronics",
            195 to "adidas AG",
            194 to "Geneq Inc.",
            193 to "Shenzhen Excelsecu Data Technology Co.,Ltd",
            192 to "AMICCOM Electronics Corporation",
            191 to "Stalmart Technology Limited",
            190 to "AAMP of America",
            189 to "Aplix Corporation",
            188 to "Ace Sensor Inc",
            187 to "S-Power Electronics Limited",
            186 to "Starkey Hearing Technologies",
            185 to "Johnson Controls, Inc.",
            184 to "Qualcomm Innovation Center, Inc. (QuIC)",
            183 to "TreLab Ltd",
            182 to "Meso international",
            181 to "Swirl Networks",
            180 to "BDE Technology Co., Ltd.",
            179 to "Clarinox Technologies Pty. Ltd.",
            178 to "Bekey A/S",
            177 to "Saris Cycling Group, Inc",
            176 to "Passif Semiconductor Corp",
            175 to "Cinetix",
            174 to "Omegawave Oy",
            173 to "Peter Systemtechnik GmbH",
            172 to "Green Throttle Games",
            171 to "Ingenieur-Systemgruppe Zahn GmbH",
            170 to "CAEN RFID srl",
            169 to "MARELLI EUROPE S.P.A.",
            168 to "ARP Devices Limited",
            167 to "Visteon Corporation",
            166 to "Panda Ocean Inc.",
            165 to "OTL Dynamics LLC",
            164 to "LINAK A/S",
            163 to "Meta Watch Ltd.",
            162 to "Vertu Corporation Limited",
            161 to "SR-Medizinelektronik",
            160 to "Kensington Computer Products Group",
            159 to "Suunto Oy",
            158 to "Bose Corporation",
            157 to "Geoforce Inc.",
            156 to "Colorfy, Inc.",
            155 to "Jiangsu Toppower Automotive Electronics Co., Ltd.",
            154 to "Alpwise",
            153 to "i.Tech Dynamic Global Distribution Ltd.",
            152 to "zero1.tv GmbH",
            151 to "ConnecteDevice Ltd.",
            150 to "ODM Technology, Inc.",
            149 to "NEC Lighting, Ltd.",
            148 to "Airoha Technology Corp.",
            147 to "Universal Electronics, Inc.",
            146 to "ThinkOptics, Inc.",
            145 to "Advanced PANMOBIL systems GmbH & Co. KG",
            144 to "Funai Electric Co., Ltd.",
            143 to "Telit Wireless Solutions GmbH",
            142 to "Quintic Corp",
            141 to "Zscan Software",
            140 to "Gimbal Inc.",
            139 to "Topcon Positioning Systems, LLC",
            138 to "Jawbone",
            137 to "GN Hearing A/S",
            136 to "Ecotest",
            135 to "Garmin International, Inc.",
            134 to "Equinux AG",
            133 to "BlueRadios, Inc.",
            132 to "Ludus Helsinki Ltd.",
            131 to "TimeKeeping Systems, Inc.",
            130 to "DSEA A/S",
            129 to "WuXi Vimicro",
            128 to "DeLorme Publishing Company, Inc.",
            127 to "Autonet Mobile",
            126 to "Sports Tracking Technologies Ltd.",
            125 to "Seers Technology Co., Ltd.",
            124 to "A & R Cambridge",
            123 to "Hanlynn Technologies",
            122 to "MStar Semiconductor, Inc.",
            121 to "lesswire AG",
            120 to "Nike, Inc.",
            119 to "Laird Connectivity LLC",
            118 to "Creative Technology Ltd.",
            117 to "Samsung Electronics Co. Ltd.",
            116 to "Zomm, LLC",
            115 to "Group Sense Ltd.",
            114 to "ShangHai Super Smart Electronics Co. Ltd.",
            113 to "connectBlue AB",
            112 to "Monster, LLC",
            111 to "Sound ID",
            110 to "Summit Data Communications, Inc.",
            109 to "BriarTek, Inc",
            108 to "Beautiful Enterprise Co., Ltd.",
            107 to "Polar Electro OY",
            106 to "LTIMINDTREE LIMITED",
            105 to "A&D Engineering, Inc.",
            104 to "General Motors",
            103 to "GN Audio A/S",
            102 to "9Solutions Oy",
            101 to "HP, Inc.",
            100 to "Band XI International, LLC",
            99 to "MiCommand Inc.",
            98 to "Gibson Guitars",
            97 to "RDA Microelectronics",
            96 to "RivieraWaves S.A.S",
            95 to "Wicentric, Inc.",
            94 to "Stonestreet One, LLC",
            93 to "Realtek Semiconductor Corporation",
            92 to "Belkin International, Inc.",
            91 to "Ralink Technology Corporation",
            90 to "EM Microelectronic-Marin SA",
            89 to "Nordic Semiconductor ASA",
            88 to "Vizio, Inc.",
            87 to "Harman International Industries, Inc.",
            86 to "Sony Ericsson Mobile Communications",
            85 to "Plantronics, Inc.",
            84 to "3DiJoy Corporation",
            83 to "Free2move AB",
            82 to "J&M Corporation",
            81 to "Tzero Technologies, Inc.",
            80 to "SiRF Technology, Inc.",
            79 to "APT Ltd.",
            78 to "Avago Technologies",
            77 to "Staccato Communications, Inc.",
            76 to "Apple, Inc.",
            75 to "Continental Automotive Systems",
            74 to "Accel Semiconductor Ltd.",
            73 to "3DSP Corporation",
            72 to "Marvell Technology Group Ltd.",
            71 to "Bluegiga",
            70 to "MediaTek, Inc.",
            69 to "Atheros Communications, Inc.",
            68 to "Socket Mobile",
            67 to "PARROT AUTOMOTIVE SAS",
            66 to "CONWISE Technology Corporation Ltd",
            65 to "Integrated Silicon Solution Taiwan, Inc.",
            64 to "Seiko Epson Corporation",
            63 to "Bluetooth SIG, Inc",
            62 to "Systems and Chips, Inc",
            61 to "IPextreme, Inc.",
            60 to "BlackBerry Limited",
            59 to "Gennum Corporation",
            58 to "Panasonic Holdings Corporation",
            57 to "Integrated System Solution Corp.",
            56 to "Syntronix Corporation",
            55 to "Mobilian Corporation",
            54 to "Renesas Electronics Corporation",
            53 to "Eclipse (HQ Espana) S.L.",
            52 to "Computer Access Technology Corporation (CATC)",
            51 to "Commil Ltd",
            50 to "Red-M (Communications) Ltd",
            49 to "Synopsys, Inc.",
            48 to "ST Microelectronics",
            47 to "MewTel Technology Inc.",
            46 to "Norwood Systems",
            45 to "GCT Semiconductor",
            44 to "Macronix International Co. Ltd.",
            43 to "Tenovis",
            42 to "Symbol Technologies, Inc.",
            41 to "Hitachi Ltd",
            40 to "R F Micro Devices",
            39 to "Open Interface",
            38 to "C Technologies",
            37 to "NXP B.V.",
            36 to "Alcatel",
            35 to "WavePlus Technology Co., Ltd.",
            34 to "NEC Corporation",
            33 to "Mansella Ltd",
            32 to "BandSpeed, Inc.",
            31 to "AVM Berlin",
            30 to "Inventel",
            29 to "Qualcomm",
            28 to "Conexant Systems Inc.",
            27 to "Signia Technologies, Inc.",
            26 to "TTPCom Limited",
            25 to "Rohde & Schwarz GmbH & Co. KG",
            24 to "Transilica, Inc.",
            23 to "Newlogic",
            22 to "KC Technology Inc.",
            21 to "RTX A/S",
            20 to "Mitsubishi Electric Corporation",
            19 to "Atmel Corporation",
            18 to "Zeevo, Inc.",
            17 to "Widcomm, Inc.",
            16 to "Mitel Semiconductor",
            15 to "Broadcom Corporation",
            14 to "Parthus Technologies Inc.",
            13 to "Texas Instruments Inc.",
            12 to "Digianswer A/S",
            11 to "Silicon Wave",
            10 to "Qualcomm Technologies International, Ltd. (QTIL)",
            9 to "Infineon Technologies AG",
            8 to "Motorola",
            7 to "Lucent",
            6 to "Microsoft",
            5 to "3Com",
            4 to "Toshiba Corp.",
            3 to "IBM Corp.",
            2 to "Intel Corp.",
            1 to "Nokia Mobile Phones",
            0 to "Ericsson AB"
        )
    }
}
