/*
 * Decompiled with CFR 0.152.
 */
package dev.miku.r2dbc.mysql.client;

import dev.miku.r2dbc.mysql.client.San;
import dev.miku.r2dbc.mysql.util.AddressUtils;
import dev.miku.r2dbc.mysql.util.AssertUtils;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.security.cert.Certificate;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLSession;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.util.annotation.Nullable;

final class MySqlHostVerifier {
    private static final Logger logger = LoggerFactory.getLogger(MySqlHostVerifier.class);
    private static final String COMMON_NAME = "CN";

    private MySqlHostVerifier() {
    }

    static void accept(String host, SSLSession session) throws SSLException {
        AssertUtils.requireNonNull(host, "host must not be null");
        AssertUtils.requireNonNull(session, "session must not be null");
        Certificate[] certs = session.getPeerCertificates();
        if (certs.length < 1) {
            throw new SSLException(String.format("Certificate for '%s' does not exists", host));
        }
        if (!(certs[0] instanceof X509Certificate)) {
            throw new SSLException(String.format("Certificate for '%s' must be X509Certificate (not javax) rather than %s", host, certs[0].getClass()));
        }
        MySqlHostVerifier.accepted(host, (X509Certificate)certs[0]);
    }

    private static void accepted(String host, X509Certificate cert) throws SSLException {
        HostType type = MySqlHostVerifier.determineHostType(host);
        List<San> sans = MySqlHostVerifier.extractSans(cert);
        if (!sans.isEmpty()) {
            switch (type) {
                case IP_V4: {
                    MySqlHostVerifier.matchIpv4(host, sans);
                    break;
                }
                case IP_V6: {
                    MySqlHostVerifier.matchIpv6(host, sans);
                    break;
                }
                default: {
                    MySqlHostVerifier.matchDns(host, sans);
                    break;
                }
            }
        } else {
            String cn = MySqlHostVerifier.extractCn(cert);
            if (cn == null) {
                throw new SSLException(String.format("Certificate for '%s' does not contain the Common Name", host));
            }
            MySqlHostVerifier.matchCn(host, cn);
        }
    }

    private static void matchIpv4(String ip, List<San> sans) throws SSLPeerUnverifiedException {
        for (San san : sans) {
            if (7 != san.getType() || !ip.equals(san.getValue())) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("Certificate for '{}' matched by IPv4 value '{}' of the Subject Alternative Names", (Object)ip, (Object)san.getValue());
            }
            return;
        }
        throw new SSLPeerUnverifiedException(String.format("Certificate for '%s' does not match any of the Subject Alternative Names: %s", ip, sans));
    }

    private static void matchIpv6(String ip, List<San> sans) throws SSLPeerUnverifiedException {
        String host = MySqlHostVerifier.normaliseIpv6(ip);
        for (San san : sans) {
            if (san.getType() != 7 || !host.equals(MySqlHostVerifier.normaliseIpv6(san.getValue()))) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("Certificate for '{}' matched by IPv6 value '{}' of the Subject Alternative Names", (Object)ip, (Object)san.getValue());
            }
            return;
        }
        throw new SSLPeerUnverifiedException(String.format("Certificate for '%s' does not match any of the Subject Alternative Names: %s", ip, sans));
    }

    private static void matchDns(String hostname, List<San> sans) throws SSLPeerUnverifiedException {
        String host = hostname.toLowerCase(Locale.ROOT);
        if (host.isEmpty() || host.charAt(0) == '.' || host.endsWith("..")) {
            throw new SSLPeerUnverifiedException(String.format("Certificate for '%s' cannot match the Subject Alternative Names because it is invalid name", hostname));
        }
        for (San san : sans) {
            if (san.getType() != 2 || !MySqlHostVerifier.matchHost(host, san.getValue().toLowerCase(Locale.ROOT))) continue;
            if (logger.isDebugEnabled()) {
                logger.debug("Certificate for '{}' matched by DNS name '{}' of the Subject Alternative Names", (Object)host, (Object)san.getValue());
            }
            return;
        }
        throw new SSLPeerUnverifiedException(String.format("Certificate for '%s' does not match any of the Subject Alternative Names: %s", hostname, sans));
    }

    private static void matchCn(String hostname, String commonName) throws SSLPeerUnverifiedException {
        String host = hostname.toLowerCase(Locale.ROOT);
        String cn = commonName.toLowerCase(Locale.ROOT);
        if (host.isEmpty() || host.charAt(0) == '.' || host.endsWith("..") || !MySqlHostVerifier.matchHost(host, cn)) {
            throw new SSLPeerUnverifiedException(String.format("Certificate for '%s' does not match the Common Name: %s", hostname, commonName));
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Certificate for '{}' matched by Common Name '{}'", (Object)hostname, (Object)commonName);
        }
    }

    private static boolean matchHost(String host, String pattern) {
        String postfix;
        String prefix;
        if (pattern.isEmpty() || pattern.charAt(0) == '.' || pattern.endsWith("..")) {
            return false;
        }
        int asteriskIndex = pattern.indexOf(42);
        if (asteriskIndex < 0) {
            return host.equals(pattern);
        }
        int patternSize = pattern.length();
        if (patternSize == 1) {
            return false;
        }
        if (asteriskIndex > 0 && !host.startsWith(prefix = pattern.substring(0, asteriskIndex))) {
            return false;
        }
        int postfixSize = patternSize - asteriskIndex - 1;
        if (postfixSize > 0 && !host.endsWith(postfix = pattern.substring(asteriskIndex + 1))) {
            return false;
        }
        int remainderIndex = host.length() - postfixSize;
        if (remainderIndex <= asteriskIndex) {
            return false;
        }
        String remainder = host.substring(asteriskIndex, remainderIndex);
        return !remainder.contains(".");
    }

    @Nullable
    private static String extractCn(X509Certificate cert) throws SSLException {
        LdapName name;
        String principal = cert.getSubjectX500Principal().getName("RFC2253");
        try {
            name = new LdapName(principal);
        }
        catch (InvalidNameException e) {
            throw new SSLException(e.getMessage(), e);
        }
        for (Rdn rdn : name.getRdns()) {
            if (!rdn.getType().equalsIgnoreCase(COMMON_NAME)) continue;
            return rdn.getValue().toString();
        }
        return null;
    }

    private static List<San> extractSans(X509Certificate cert) {
        try {
            Collection<List<?>> pairs = cert.getSubjectAlternativeNames();
            if (pairs == null || pairs.isEmpty()) {
                return Collections.emptyList();
            }
            ArrayList<San> sans = new ArrayList<San>();
            for (List<?> pair : pairs) {
                Integer type;
                if (pair == null || pair.size() < 2 || (type = MySqlHostVerifier.determineSubjectType(pair.get(0))) == null) continue;
                if (2 == type || 7 == type) {
                    Object value = pair.get(1);
                    if (value instanceof String) {
                        sans.add(new San((String)value, type));
                        continue;
                    }
                    if (value instanceof byte[]) {
                        logger.warn("Certificate contains an ASN.1 DER encoded form in Subject Alternative Names, but DER is unsupported now");
                        continue;
                    }
                    if (!logger.isWarnEnabled()) continue;
                    logger.warn("Certificate contains an unknown value of Subject Alternative Names: {}", value.getClass());
                    continue;
                }
                logger.warn("Certificate contains an unknown type of Subject Alternative Names: {}", (Object)type);
            }
            return sans;
        }
        catch (CertificateParsingException ignored) {
            return Collections.emptyList();
        }
    }

    private static String normaliseIpv6(String ip) {
        try {
            return InetAddress.getByName(ip).getHostAddress();
        }
        catch (UnknownHostException unexpected) {
            return ip;
        }
    }

    private static HostType determineHostType(String hostname) {
        if (AddressUtils.isIpv4(hostname)) {
            return HostType.IP_V4;
        }
        int maxIndex = hostname.length() - 1;
        String host = hostname.charAt(0) == '[' && hostname.charAt(maxIndex) == ']' ? hostname.substring(1, maxIndex) : hostname;
        if (AddressUtils.isIpv6(host)) {
            return HostType.IP_V6;
        }
        return HostType.DNS;
    }

    @Nullable
    private static Integer determineSubjectType(Object type) {
        if (type instanceof Integer) {
            return (Integer)type;
        }
        try {
            return Integer.parseInt(type.toString());
        }
        catch (NumberFormatException e) {
            return null;
        }
    }

    private static enum HostType {
        IP_V6,
        IP_V4,
        DNS;

    }
}

