package dev.orne.beans;

/*-
 * #%L
 * Orne Beans
 * %%
 * Copyright (C) 2020 - 2023 Orne Developments
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import javax.validation.constraints.NotNull;

import org.apiguardian.api.API;
import org.apiguardian.api.API.Status;

import dev.orne.test.rnd.GeneratorMethod;

/**
 * Implementation for {@code Identity} for identities composed
 * of a single inner {@code String} value.
 * 
 * @author <a href="mailto:wamphiry@orne.dev">(w) Iker Hernaez</a>
 * @version 1.0, 2020-05
 * @since 0.1
 */
@API(status=Status.STABLE, since="0.1")
public class StringIdentity
extends AbstractSimpleIdentity<String> {

    /** The serial version UID. */
    private static final long serialVersionUID = 5660267975323191055L;

    /**
     * Creates a new instance.
     * 
     * @param value The identity value
     */
    @GeneratorMethod
    public StringIdentity(
            final String value) {
        super(value);
    }

    /**
     * Copy constructor.
     * 
     * @param copy The instance to copy
     */
    public StringIdentity(
            final @NotNull StringIdentity copy) {
        super(copy);
    }

    /**
     * Resolves the specified identity token to a valid {@code StringIdentity}.
     * 
     * @param token The identity token
     * @return The resolved identity token
     * @throws NullPointerException If the identity token is {@code null}
     * @throws UnrecognizedIdentityTokenException If the identity token is not
     * a valid identity token or it doesn't start with the expected prefix
     */
    @IdentityTokenResolver
    public static @NotNull StringIdentity fromIdentityToken(
            final @NotNull String token) {
        return new StringIdentity(extractTokenValue(
                IdentityTokenFormatter.DEFAULT_PREFIX,
                token));
    }

    /**
     * Extracts the {@code String} value of a token generated by
     * {@code StringIdentity}.
     * <p>
     * Note that the resulting value can be {@code null}. If a non-null
     * value is expected use{@code extractRequiredTokenValue(String, String)}.
     * 
     * @param prefix The expected identity token prefix.
     * @param token The identity token.
     * @return The extracted {@code String} value.
     * @throws NullPointerException If the identity token is {@code null}
     * @throws UnrecognizedIdentityTokenException If the identity token is not
     * a valid simple identity token or if it doesn't start with the expected
     * prefix.
     * @see #extractRequiredTokenValue(String, String)
     */
    public static String extractTokenValue(
            final @NotNull String prefix,
            final @NotNull String token) {
        return IdentityTokenFormatter.parse(prefix, token);
    }

    /**
     * Extracts the {@code String} value of a token generated by
     * {@code StringIdentity}.
     * <p>
     * If the resulting value is {@code null} an exception is thrown.
     * 
     * @param prefix The expected identity token prefix.
     * @param token The identity token.
     * @return The extracted {@code String} value.
     * @throws NullPointerException If the identity token is {@code null}
     * @throws UnrecognizedIdentityTokenException If the identity token is not
     * a valid simple identity token, if it doesn't start with the expected
     * prefix or if the extracted value is null.
     * @see #extractTokenValue(String, String)
     */
    public static @NotNull String extractRequiredTokenValue(
            final @NotNull String prefix,
            final @NotNull String token) {
        final String result = extractTokenValue(prefix, token);
        if (result == null) {
            throw new UnrecognizedIdentityTokenException(
                    "Unrecognized identity token: " + token);
        }
        return result;
    }
}
