/*
 * Decompiled with CFR 0.152.
 */
package dev.sigstore.fulcio.client;

import com.google.common.annotations.VisibleForTesting;
import dev.sigstore.encryption.certificates.Certificates;
import dev.sigstore.encryption.certificates.transparency.CTLogInfo;
import dev.sigstore.encryption.certificates.transparency.CTVerificationResult;
import dev.sigstore.encryption.certificates.transparency.CTVerifier;
import dev.sigstore.encryption.certificates.transparency.SignedCertificateTimestamp;
import dev.sigstore.encryption.certificates.transparency.VerifiedSCT;
import dev.sigstore.fulcio.client.FulcioVerificationException;
import dev.sigstore.trustroot.CertificateAuthority;
import dev.sigstore.trustroot.SigstoreTrustedRoot;
import dev.sigstore.trustroot.TransparencyLog;
import java.io.IOException;
import java.security.InvalidAlgorithmParameterException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorException;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.PKIXParameters;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

public class FulcioVerifier {
    private final List<CertificateAuthority> cas;
    private final List<TransparencyLog> ctLogs;
    private final CTVerifier ctVerifier;

    public static FulcioVerifier newFulcioVerifier(SigstoreTrustedRoot trustRoot) throws InvalidAlgorithmParameterException, CertificateException, InvalidKeySpecException, NoSuchAlgorithmException {
        return FulcioVerifier.newFulcioVerifier(trustRoot.getCAs(), trustRoot.getCTLogs());
    }

    public static FulcioVerifier newFulcioVerifier(List<CertificateAuthority> cas, List<TransparencyLog> ctLogs) throws InvalidKeySpecException, NoSuchAlgorithmException, InvalidAlgorithmParameterException, CertificateException {
        ArrayList<CTLogInfo> logs = new ArrayList<CTLogInfo>();
        for (TransparencyLog ctLog : ctLogs) {
            logs.add(new CTLogInfo(ctLog.getPublicKey().toJavaPublicKey(), "CT Log", ctLog.getBaseUrl().toString()));
        }
        CTVerifier verifier = new CTVerifier(logId -> logs.stream().filter(ctLogInfo -> Arrays.equals(ctLogInfo.getID(), logId)).findFirst().orElse(null));
        for (CertificateAuthority ca : cas) {
            ca.asTrustAnchor();
        }
        return new FulcioVerifier(cas, ctLogs, verifier);
    }

    private FulcioVerifier(List<CertificateAuthority> cas, List<TransparencyLog> ctLogs, CTVerifier ctVerifier) {
        this.cas = cas;
        this.ctLogs = ctLogs;
        this.ctVerifier = ctVerifier;
    }

    @VisibleForTesting
    void verifySct(CertPath fullCertPath) throws FulcioVerificationException {
        if (this.ctLogs.size() == 0) {
            throw new FulcioVerificationException("No ct logs were provided to verifier");
        }
        if (!Certificates.getEmbeddedSCTs(Certificates.getLeaf(fullCertPath)).isPresent()) {
            throw new FulcioVerificationException("No valid SCTs were found during verification");
        }
        this.verifyEmbeddedScts(fullCertPath);
    }

    private void verifyEmbeddedScts(CertPath certPath) throws FulcioVerificationException {
        CTVerificationResult result;
        List<? extends Certificate> certs = certPath.getCertificates();
        try {
            result = this.ctVerifier.verifySignedCertificateTimestamps(certs, null, null);
        }
        catch (CertificateEncodingException cee) {
            throw new FulcioVerificationException("Certificates could not be parsed during SCT verification");
        }
        for (VerifiedSCT validSct : result.getValidSCTs()) {
            Instant entryTime;
            SignedCertificateTimestamp sct = validSct.sct;
            byte[] logId = sct.getLogID();
            Optional<TransparencyLog> ctLog = TransparencyLog.find(this.ctLogs, logId, entryTime = Instant.ofEpochMilli(sct.getTimestamp()));
            if (!ctLog.isPresent()) continue;
            return;
        }
        throw new FulcioVerificationException("No valid SCTs were found, all(" + (result.getValidSCTs().size() + result.getInvalidSCTs().size()) + ") SCTs were invalid");
    }

    public void verifySigningCertificate(CertPath signingCertificate) throws FulcioVerificationException, IOException {
        CertPath fullCertPath = this.validateCertPath(signingCertificate);
        this.verifySct(fullCertPath);
    }

    public CertPath trimTrustedParent(CertPath signingCertificate) throws FulcioVerificationException, CertificateException {
        for (CertificateAuthority ca : this.cas) {
            if (!Certificates.containsParent(signingCertificate, ca.getCertPath())) continue;
            return Certificates.trimParent(signingCertificate, ca.getCertPath());
        }
        throw new FulcioVerificationException("Certificate does not chain to trusted roots");
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    CertPath validateCertPath(CertPath signingCertificate) throws FulcioVerificationException {
        CertPathValidator cpv;
        try {
            cpv = CertPathValidator.getInstance("PKIX");
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("No PKIX CertPathValidator, we probably shouldn't be here, but this seems to be a system library error not a program control flow issue", e);
        }
        X509Certificate leaf = Certificates.getLeaf(signingCertificate);
        List<CertificateAuthority> validCAs = CertificateAuthority.find(this.cas, leaf.getNotBefore().toInstant());
        if (validCAs.size() == 0) {
            throw new FulcioVerificationException("No valid Certificate Authorities found when validating certificate");
        }
        LinkedHashMap<String, String> caVerificationFailure = new LinkedHashMap<String, String>();
        Iterator<CertificateAuthority> iterator = validCAs.iterator();
        while (true) {
            PKIXParameters pkixParams;
            if (!iterator.hasNext()) {
                String errors = caVerificationFailure.entrySet().stream().map(entry -> (String)entry.getKey() + " (" + (String)entry.getValue() + ")").collect(Collectors.joining("\n"));
                throw new FulcioVerificationException("Certificate was not verifiable against CAs\n" + errors);
            }
            CertificateAuthority ca = iterator.next();
            try {
                pkixParams = new PKIXParameters(Collections.singleton(ca.asTrustAnchor()));
            }
            catch (InvalidAlgorithmParameterException | CertificateException e) {
                throw new RuntimeException("Can't create PKIX parameters for fulcioRoot. This should have been checked when generating a verifier instance", e);
            }
            pkixParams.setRevocationEnabled(false);
            Date dateInValidityPeriod = new Date(Certificates.getLeaf(signingCertificate).getNotBefore().getTime());
            pkixParams.setDate(dateInValidityPeriod);
            try {
                CertPath fullCertPath;
                block12: {
                    if (Certificates.isSelfSigned(signingCertificate)) {
                        if (Certificates.containsParent(signingCertificate, ca.getCertPath())) {
                            fullCertPath = signingCertificate;
                            break block12;
                        } else {
                            caVerificationFailure.put(ca.getUri().toString(), "Trusted root in chain does not match");
                            continue;
                        }
                    }
                    fullCertPath = Certificates.append(ca.getCertPath(), signingCertificate);
                }
                cpv.validate(fullCertPath, pkixParams);
                return fullCertPath;
            }
            catch (InvalidAlgorithmParameterException | CertPathValidatorException | CertificateException ve) {
                caVerificationFailure.put(ca.getUri().toString(), ve.getMessage());
                continue;
            }
            break;
        }
    }
}

