/*
 * Decompiled with CFR 0.152.
 */
package dev.sigstore;

import com.google.api.client.util.Preconditions;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.hash.Hashing;
import com.google.common.io.Files;
import dev.sigstore.KeylessVerificationException;
import dev.sigstore.TrustedRootProvider;
import dev.sigstore.VerificationOptions;
import dev.sigstore.bundle.Bundle;
import dev.sigstore.dsse.InTotoPayload;
import dev.sigstore.encryption.certificates.Certificates;
import dev.sigstore.encryption.signers.Verifiers;
import dev.sigstore.fulcio.client.FulcioVerificationException;
import dev.sigstore.fulcio.client.FulcioVerifier;
import dev.sigstore.rekor.client.HashedRekordRequest;
import dev.sigstore.rekor.client.RekorEntry;
import dev.sigstore.rekor.client.RekorTypeException;
import dev.sigstore.rekor.client.RekorTypes;
import dev.sigstore.rekor.client.RekorVerificationException;
import dev.sigstore.rekor.client.RekorVerifier;
import dev.sigstore.rekor.dsse.v0_0_1.Dsse;
import dev.sigstore.rekor.dsse.v0_0_1.PayloadHash;
import dev.sigstore.trustroot.SigstoreTrustedRoot;
import dev.sigstore.tuf.SigstoreTufClient;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SignatureException;
import java.security.cert.CertPath;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.sql.Date;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import org.bouncycastle.util.encoders.DecoderException;
import org.bouncycastle.util.encoders.Hex;

public class KeylessVerifier {
    private final FulcioVerifier fulcioVerifier;
    private final RekorVerifier rekorVerifier;

    private KeylessVerifier(FulcioVerifier fulcioVerifier, RekorVerifier rekorVerifier) {
        this.fulcioVerifier = fulcioVerifier;
        this.rekorVerifier = rekorVerifier;
    }

    public static Builder builder() {
        return new Builder();
    }

    public void verify(Path artifact, Bundle bundle, VerificationOptions options) throws KeylessVerificationException {
        try {
            byte[] artifactDigest = Files.asByteSource((File)artifact.toFile()).hash(Hashing.sha256()).asBytes();
            this.verify(artifactDigest, bundle, options);
        }
        catch (IOException e) {
            throw new KeylessVerificationException("Could not hash provided artifact path: " + artifact);
        }
    }

    public void verify(byte[] artifactDigest, Bundle bundle, VerificationOptions options) throws KeylessVerificationException {
        if (bundle.getDsseEnvelope().isEmpty() && bundle.getMessageSignature().isEmpty()) {
            throw new IllegalStateException("Bundle must contain a message signature or DSSE envelope to verify");
        }
        if (bundle.getEntries().isEmpty()) {
            throw new KeylessVerificationException("Cannot verify bundle without tlog entry");
        }
        if (bundle.getEntries().size() > 1) {
            throw new KeylessVerificationException("Bundle verification expects 1 entry, but found " + bundle.getEntries().size());
        }
        if (!bundle.getTimestamps().isEmpty()) {
            throw new KeylessVerificationException("Cannot verify bundles with timestamp verification material");
        }
        CertPath signingCert = bundle.getCertPath();
        X509Certificate leafCert = Certificates.getLeaf(signingCert);
        try {
            this.fulcioVerifier.verifySigningCertificate(signingCert);
        }
        catch (FulcioVerificationException | IOException ex) {
            throw new KeylessVerificationException("Fulcio certificate was not valid: " + ex.getMessage(), ex);
        }
        this.checkCertificateMatchers(leafCert, options.getCertificateMatchers());
        RekorEntry rekorEntry = bundle.getEntries().get(0);
        try {
            this.rekorVerifier.verifyEntry(rekorEntry);
        }
        catch (RekorVerificationException ex) {
            throw new KeylessVerificationException("Rekor entry signature was not valid", ex);
        }
        java.util.Date entryTime = Date.from(rekorEntry.getIntegratedTimeInstant());
        try {
            leafCert.checkValidity(entryTime);
        }
        catch (CertificateNotYetValidException e) {
            throw new KeylessVerificationException("Signing time was before certificate validity", e);
        }
        catch (CertificateExpiredException e) {
            throw new KeylessVerificationException("Signing time was after certificate expiry", e);
        }
        if (bundle.getMessageSignature().isPresent()) {
            this.checkMessageSignature(bundle.getMessageSignature().get(), rekorEntry, artifactDigest, leafCert);
        } else {
            this.checkDsseEnvelope(rekorEntry, bundle.getDsseEnvelope().get(), artifactDigest, leafCert);
        }
    }

    @VisibleForTesting
    void checkCertificateMatchers(X509Certificate cert, List<VerificationOptions.CertificateMatcher> matchers) throws KeylessVerificationException {
        try {
            if (matchers.size() > 0 && matchers.stream().noneMatch(matcher -> matcher.test(cert))) {
                String matcherSpec = matchers.stream().map(Object::toString).collect(Collectors.joining(",", "[", "]"));
                throw new KeylessVerificationException("No provided certificate identities matched values in certificate: " + matcherSpec);
            }
        }
        catch (VerificationOptions.UncheckedCertificateException ce) {
            throw new KeylessVerificationException("Could not verify certificate identities: " + ce.getMessage());
        }
    }

    private void checkMessageSignature(Bundle.MessageSignature messageSignature, RekorEntry rekorEntry, byte[] artifactDigest, X509Certificate leafCert) throws KeylessVerificationException {
        byte[] bundleDigest;
        if (messageSignature.getMessageDigest().isPresent() && !Arrays.equals(artifactDigest, bundleDigest = messageSignature.getMessageDigest().get().getDigest())) {
            throw new KeylessVerificationException("Provided artifact digest does not match digest used for verification\nprovided(hex) : " + Hex.toHexString((byte[])artifactDigest) + "\nverification(hex) : " + Hex.toHexString((byte[])bundleDigest));
        }
        byte[] signature = messageSignature.getSignature();
        try {
            if (!Verifiers.newVerifier(leafCert.getPublicKey()).verifyDigest(artifactDigest, signature)) {
                throw new KeylessVerificationException("Artifact signature was not valid");
            }
        }
        catch (InvalidKeyException | NoSuchAlgorithmException ex) {
            throw new RuntimeException(ex);
        }
        catch (SignatureException ex) {
            throw new KeylessVerificationException("Signature could not be processed: " + ex.getMessage(), ex);
        }
        try {
            RekorTypes.getHashedRekord(rekorEntry);
            String calculatedHashedRekord = Base64.getEncoder().encodeToString(HashedRekordRequest.newHashedRekordRequest(artifactDigest, Certificates.toPemBytes(leafCert), signature).toJsonPayload().getBytes(StandardCharsets.UTF_8));
            if (!Objects.equals(calculatedHashedRekord, rekorEntry.getBody())) {
                throw new KeylessVerificationException("Provided verification materials are inconsistent with log entry");
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Unexpected IOException on valid leafCert", e);
        }
        catch (RekorTypeException re) {
            throw new KeylessVerificationException("Unexpected rekor type", re);
        }
    }

    private void checkDsseEnvelope(RekorEntry rekorEntry, Bundle.DsseEnvelope dsseEnvelope, byte[] artifactDigest, X509Certificate leafCert) throws KeylessVerificationException {
        byte[] payloadDigest;
        Dsse rekorDsse;
        if (!Objects.equals("application/vnd.in-toto+json", dsseEnvelope.getPayloadType())) {
            throw new KeylessVerificationException("DSSE envelope must have payload type application/vnd.in-toto+json, but found '" + dsseEnvelope.getPayloadType() + "'");
        }
        InTotoPayload payload = InTotoPayload.from(dsseEnvelope);
        if (payload.getSubject().stream().noneMatch(subject -> {
            if (subject.getDigest().containsKey("sha256")) {
                try {
                    byte[] digestBytes = Hex.decode((String)subject.getDigest().get("sha256"));
                    return Arrays.equals(artifactDigest, digestBytes);
                }
                catch (DecoderException decoderException) {
                    // empty catch block
                }
            }
            return false;
        })) {
            String providedHashes = payload.getSubject().stream().map(s -> s.getDigest().getOrDefault("sha256", "no-sha256-hash")).collect(Collectors.joining(",", "[", "]"));
            throw new KeylessVerificationException("Provided artifact digest does not match any subject sha256 digests in DSSE payload\nprovided(hex) : " + Hex.toHexString((byte[])artifactDigest) + "\nverification  : " + providedHashes);
        }
        if (dsseEnvelope.getSignatures().size() != 1) {
            throw new KeylessVerificationException("DSSE envelope must have exactly 1 signature, but found: " + dsseEnvelope.getSignatures().size());
        }
        try {
            if (!Verifiers.newVerifier(leafCert.getPublicKey()).verify(dsseEnvelope.getPAE(), dsseEnvelope.getSignature())) {
                throw new KeylessVerificationException("DSSE signature was not valid");
            }
        }
        catch (InvalidKeyException | NoSuchAlgorithmException ex) {
            throw new RuntimeException(ex);
        }
        catch (SignatureException se) {
            throw new KeylessVerificationException("Signature could not be processed", se);
        }
        try {
            rekorDsse = RekorTypes.getDsse(rekorEntry);
        }
        catch (RekorTypeException re) {
            throw new KeylessVerificationException("Unexpected rekor type", re);
        }
        PayloadHash.Algorithm algorithm = rekorDsse.getPayloadHash().getAlgorithm();
        if (algorithm != PayloadHash.Algorithm.SHA_256) {
            throw new KeylessVerificationException("Cannot process DSSE entry with hashing algorithm " + algorithm.toString());
        }
        try {
            payloadDigest = Hex.decode((String)rekorDsse.getPayloadHash().getValue());
        }
        catch (DecoderException de) {
            throw new KeylessVerificationException("Could not decode hex sha256 artifact hash in hashrekord", de);
        }
        byte[] calculatedDigest = Hashing.sha256().hashBytes(dsseEnvelope.getPayload()).asBytes();
        if (!Arrays.equals(calculatedDigest, payloadDigest)) {
            throw new KeylessVerificationException("Digest of DSSE payload in bundle does not match DSSE payload digest in log entry");
        }
        if (rekorDsse.getSignatures().size() != 1) {
            throw new KeylessVerificationException("DSSE log entry must have exactly 1 signature, but found: " + rekorDsse.getSignatures().size());
        }
        if (!Base64.getEncoder().encodeToString(dsseEnvelope.getSignature()).equals(rekorDsse.getSignatures().get(0).getSignature())) {
            throw new KeylessVerificationException("Provided DSSE signature materials are inconsistent with DSSE log entry");
        }
    }

    public static class Builder {
        private TrustedRootProvider trustedRootProvider;

        public KeylessVerifier build() throws InvalidAlgorithmParameterException, CertificateException, InvalidKeySpecException, NoSuchAlgorithmException, IOException, InvalidKeyException {
            Preconditions.checkNotNull((Object)this.trustedRootProvider);
            SigstoreTrustedRoot trustedRoot = this.trustedRootProvider.get();
            FulcioVerifier fulcioVerifier = FulcioVerifier.newFulcioVerifier(trustedRoot);
            RekorVerifier rekorVerifier = RekorVerifier.newRekorVerifier(trustedRoot);
            return new KeylessVerifier(fulcioVerifier, rekorVerifier);
        }

        public Builder sigstorePublicDefaults() {
            SigstoreTufClient.Builder sigstoreTufClientBuilder = SigstoreTufClient.builder().usePublicGoodInstance();
            this.trustedRootProvider = TrustedRootProvider.from(sigstoreTufClientBuilder);
            return this;
        }

        public Builder sigstoreStagingDefaults() {
            SigstoreTufClient.Builder sigstoreTufClientBuilder = SigstoreTufClient.builder().useStagingInstance();
            this.trustedRootProvider = TrustedRootProvider.from(sigstoreTufClientBuilder);
            return this;
        }

        public Builder trustedRootProvider(TrustedRootProvider trustedRootProvider) {
            this.trustedRootProvider = trustedRootProvider;
            return this;
        }
    }
}

