package dev.vankka.dependencydownload.resource;

import dev.vankka.dependencydownload.dependency.Dependency;
import dev.vankka.dependencydownload.dependency.SnapshotDependency;
import dev.vankka.dependencydownload.dependency.StandardDependency;
import dev.vankka.dependencydownload.relocation.Relocation;
import org.jetbrains.annotations.NotNull;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.*;
import java.util.stream.Collectors;

/**
 * An object representation of the resource generated by the gradle plugin.
 */
public class DependencyDownloadResource {

    private final List<Dependency> dependencies = new ArrayList<>();
    private final List<Relocation> relocations = new ArrayList<>();

    public DependencyDownloadResource(@NotNull URL resourceURL) throws IOException {
        List<String> lines;
        try (InputStream inputStream = resourceURL.openStream()) {
            lines = new BufferedReader(new InputStreamReader(inputStream))
                    .lines()
                    .collect(Collectors.toList());
        }

        readFile(lines);
    }

    public DependencyDownloadResource(@NotNull String resourceContents) {
        readFile(Arrays.asList(resourceContents.split("\n")));
    }

    public DependencyDownloadResource(@NotNull List<String> resourceLines) {
        readFile(resourceLines);
    }

    private void readFile(List<String> lines) {
        String hashingAlgorithm = null;

        int relocationStep = -1;
        String pattern = null;
        String replacement = null;
        Set<String> include = null;
        for (String line : lines) {
            line = line.trim();
            if (line.isEmpty()) {
                continue;
            }

            if (line.startsWith("===ALGORITHM")) {
                String[] algorithmParts = line.split(" ");
                if (algorithmParts.length != 2) {
                    throw new IllegalArgumentException("Resource format is invalid: hashing algorithm: " + line);
                }
                hashingAlgorithm = algorithmParts[1];
                continue;
            } else if (hashingAlgorithm == null) {
                throw new IllegalArgumentException("Resource format is invalid: no hashing algorithm");
            } else if (line.startsWith("===RELOCATIONS") && relocationStep == -1) {
                relocationStep = 0;
                continue;
            } else if (relocationStep != -1) {
                switch (relocationStep) {
                    case 0:
                        pattern = line;
                        break;
                    case 1:
                        replacement = line;
                        break;
                    case 2:
                    case 3:
                        if (!line.startsWith("[") || !line.endsWith("]")) {
                            throw new IllegalArgumentException("Resource format is invalid: expecting a includes/excludes: " + line);
                        }
                        line = line.substring(1, line.length() - 1);

                        Set<String> values;
                        if (!line.trim().isEmpty()) {
                            values = new HashSet<>(Arrays.asList(line.split(",")));
                        } else {
                            values = Collections.emptySet();
                        }

                        if (relocationStep == 2) {
                            include = values;
                        } else {
                            if (replacement == null) {
                                throw new IllegalStateException("Replacement may not be null");
                            }
                            relocations.add(new Relocation(pattern, replacement, include, values));
                            include = null;
                            relocationStep = 0;
                            continue;
                        }
                }

                relocationStep++;
                continue;
            }

            String[] parts = line.split(" ");
            if (parts.length != 2) {
                throw new IllegalArgumentException("Resource format is invalid: invalid dependency: " + line);
            }
            String maven = parts[0];
            String hash = parts[1];

            String[] mavenParts = maven.split(":");
            int partCount = mavenParts.length;
            if (partCount < 3 || partCount > 5) {
                throw new IllegalArgumentException("Resource format is invalid: invalid dependency GAV: " + maven + " (" + partCount + ")");
            }

            String group = mavenParts[0];
            String artifact = mavenParts[1];
            String version = mavenParts[2];
            String snapshotTimestamp;
            String classifier;
            if (version.endsWith("-SNAPSHOT")) {
                snapshotTimestamp = mavenParts[3];
                if (partCount == 5) {
                    classifier = mavenParts[4];
                } else {
                    classifier = null;
                }
            } else {
                snapshotTimestamp = null;
                if (partCount == 4) {
                    classifier = mavenParts[3];
                } else {
                    classifier = null;
                }
            }

            Dependency dependency;
            if (snapshotTimestamp == null) {
                dependency = new StandardDependency(
                        group,
                        artifact,
                        version,
                        classifier,
                        hash,
                        hashingAlgorithm
                );
            } else {
                dependency = new SnapshotDependency(
                        group,
                        artifact,
                        version,
                        classifier,
                        snapshotTimestamp,
                        hash,
                        hashingAlgorithm
                );
            }

            dependencies.add(dependency);
        }
        if (relocationStep > 1) { // has pattern & replacement w/o include and/or exclude
            relocations.add(new Relocation(pattern, replacement, include, null));
        }
    }

    /**
     * Gets the dependencies defined in the resource.
     * @return the dependencies defined in this resource
     */
    public List<Dependency> getDependencies() {
        return dependencies;
    }

    /**
     * Gets the relocations defined in the resource.
     * @return the relocations defined in this resource
     */
    public List<Relocation> getRelocations() {
        return relocations;
    }
}
