/*
 * Decompiled with CFR 0.152.
 */
package digital.pragmatech.testing;

import digital.pragmatech.testing.ContextCacheEntry;
import digital.pragmatech.testing.OptimizationStatistics;
import digital.pragmatech.testing.TimelineData;
import digital.pragmatech.testing.TimelineEntry;
import digital.pragmatech.testing.optimization.ContextOptimizationOpportunity;
import digital.pragmatech.testing.reporting.ContextTimelineEvent;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.test.context.MergedContextConfiguration;

public class ContextCacheTracker {
    private static final Logger logger = LoggerFactory.getLogger(ContextCacheTracker.class);
    private final Map<MergedContextConfiguration, List<String>> contextToTestMethods = new ConcurrentHashMap<MergedContextConfiguration, List<String>>();
    private final Map<MergedContextConfiguration, ContextCacheEntry> cacheEntries = new ConcurrentHashMap<MergedContextConfiguration, ContextCacheEntry>();
    private final Map<String, MergedContextConfiguration> testClassToContext = new ConcurrentHashMap<String, MergedContextConfiguration>();
    private final List<MergedContextConfiguration> contextCreationOrder = new CopyOnWriteArrayList<MergedContextConfiguration>();
    private final AtomicInteger totalContextsCreated = new AtomicInteger(0);
    private final AtomicInteger cacheHits = new AtomicInteger(0);
    private final AtomicInteger cacheMisses = new AtomicInteger(0);

    public void recordTestClassForContext(MergedContextConfiguration config, String testClassName) {
        this.testClassToContext.put(testClassName, config);
        this.cacheEntries.computeIfAbsent(config, k -> {
            ContextCacheEntry entry = new ContextCacheEntry(config);
            logger.debug("Created new context cache entry for config: {}", (Object)config);
            return entry;
        }).addTestClass(testClassName);
    }

    public void recordTestMethodForContext(MergedContextConfiguration config, String testClassName, String methodName) {
        String testMethodIdentifier = testClassName + "." + methodName;
        this.contextToTestMethods.computeIfAbsent(config, k -> new CopyOnWriteArrayList()).add(testMethodIdentifier);
        ContextCacheEntry entry = this.cacheEntries.get(config);
        if (entry != null) {
            entry.addTestMethod(testClassName, methodName);
        }
        logger.debug("Recorded test method {} for context config: {}", (Object)testMethodIdentifier, (Object)config);
    }

    public void recordContextCreation(MergedContextConfiguration config, long loadTimeMs) {
        ContextCacheEntry entry = this.cacheEntries.get(config);
        if (entry != null) {
            MergedContextConfiguration nearestConfig;
            entry.recordCreation(loadTimeMs);
            this.contextCreationOrder.add(config);
            this.totalContextsCreated.incrementAndGet();
            this.cacheMisses.incrementAndGet();
            if (this.contextCreationOrder.size() > 1 && (nearestConfig = this.findNearestContext(config)) != null) {
                entry.setNearestContext(nearestConfig);
                logger.info("New context {} is most similar to existing context {} (load time: {}ms)", new Object[]{config, nearestConfig, loadTimeMs});
            }
        }
    }

    public void recordContextCreation(MergedContextConfiguration config, long loadTimeMs, long heapMemoryUsedBytes, int availableProcessors) {
        ContextCacheEntry entry = this.cacheEntries.get(config);
        if (entry != null) {
            MergedContextConfiguration nearestConfig;
            entry.recordCreation(loadTimeMs);
            entry.setContextDiagnostic(heapMemoryUsedBytes, availableProcessors);
            this.contextCreationOrder.add(config);
            this.totalContextsCreated.incrementAndGet();
            this.cacheMisses.incrementAndGet();
            if (this.contextCreationOrder.size() > 1 && (nearestConfig = this.findNearestContext(config)) != null) {
                entry.setNearestContext(nearestConfig);
                logger.info("New context {} is most similar to existing context {} (load time: {}ms, heap: {}MB, processors: {})", new Object[]{config, nearestConfig, loadTimeMs, (double)heapMemoryUsedBytes / 1048576.0, availableProcessors});
            }
        }
    }

    public void recordBeanDefinitions(MergedContextConfiguration config, String[] beanNames) {
        ContextCacheEntry entry = this.cacheEntries.get(config);
        if (entry != null) {
            entry.setBeanDefinitions(beanNames);
            logger.debug("Recorded {} bean definitions for context: {}", (Object)beanNames.length, (Object)config);
        }
    }

    public void recordContextCacheHit(MergedContextConfiguration config) {
        ContextCacheEntry entry = this.cacheEntries.get(config);
        if (entry != null) {
            entry.recordCacheHit();
            this.cacheHits.incrementAndGet();
        }
    }

    private MergedContextConfiguration findNearestContext(MergedContextConfiguration targetConfig) {
        if (targetConfig == null) {
            return null;
        }
        MergedContextConfiguration nearestConfig = null;
        int highestScore = 0;
        for (Map.Entry<MergedContextConfiguration, ContextCacheEntry> entry : this.cacheEntries.entrySet()) {
            int score;
            ContextCacheEntry candidate;
            if (entry.getKey().equals((Object)targetConfig) || !(candidate = entry.getValue()).isCreated() || (score = this.calculateSimilarityScore(targetConfig, entry.getKey())) <= highestScore) continue;
            highestScore = score;
            nearestConfig = entry.getKey();
        }
        return nearestConfig;
    }

    private int calculateSimilarityScore(MergedContextConfiguration config1, MergedContextConfiguration config2) {
        int score = 0;
        HashSet<Class> classes1 = new HashSet<Class>(Arrays.asList(config1.getClasses()));
        HashSet<Class> classes2 = new HashSet<Class>(Arrays.asList(config2.getClasses()));
        HashSet<Class> commonClasses = new HashSet<Class>(classes1);
        commonClasses.retainAll(classes2);
        score += commonClasses.size() * 10;
        HashSet<String> profiles1 = new HashSet<String>(Arrays.asList(config1.getActiveProfiles()));
        HashSet<String> profiles2 = new HashSet<String>(Arrays.asList(config2.getActiveProfiles()));
        if (profiles1.equals(profiles2)) {
            score += 5;
        }
        if (config1.getContextLoader() != null && config2.getContextLoader() != null && config1.getContextLoader().getClass().equals(config2.getContextLoader().getClass())) {
            score += 3;
        }
        HashSet<String> props1 = new HashSet<String>(Arrays.asList(config1.getPropertySourceProperties()));
        HashSet<String> props2 = new HashSet<String>(Arrays.asList(config2.getPropertySourceProperties()));
        HashSet<String> commonProps = new HashSet<String>(props1);
        commonProps.retainAll(props2);
        score += commonProps.size();
        if (config1.getContextInitializerClasses().equals(config2.getContextInitializerClasses())) {
            score += 2;
        }
        return score;
    }

    public Collection<ContextCacheEntry> getAllEntries() {
        return Collections.unmodifiableCollection(this.cacheEntries.values());
    }

    public Optional<MergedContextConfiguration> getContextForTestClass(String testClassName) {
        return Optional.ofNullable(this.testClassToContext.get(testClassName));
    }

    public Optional<ContextCacheEntry> getCacheEntry(MergedContextConfiguration config) {
        return Optional.ofNullable(this.cacheEntries.get(config));
    }

    public Optional<Instant> getEarliestContextCreationTime() {
        return this.cacheEntries.values().stream().filter(ContextCacheEntry::isCreated).map(ContextCacheEntry::getCreationTime).filter(Objects::nonNull).min(Instant::compareTo);
    }

    public Optional<Instant> getLatestContextAccessTime() {
        return this.cacheEntries.values().stream().map(ContextCacheEntry::getLastUsedTime).filter(Objects::nonNull).max(Instant::compareTo);
    }

    public OptimizationStatistics calculateOptimizationStatistics() {
        List<ContextCacheEntry> createdEntries = this.cacheEntries.values().stream().filter(ContextCacheEntry::isCreated).collect(Collectors.toList());
        if (createdEntries.isEmpty()) {
            return new OptimizationStatistics(0L, 0L, 0L, 0, Collections.emptyList());
        }
        long totalContextCreationTimeMs = createdEntries.stream().mapToLong(ContextCacheEntry::getContextLoadTimeMs).sum();
        long potentialTimeSavingsMs = this.calculatePotentialTimeSavings(createdEntries);
        long wastedTimeMs = this.calculateWastedTime(createdEntries);
        List<ContextOptimizationOpportunity> opportunities = this.identifyOptimizationOpportunities(createdEntries);
        return new OptimizationStatistics(totalContextCreationTimeMs, potentialTimeSavingsMs, wastedTimeMs, createdEntries.size(), opportunities);
    }

    private long calculatePotentialTimeSavings(List<ContextCacheEntry> entries) {
        long savings = 0L;
        for (ContextCacheEntry entry : entries) {
            ContextCacheEntry nearestEntry;
            Optional<MergedContextConfiguration> nearest = entry.getNearestContext();
            if (!nearest.isPresent() || (nearestEntry = this.cacheEntries.get(nearest.get())) == null || !nearestEntry.isCreated()) continue;
            savings += entry.getContextLoadTimeMs();
        }
        return savings;
    }

    private long calculateWastedTime(List<ContextCacheEntry> entries) {
        return entries.stream().filter(entry -> entry.getContextLoadTimeMs() > 1000L).mapToLong(entry -> entry.getContextLoadTimeMs() - 1000L).sum();
    }

    private List<ContextOptimizationOpportunity> identifyOptimizationOpportunities(List<ContextCacheEntry> entries) {
        return entries.stream().filter(entry -> entry.getContextLoadTimeMs() > 500L).sorted((a, b) -> Long.compare(b.getContextLoadTimeMs(), a.getContextLoadTimeMs())).limit(5L).map(entry -> {
            String recommendation = this.generateRecommendation((ContextCacheEntry)entry);
            return new ContextOptimizationOpportunity(entry.getTestClasses().iterator().next(), entry.getContextLoadTimeMs(), entry.getBeanDefinitionCount(), recommendation);
        }).collect(Collectors.toList());
    }

    private String generateRecommendation(ContextCacheEntry entry) {
        Optional<MergedContextConfiguration> nearest = entry.getNearestContext();
        if (nearest.isPresent()) {
            return "Consider harmonizing with similar context to save " + entry.getContextLoadTimeMs() + "ms";
        }
        if (entry.getBeanDefinitionCount() > 100) {
            return "Large context (" + entry.getBeanDefinitionCount() + " beans) - consider using @TestConfiguration to reduce scope";
        }
        if (entry.getContextLoadTimeMs() > 2000L) {
            return "Slow context load (" + entry.getContextLoadTimeMs() + "ms) - review component scanning and auto-configuration";
        }
        return "Consider optimizing test setup to reduce context load time";
    }

    public TimelineData getTimelineData() {
        List createdEntries = this.cacheEntries.values().stream().filter(ContextCacheEntry::isCreated).sorted((a, b) -> {
            Instant timeA = a.getCreationTime();
            Instant timeB = b.getCreationTime();
            if (timeA == null && timeB == null) {
                return 0;
            }
            if (timeA == null) {
                return 1;
            }
            if (timeB == null) {
                return -1;
            }
            return timeA.compareTo(timeB);
        }).collect(Collectors.toList());
        if (createdEntries.isEmpty()) {
            return new TimelineData(Collections.emptyList(), null, null, Collections.emptyList());
        }
        Instant earliestCreation = ((ContextCacheEntry)createdEntries.get(0)).getCreationTime();
        Instant latestAccess = createdEntries.stream().map(ContextCacheEntry::getLastUsedTime).filter(Objects::nonNull).max(Instant::compareTo).orElse(Instant.now());
        ArrayList<ContextTimelineEvent> events = new ArrayList<ContextTimelineEvent>();
        List<String> contextColors = Arrays.asList("#e74c3c", "#3498db", "#27ae60", "#f39c12", "#9b59b6", "#e67e22", "#1abc9c", "#34495e", "#e91e63", "#ff5722");
        for (int i = 0; i < createdEntries.size(); ++i) {
            ContextCacheEntry entry = (ContextCacheEntry)createdEntries.get(i);
            Object contextLabel = "Context " + (i + 1);
            if (!entry.getTestClasses().isEmpty()) {
                String firstTestClass = entry.getTestClasses().iterator().next();
                String simpleName = firstTestClass.substring(firstTestClass.lastIndexOf(46) + 1);
                contextLabel = simpleName;
            }
            String color = contextColors.get(i % contextColors.size());
            long creationTimeSeconds = Duration.between(earliestCreation, entry.getCreationTime()).toMillis() / 1000L;
            events.add(new ContextTimelineEvent((String)contextLabel, color, creationTimeSeconds, entry.getContextLoadTimeMs(), entry.getTestClasses().size(), entry.getHitCount(), entry.getBeanDefinitionCount()));
        }
        ArrayList<TimelineEntry> timelineEntries = new ArrayList<TimelineEntry>();
        for (int i = 0; i < createdEntries.size(); ++i) {
            ContextCacheEntry entry = (ContextCacheEntry)createdEntries.get(i);
            String contextLabel = ((ContextTimelineEvent)events.get(i)).contextName();
            long creationStartMs = Duration.between(earliestCreation, entry.getCreationTime()).toMillis();
            timelineEntries.add(new TimelineEntry(contextLabel, "Creation", creationStartMs, creationStartMs + entry.getContextLoadTimeMs(), ((ContextTimelineEvent)events.get(i)).color(), entry.getContextLoadTimeMs() + "ms load time", entry.getConfiguration().hashCode()));
        }
        return new TimelineData(timelineEntries, earliestCreation, latestAccess, events);
    }

    public void clear() {
        this.contextToTestMethods.clear();
        this.cacheEntries.clear();
        this.testClassToContext.clear();
        this.contextCreationOrder.clear();
        this.totalContextsCreated.set(0);
        this.cacheHits.set(0);
        this.cacheMisses.set(0);
    }
}

