package edu.byu.hbll.solr;

import edu.byu.hbll.misc.Resources;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.CloudSolrClient;
import org.apache.solr.client.solrj.impl.Http2SolrClient;
import org.apache.solr.client.solrj.impl.LBHttp2SolrClient;
import org.apache.solr.client.solrj.impl.ZkClientClusterStateProvider;
import org.apache.solr.client.solrj.request.CollectionAdminRequest;
import org.apache.solr.client.solrj.request.ConfigSetAdminRequest;

/**
 * Creates or reloads a SolrCloud collection based on the provided configset, or an existing
 * configset if one is not provided.
 *
 * <p>Example:
 *
 * <pre>{@code
 * new SolrCollectionInitializer()
 *     .zkHosts(Arrays.asList("zk1.example.com","zk2.example.com","zk3.example.com"))
 *     .chroot("/solr")
 *     .collectionName("search")
 *     .configsetName("search")
 *     .configsetPath(Paths.get("configsets/search/conf"))
 *     .shardCount(1)
 *     .replicaCount(2)
 *     .initialize();
 * }</pre>
 *
 * @author bwelker
 */
public class SolrCollectionInitializer {

  /** Default Zookeeper host list: ["localhost"]. */
  public static final List<String> DEFAULT_ZK_HOSTS = List.of("localhost");

  /** Default Zookeeper chroot path: "/solr". */
  public static final String DEFAULT_CHROOT = "/solr";

  /** Default shard count: 1. */
  public static final int DEFAULT_SHARD_COUNT = 1;

  /** Default replica count: 1. */
  public static final int DEFAULT_REPLICA_COUNT = 1;

  /** The zookeeper hosts. */
  private List<String> zkHosts = DEFAULT_ZK_HOSTS;
  /** The zookeeper chroot. */
  private String chroot = DEFAULT_CHROOT;
  /** The Solr base uris. */
  private List<String> baseUris = List.of();
  /** The collection to create. */
  private String collectionName;
  /** The name of the configset. */
  private String configsetName;
  /** The configset to create. */
  private Path configsetPath;
  /** The name of the classpath resource containing the configset. */
  private String configsetResourceName;
  /** The number of shards to create. */
  private int shardCount = DEFAULT_SHARD_COUNT;
  /** The number of replicas to create. */
  private int replicaCount = DEFAULT_REPLICA_COUNT;

  /**
   * Sets the zookeeper hosts for this initializer.
   *
   * @param zkHosts the zookeeper hosts.
   * @return SolrCollectionInitializer
   */
  public SolrCollectionInitializer zkHosts(List<String> zkHosts) {
    this.zkHosts = Objects.requireNonNull(zkHosts);
    if (zkHosts.isEmpty()) {
      throw new IllegalArgumentException("Zookeeper Hosts cannot be empty!");
    }
    return this;
  }

  /**
   * Sets the zookeeper chroot for this initializer.
   *
   * @param chroot the zookeeper chroot.
   * @return SolrCollectionInitializer
   */
  public SolrCollectionInitializer chroot(String chroot) {
    this.chroot = Objects.requireNonNull(chroot);
    return this;
  }

  /**
   * Sets the solr base uris. If set, zkHosts and chroot are ignored.
   *
   * <p>Note: The recommended practice is to specify solr base uris rather than interface directly
   * with zookeeper.
   * https://solr.apache.org/guide/solr/latest/upgrade-notes/major-changes-in-solr-9.html#solrj
   *
   * @param baseUris the solr base uris.
   * @return SolrCollectionInitializer
   */
  public SolrCollectionInitializer baseUris(List<String> baseUris) {
    this.baseUris = Objects.requireNonNull(baseUris);
    return this;
  }

  /**
   * Sets the collection name for this initializer.
   *
   * @param collectionName the collection name.
   * @return SolrCollectionInitializer
   */
  public SolrCollectionInitializer collectionName(String collectionName) {
    this.collectionName = Objects.requireNonNull(collectionName);
    return this;
  }

  /**
   * Sets the configset name for this initializer.
   *
   * @param configsetName the name of the configset.
   * @return SolrCollectionInitializer
   */
  public SolrCollectionInitializer configsetName(String configsetName) {
    this.configsetName = configsetName;
    return this;
  }

  /**
   * Sets the location of the configset for this initializer.
   *
   * @param configsetPath the location of the configset.
   * @return SolrCollectionInitializer
   */
  public SolrCollectionInitializer configsetPath(Path configsetPath) {
    this.configsetPath = configsetPath;
    return this;
  }

  /**
   * Sets the name of the classpath resource containing the configset.
   *
   * @param configsetResourceName the name of the classpath resource containing the configset.
   * @return SolrCollectionInitializer
   */
  public SolrCollectionInitializer configsetResourceName(String configsetResourceName) {
    this.configsetResourceName = configsetResourceName;
    return this;
  }

  /**
   * Sets the shard count for this initializer.
   *
   * @param shardCount the shard count for the collection.
   * @return SolrCollectionInitializer
   */
  public SolrCollectionInitializer shardCount(int shardCount) {
    this.shardCount = shardCount;
    if (shardCount < 1) {
      throw new IllegalArgumentException("Shard Count must be greater than 0.");
    }
    return this;
  }

  /**
   * Sets the replica count for this initializer.
   *
   * @param replicaCount the replica count for the collection.
   * @return SolrCollectionInitializer
   */
  public SolrCollectionInitializer replicaCount(int replicaCount) {
    this.replicaCount = replicaCount;
    if (replicaCount < 1) {
      throw new IllegalArgumentException("Replica Count must be greater than 0.");
    }
    return this;
  }

  /**
   * Creates a {@link CloudSolrClient} based on the parameters set including the default collection.
   *
   * @return a new Solr client
   */
  public CloudSolrClient createCloudClient() {
    CloudSolrClient client;

    if (!baseUris.isEmpty()) {
      client = new CloudSolrClient.Builder(baseUris).build();
    } else {
      client = new CloudSolrClient.Builder(zkHosts, Optional.ofNullable(chroot)).build();
    }

    client.setDefaultCollection(collectionName);
    return client;
  }

  /**
   * Creates a Solr client based on an {@link LBHttp2SolrClient} and the parameters set including
   * the default collection.
   *
   * @return a new Solr client
   */
  public SolrClient createHttpClient() {
    Http2SolrClient httpSolrClient = new Http2SolrClient.Builder().build();
    String[] baseUris = this.baseUris.toArray(new String[] {});
    LBHttp2SolrClient solrClient = new LBHttp2SolrClient(httpSolrClient, baseUris);
    return new CollectionSolrClient(solrClient, collectionName);
  }

  /**
   * Uploads the solr config (if one is provided), and initializes or reloads the solr collection
   * with the config. This is done using the {@link LBHttp2SolrClient}.
   *
   * @return a solr client for this collection
   * @throws IOException If an IO error occurs.
   * @throws SolrServerException If a Solr error occurs.
   */
  public SolrClient initializeHttp() throws IOException, SolrServerException {
    SolrClient client = createHttpClient();
    initialize(client);
    return client;
  }

  /**
   * Uploads the solr config (if one is provided), and initializes or reloads the solr collection
   * with the config. This is done using the {@link CloudSolrClient}.
   *
   * @return a solr client for this collection
   * @throws IOException If an IO error occurs.
   * @throws SolrServerException If a Solr error occurs.
   */
  public CloudSolrClient initialize() throws IOException, SolrServerException {
    CloudSolrClient client = createCloudClient();
    initialize(client);
    return client;
  }

  /**
   * Uploads the solr config (if one is provided), and initializes or reloads the solr collection
   * with the config.
   *
   * @param client the Solr client to use
   * @throws IOException If an IO error occurs.
   * @throws SolrServerException If a Solr error occurs.
   */
  private void initialize(SolrClient client) throws IOException, SolrServerException {
    if (configsetName == null) {
      configsetName = collectionName;
    }

    Path configsetPath =
        this.configsetPath == null && configsetResourceName != null
            ? Resources.extract(configsetResourceName)
            : this.configsetPath;

    if (configsetPath != null) {
      if (!baseUris.isEmpty()) {
        ConfigSetAdminRequest.Upload uploadRequest = new ConfigSetAdminRequest.Upload();
        uploadRequest.setConfigSetName(configsetName);
        uploadRequest.setUploadFile(zipDirectory(configsetPath).toFile(), "application/zip");
        uploadRequest.setOverwrite(true);
        uploadRequest.setCleanup(true);
        client.request(uploadRequest);
      } else {
        try (ZkClientClusterStateProvider provider =
            new ZkClientClusterStateProvider(zkHosts, chroot)) {
          provider.uploadConfig(configsetPath, configsetName);
        }
      }
    }

    if (collectionName != null) {
      List<String> existingCollectionNames = CollectionAdminRequest.listCollections(client);

      // If the collection exists, reload it. Otherwise, create it.
      if (existingCollectionNames.contains(collectionName)) {
        client.request(CollectionAdminRequest.reloadCollection(collectionName));
      } else {
        client.request(
            CollectionAdminRequest.createCollection(
                collectionName, configsetName, shardCount, replicaCount));
      }
    }
  }

  /**
   * Zips the given directory into a temporary file and returns it.
   *
   * @param directory the directory to zip
   * @return the zipped archive
   */
  private Path zipDirectory(Path directory) {
    try {
      Path zipFile = Files.createTempFile("dir.", ".zip");

      try (ZipOutputStream archive = new ZipOutputStream(Files.newOutputStream(zipFile))) {
        Files.walk(directory)
            .filter(p -> !p.equals(directory))
            .forEach(
                p -> {
                  try {
                    archive.putNextEntry(new ZipEntry(directory.relativize(p).toString()));

                    if (!Files.isDirectory(p)) {
                      try (InputStream fis = Files.newInputStream(p)) {
                        fis.transferTo(archive);
                      }
                    }

                    archive.closeEntry();
                  } catch (IOException e) {
                    throw new UncheckedIOException(e);
                  }
                });
      }

      return zipFile;
    } catch (IOException e) {
      throw new UncheckedIOException(e);
    }
  }
}
