package edu.byu.hbll.solr;

import com.fasterxml.jackson.databind.JsonNode;
import java.time.Instant;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import org.apache.solr.common.SolrInputDocument;

/**
 * Class which builds a solr input document from a json document. All documents in solr are flat,
 * this class generically flattens the json document.
 *
 * @author danielrodziewicz
 */
public class SolrDocumentBuilder {

  private static final String MULTIVALUE_SUFFIX = "s";

  /**
   * Method which returns the {@link SolrInputDocument} generated from the {@link JsonNode}; the
   * {@link SolrInputDocument} is a flattened version of the {@link JsonNode}.
   *
   * @param node the {@link JsonNode} to be flattened.
   * @return the {@link SolrInputDocument} generated.
   */
  public SolrInputDocument buildSolrInputDocument(JsonNode node) {
    return buildSolrInputDocument(node, Collections.emptySet());
  }

  /**
   * Method which returns the {@link SolrInputDocument} generated from the {@link JsonNode}; the
   * {@link SolrInputDocument} is a flattened version of the {@link JsonNode}.
   *
   * @param node the {@link JsonNode} to be flattened.
   * @param ignoreFields List of field names to ignore.
   * @return the {@link SolrInputDocument} generated.
   */
  public SolrInputDocument buildSolrInputDocument(JsonNode node, Collection<String> ignoreFields) {
    SolrInputDocument solrInputDocument = new SolrInputDocument();
    // Recursively add fields
    addValues(solrInputDocument, node, "", false, ignoreFields);
    return solrInputDocument;
  }

  /**
   * Helper method which recursively adds fields to solr flat document.
   *
   * @param solrInputDocument the Solr document being populated
   * @param node the JSON document to be flattened
   * @param name the current field name
   * @param multivalued whether or not this node represents multivalued fields
   * @param ignoreFields field names included in this collection will be ignored
   */
  private void addValues(
      SolrInputDocument solrInputDocument,
      JsonNode node,
      String name,
      boolean multivalued,
      Collection<String> ignoreFields) {

    // Don't do any processing on fields that are marked to be ignored.
    if (ignoreFields.contains(name)) {
      return;
    }

    if (node.isValueNode()) {
      addValues(solrInputDocument, node, name, multivalued);
    } else if (node.isArray()) {
      node.forEach(element -> addValues(solrInputDocument, element, name, true, ignoreFields));
    } else if (node.isObject()) {
      Iterator<String> it = node.fieldNames();
      while (it.hasNext()) {
        String fieldName = it.next();
        JsonNode child = node.get((fieldName));
        addValues(solrInputDocument, child, buildName(name, fieldName), multivalued, ignoreFields);
      }
    }
  }

  /**
   * Helper method to add the values to the {@link SolrInputDocument} being generated.
   *
   * @param doc the Solr document being populated
   * @param name the generated field name to be added to the Solr document
   * @param node the JSON document to be flattened
   * @param multivalued whether or not this node represents multivalued fields
   */
  private void addValues(SolrInputDocument doc, JsonNode node, String name, boolean multivalued) {
    String suffix = "_s";
    if (node.isBoolean()) {
      suffix = "_b";
    } else if (node.isNumber()) {
      suffix = "_d";
    } else if (isInstant(node.asText())) {
      suffix = "_dt";
    }
    addField(doc, name, multivalued, node.asText(), suffix);
  }

  /**
   * Helper method to change the type of field being added based on the multivalued flag.
   *
   * @param doc the document to populate
   * @param name the name of the field to add
   * @param multivalued whether or not this node represents multivalued fields
   * @param value the value to add
   * @param type the field type to add to the document
   */
  private void addField(
      SolrInputDocument doc, String name, boolean multivalued, String value, String type) {
    if (multivalued) {
      doc.addField(name + type + MULTIVALUE_SUFFIX, value);
    } else {
      doc.addField(name + type, value);
    }
  }

  /**
   * Helper method to check if a string is an ISO-8601 {@link Instant}.
   *
   * @param value the string to be tested
   * @return boolean {@code true} if the value represents an instant; {@code false} otherwise
   */
  private boolean isInstant(String value) {
    try {
      Instant.parse(value);
      return true;
    } catch (Exception e) {
      return false;
    }
  }

  /**
   * Method which builds the names for flattened fields.
   *
   * @param name the current field name.
   * @param field the field to be appended to the name.
   * @return the new field name.
   */
  private String buildName(String name, String field) {
    if (name.isEmpty()) {
      return field;
    } else {
      return name + "_" + field;
    }
  }

  /**
   * Method which adds custom string fields to {@link SolrInputDocument}.
   *
   * @param valueMap a map containing the keys and values to be added.
   * @param document the {@link SolrInputDocument} to add the keys and values to.
   */
  public void addCustomStringFields(Map<String, String> valueMap, SolrInputDocument document) {
    valueMap.forEach(
        (key, value) -> {
          document.addField(key, value);
        });
  }

  /**
   * Method which adds custom integer fields to {@link SolrInputDocument}.
   *
   * @param valueMap a map containing the keys and values to be added.
   * @param document the {@link SolrInputDocument} to add the keys and values to.
   */
  public void addCustomIntegerFields(Map<String, Integer> valueMap, SolrInputDocument document) {
    valueMap.forEach(
        (key, value) -> {
          document.addField(key, value);
        });
  }

  /**
   * Method which adds custom boolean fields to {@link SolrInputDocument}.
   *
   * @param valueMap a map containing the keys and values to be added.
   * @param document the {@link SolrInputDocument} to add the keys and values to.
   */
  public void addCustomBooleanFields(Map<String, Boolean> valueMap, SolrInputDocument document) {
    valueMap.forEach(
        (key, value) -> {
          document.addField(key, value);
        });
  }

  /**
   * Method which adds custom long fields to {@link SolrInputDocument}.
   *
   * @param valueMap a map containing the keys and values to be added.
   * @param document the {@link SolrInputDocument} to add the keys and values to.
   */
  public void addCustomLongFields(Map<String, Long> valueMap, SolrInputDocument document) {
    valueMap.forEach(
        (key, value) -> {
          document.addField(key, value);
        });
  }
}
