package edu.byu.hbll.solr.auto;

import edu.byu.hbll.solr.SolrCollectionInitializer;
import java.io.IOException;
import javax.annotation.PreDestroy;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.CloudSolrClient;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * Auto configuration for a {@link CloudSolrClient}.
 *
 * @author Charles Draper
 */
@Configuration
@EnableConfigurationProperties(SolrProperties.class)
public class SolrAutoConfiguration {

  private SolrProperties properties;
  private SolrClient solrClient;

  /**
   * Creates a new configuration with the given properties.
   *
   * @param properties the properties
   */
  public SolrAutoConfiguration(SolrProperties properties) {
    this.properties = properties;
  }

  /**
   * Creates the {@link SolrClient} bean.
   *
   * @return the solr client
   * @throws SolrServerException if a Solr error occurs
   * @throws IOException if an I/O error occurs
   */
  @Bean
  @ConditionalOnMissingBean
  public SolrClient client() throws IOException, SolrServerException {
    if (!properties.isEnabled()) {
      return null;
    }

    SolrCollectionInitializer initializer =
        new SolrCollectionInitializer()
            .zkHosts(properties.getZkHosts())
            .chroot(properties.getChroot())
            .baseUris(properties.getBaseUris())
            .collectionName(properties.getCollectionName())
            .configsetName(properties.getConfigsetName())
            .configsetPath(properties.getConfigsetPath())
            .configsetResourceName(properties.getConfigsetResourceName())
            .shardCount(properties.getShardCount())
            .replicaCount(properties.getReplicaCount());

    if (properties.isInitialize()) {
      if (properties.isHttp()) {
        return initializer.initializeHttp();
      } else {
        return initializer.initialize();
      }
    } else {
      if (properties.isHttp()) {
        return initializer.createHttpClient();
      } else {
        return initializer.createCloudClient();
      }
    }
  }

  /**
   * Closes the {@link SolrClient}.
   *
   * @throws IOException if an I/O error occurs
   */
  @PreDestroy
  public void preDestroy() throws IOException {
    if (solrClient != null) {
      solrClient.close();
    }
  }
}
