/*
 * Decompiled with CFR 0.152.
 */
package edu.iu.uits.mail;

import java.io.FileInputStream;
import java.io.IOException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Optional;
import java.util.Properties;
import javax.mail.Address;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1EncodableVector;
import org.bouncycastle.asn1.cms.AttributeTable;
import org.bouncycastle.asn1.cms.IssuerAndSerialNumber;
import org.bouncycastle.asn1.smime.SMIMECapabilitiesAttribute;
import org.bouncycastle.asn1.smime.SMIMECapability;
import org.bouncycastle.asn1.smime.SMIMECapabilityVector;
import org.bouncycastle.asn1.smime.SMIMEEncryptionKeyPreferenceAttribute;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.cert.jcajce.JcaCertStore;
import org.bouncycastle.cms.jcajce.JcaSimpleSignerInfoGeneratorBuilder;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.mail.smime.SMIMEException;
import org.bouncycastle.mail.smime.SMIMESignedGenerator;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.util.Store;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MailSigner {
    private static final Logger log = LoggerFactory.getLogger(MailSigner.class);
    public static final String CERT_PASSWORD_PROPERTY_TEMPLATE = "mail.keystore.%s.password";
    private static final String LOCAL_ADDRESS_REGEX = "^(.*)@.*$";
    private KeyStore keyStore;
    private Properties properties;

    public MailSigner(Properties properties) {
        this.properties = properties;
        String keyStoreFile = properties.getProperty("mail.keystore.file");
        String keyStorePassword = properties.getProperty("mail.keystore.password");
        try {
            if (keyStoreFile != null && !keyStoreFile.trim().isEmpty() && keyStorePassword != null && !keyStorePassword.trim().isEmpty()) {
                this.keyStore = KeyStore.getInstance("JKS");
                this.keyStore.load(new FileInputStream(keyStoreFile), keyStorePassword.toCharArray());
            } else {
                log.warn("No mail keystore file or password set.  No emails will be signed.");
            }
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            log.error("Caught exception attempting to load mail keystore.  No emails will be signed.", (Throwable)e);
        }
    }

    public MailSigner(Properties properties, KeyStore keyStore) {
        this.properties = properties;
        this.keyStore = keyStore;
    }

    public Optional<MimeMessage> signMessage(MimeMessage mimeMessage) {
        if (this.keyStore == null) {
            log.warn("No keystore provided so the message will not be signed");
            return Optional.empty();
        }
        try {
            Address[] from = mimeMessage.getFrom();
            Optional<String> signingAddress = Arrays.stream(from).map(Address::toString).filter(address -> {
                try {
                    return this.keyStore.containsAlias((String)address);
                }
                catch (KeyStoreException e) {
                    return false;
                }
            }).findFirst();
            if (signingAddress.isPresent()) {
                String alias = signingAddress.get();
                String emailKeyPassword = this.getEmailPassword(alias);
                return Optional.of(MailSigner.signMessage(mimeMessage, (PrivateKey)this.keyStore.getKey(alias, emailKeyPassword.toCharArray()), (X509Certificate)this.keyStore.getCertificateChain(alias)[0]));
            }
            log.info("Could not find an email certificate for any of the from addresses: " + from);
            return Optional.empty();
        }
        catch (KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | MessagingException e) {
            log.error("Caught exception when attempting to sign a message.  The message will be sent unsigned.", e);
            return Optional.empty();
        }
    }

    protected String getEmailPassword(String alias) {
        String emailKeyPassword = this.properties.getProperty(String.format(CERT_PASSWORD_PROPERTY_TEMPLATE, alias));
        if (emailKeyPassword == null || emailKeyPassword.trim().isEmpty()) {
            String localPart = alias.replaceAll(LOCAL_ADDRESS_REGEX, "$1");
            log.debug(String.format("No key password found for %s.  Trying local portion, %s.", alias, localPart));
            emailKeyPassword = this.properties.getProperty(String.format(CERT_PASSWORD_PROPERTY_TEMPLATE, localPart));
            if (emailKeyPassword == null || emailKeyPassword.trim().isEmpty()) {
                log.debug(String.format("No key password found for %s or %s.  Defaulting to using the keystore password.", alias, localPart));
                emailKeyPassword = this.properties.getProperty("mail.keystore.password");
            }
        }
        return emailKeyPassword;
    }

    public static MimeMessage signMessage(MimeMessage message, PrivateKey privateKey, X509Certificate certificate) {
        try {
            JcaCertStore certs = new JcaCertStore(Collections.singletonList(certificate));
            ASN1EncodableVector signedAttributes = new ASN1EncodableVector();
            SMIMECapabilityVector caps = new SMIMECapabilityVector();
            caps.addCapability(SMIMECapability.dES_EDE3_CBC);
            caps.addCapability(SMIMECapability.rC2_CBC, 128);
            caps.addCapability(SMIMECapability.dES_CBC);
            caps.addCapability(SMIMECapability.aES256_CBC);
            signedAttributes.add((ASN1Encodable)new SMIMECapabilitiesAttribute(caps));
            IssuerAndSerialNumber issuerAndSerialNumber = new IssuerAndSerialNumber(new X500Name(certificate.getIssuerDN().getName()), certificate.getSerialNumber());
            signedAttributes.add((ASN1Encodable)new SMIMEEncryptionKeyPreferenceAttribute(issuerAndSerialNumber));
            SMIMESignedGenerator gen = new SMIMESignedGenerator();
            gen.addSignerInfoGenerator(new JcaSimpleSignerInfoGeneratorBuilder().setProvider((Provider)new BouncyCastleProvider()).setSignedAttributeGenerator(new AttributeTable(signedAttributes)).build("SHA1withRSA", privateKey, certificate));
            gen.addCertificates((Store)certs);
            MimeBodyPart mimeBodyPart = new MimeBodyPart();
            Object messageContent = message.getContent();
            if (messageContent instanceof String) {
                mimeBodyPart.setContent(messageContent, message.getContentType());
            } else if (messageContent instanceof MimeMultipart) {
                mimeBodyPart.setContent((Multipart)((MimeMultipart)messageContent));
            }
            MimeMultipart signedMultipart = gen.generate(mimeBodyPart);
            MimeMessage signedMessage = new MimeMessage(message.getSession());
            signedMessage.setContent((Object)signedMultipart, signedMultipart.getContentType());
            Enumeration headers = message.getAllHeaderLines();
            while (headers.hasMoreElements()) {
                String headerLine = (String)headers.nextElement();
                if (headerLine.startsWith("Content-Type:")) continue;
                signedMessage.addHeaderLine(headerLine);
            }
            signedMessage.saveChanges();
            return signedMessage;
        }
        catch (IOException | CertificateEncodingException | MessagingException | SMIMEException | OperatorCreationException e) {
            log.error("Caught exception when attempting to sign a message. Message will be sent unsigned", e);
            return message;
        }
    }
}

