package edu.nps.moves.dis7;

import java.io.*;

/**
 * Removable parts that may be attached to an entity. Section 6.2.93.3
 *
 * Copyright (c) 2008-2016, MOVES Institute, Naval Postgraduate School. All
 * rights reserved. This work is licensed under the BSD open source license,
 * available at https://www.movesinstitute.org/licenses/bsd.html
 *
 * @author DMcG
 */
public class AttachedParts extends VariableParameter implements Serializable {

    /**
     * 0 = attached, 1 = detached. See I.2.3.1 for state transition diagram
     */
    protected short detachedIndicator = (short) 0;

    /**
     * the identification of the articulated part to which this articulation
     * parameter is attached. This field shall be specified by a 16-bit unsigned
     * integer. This field shall contain the value zero if the articulated part
     * is attached directly to the entity.
     */
    protected int partAttachedTo = (int) 0;

    /**
     * The location or station to which the part is attached
     */
    protected long parameterType;

    /**
     * The definition of the 64 bits shall be determined based on the type of
     * parameter specified in the Parameter Type field
     */
    protected EntityType attachedPartType = new EntityType();

    /**
     * Constructor
     */
    public AttachedParts() {
        recordType = (short) 1;
    }

    public int getMarshalledSize() {
        int marshalSize = 0;

        marshalSize = super.getMarshalledSize();
        marshalSize = marshalSize + 1;  // detachedIndicator
        marshalSize = marshalSize + 2;  // partAttachedTo
        marshalSize = marshalSize + 4;  // parameterType
        marshalSize = marshalSize + attachedPartType.getMarshalledSize();  // entityType

        return marshalSize;
    }

    public void setDetachedIndicator(short pDetachedIndicator) {
        detachedIndicator = pDetachedIndicator;
    }

    public short getDetachedIndicator() {
        return detachedIndicator;
    }

    public void setPartAttachedTo(int pPartAttachedTo) {
        partAttachedTo = pPartAttachedTo;
    }

    public int getPartAttachedTo() {
        return partAttachedTo;
    }

    public void setParameterType(long pParameterType) {
        parameterType = pParameterType;
    }

    public long getParameterType() {
        return parameterType;
    }

    public void setAttachedPartType(EntityType aAttachedPartType) {
        attachedPartType = aAttachedPartType;
    }

    public EntityType getAttachedPartType() {
        return attachedPartType;
    }

    public void marshal(DataOutputStream dos) {
        try {
            super.marshal(dos);
            dos.writeByte((byte) detachedIndicator);
            dos.writeShort((short) partAttachedTo);
            dos.writeInt((int) parameterType);
            attachedPartType.marshal(dos);
        } // end try  // end try  // end try  // end try 
        catch (Exception e) {
            System.out.println(e);
        }
    } // end of marshal method

    public void unmarshal(DataInputStream dis) {
        try {
            detachedIndicator = (short) dis.readUnsignedByte();
            partAttachedTo = (int) dis.readUnsignedShort();
            parameterType = dis.readInt();
            attachedPartType.unmarshal(dis);
        } // end try  // end try  // end try  // end try 
        catch (Exception e) {
            System.out.println(e);
        }
    } // end of unmarshal method 

    /**
     * Packs a Pdu into the ByteBuffer.
     *
     * @throws java.nio.BufferOverflowException if buff is too small
     * @throws java.nio.ReadOnlyBufferException if buff is read only
     * @see java.nio.ByteBuffer
     * @param buff The ByteBuffer at the position to begin writing
     * @since ??
     */
    public void marshal(java.nio.ByteBuffer buff) {
        super.marshal(buff);
        buff.put((byte) detachedIndicator);
        buff.putShort((short) partAttachedTo);
        buff.putInt((int) parameterType);
        attachedPartType.marshal(buff);
    } // end of marshal method

    /**
     * Unpacks a Pdu from the underlying data.
     *
     * @throws java.nio.BufferUnderflowException if buff is too small
     * @see java.nio.ByteBuffer
     * @param buff The ByteBuffer at the position to begin reading
     * @since ??
     */
    public void unmarshal(java.nio.ByteBuffer buff) {
        detachedIndicator = (short) (buff.get() & 0xFF);
        partAttachedTo = (int) (buff.getShort() & 0xFFFF);
        parameterType = buff.getInt();
        attachedPartType.unmarshal(buff);
    } // end of unmarshal method 


    /*
  * The equals method doesn't always work--mostly it works only on classes that consist only of primitives. Be careful.
     */
    @Override
    public boolean equals(Object obj) {

        if (this == obj) {
            return true;
        }

        if (obj == null) {
            return false;
        }

        if (getClass() != obj.getClass()) {
            return false;
        }

        return equalsImpl(obj);
    }

    /**
     * Compare all fields that contribute to the state, ignoring transient and
     * static fields, for <code>this</code> and the supplied object
     *
     * @param obj the object to compare to
     * @return true if the objects are equal, false otherwise.
     */
    public boolean equalsImpl(Object obj) {
        boolean ivarsEqual = true;

        if (!(obj instanceof AttachedParts)) {
            return false;
        }

        final AttachedParts rhs = (AttachedParts) obj;

        if (!(recordType == rhs.recordType)) {
            ivarsEqual = false;
        }
        if (!(detachedIndicator == rhs.detachedIndicator)) {
            ivarsEqual = false;
        }
        if (!(partAttachedTo == rhs.partAttachedTo)) {
            ivarsEqual = false;
        }
        if (!(parameterType == rhs.parameterType)) {
            ivarsEqual = false;
        }
        if (!(attachedPartType.equalsImpl(rhs.attachedPartType))) {
            ivarsEqual = false;
        }

        return ivarsEqual;
    }
} // end of class
