package edu.nps.moves.dis7;

import java.io.*;

/**
 * Information related to the status of a beam. This is contained in the beam
 * status field of the electromagnitec emission PDU. The first bit determines
 * whether the beam is active (0) or deactivated (1). Section 6.2.12.
 *
 * Copyright (c) 2008-2016, MOVES Institute, Naval Postgraduate School. All
 * rights reserved. This work is licensed under the BSD open source license,
 * available at https://www.movesinstitute.org/licenses/bsd.html
 *
 * @author DMcG
 */
public class BeamStatus extends Object implements Serializable {

    /**
     * First bit zero means beam is active, first bit = 1 means deactivated. The
     * rest is padding.
     */
    protected short beamState;

    /**
     * Constructor
     */
    public BeamStatus() {
    }

    public int getMarshalledSize() {
        int marshalSize = 0;

        marshalSize = marshalSize + 1;  // beamState

        return marshalSize;
    }

    public void setBeamState(short pBeamState) {
        beamState = pBeamState;
    }

    public short getBeamState() {
        return beamState;
    }

    /**
     * 0 active, 1 deactivated
     */
    public int getBeamState_beamState() {
        short val = (short) (this.beamState & (short) 0x1);
        return (int) (val >> 0);
    }

    /**
     * 0 active, 1 deactivated
     */
    public void setBeamState_beamState(int val) {
        short aVal = 0;
        this.beamState &= (short) (~0x1); // clear bits
        aVal = (short) (val << 0);
        this.beamState = (short) (this.beamState | aVal);
    }

    /**
     * padding
     */
    public int getBeamState_padding() {
        short val = (short) (this.beamState & (short) 0xFE);
        return (int) (val >> 1);
    }

    /**
     * padding
     */
    public void setBeamState_padding(int val) {
        short aVal = 0;
        this.beamState &= (short) (~0xFE); // clear bits
        aVal = (short) (val << 1);
        this.beamState = (short) (this.beamState | aVal);
    }

    public void marshal(DataOutputStream dos) {
        try {
            dos.writeByte((byte) beamState);
        } // end try 
        catch (Exception e) {
            System.out.println(e);
        }
    } // end of marshal method

    public void unmarshal(DataInputStream dis) {
        try {
            beamState = (short) dis.readUnsignedByte();
        } // end try 
        catch (Exception e) {
            System.out.println(e);
        }
    } // end of unmarshal method 

    /**
     * Packs a Pdu into the ByteBuffer.
     *
     * @throws java.nio.BufferOverflowException if buff is too small
     * @throws java.nio.ReadOnlyBufferException if buff is read only
     * @see java.nio.ByteBuffer
     * @param buff The ByteBuffer at the position to begin writing
     * @since ??
     */
    public void marshal(java.nio.ByteBuffer buff) {
        buff.put((byte) beamState);
    } // end of marshal method

    /**
     * Unpacks a Pdu from the underlying data.
     *
     * @throws java.nio.BufferUnderflowException if buff is too small
     * @see java.nio.ByteBuffer
     * @param buff The ByteBuffer at the position to begin reading
     * @since ??
     */
    public void unmarshal(java.nio.ByteBuffer buff) {
        beamState = (short) (buff.get() & 0xFF);
    } // end of unmarshal method 


    /*
  * The equals method doesn't always work--mostly it works only on classes that consist only of primitives. Be careful.
     */
    @Override
    public boolean equals(Object obj) {

        if (this == obj) {
            return true;
        }

        if (obj == null) {
            return false;
        }

        if (getClass() != obj.getClass()) {
            return false;
        }

        return equalsImpl(obj);
    }

    /**
     * Compare all fields that contribute to the state, ignoring transient and
     * static fields, for <code>this</code> and the supplied object
     *
     * @param obj the object to compare to
     * @return true if the objects are equal, false otherwise.
     */
    public boolean equalsImpl(Object obj) {
        boolean ivarsEqual = true;

        if (!(obj instanceof BeamStatus)) {
            return false;
        }

        final BeamStatus rhs = (BeamStatus) obj;

        if (!(beamState == rhs.beamState)) {
            ivarsEqual = false;
        }

        return ivarsEqual;
    }
} // end of class
