package edu.nps.moves.dis7;

import java.io.*;

/**
 * The live entity PDUs have a header with some different field names, but the
 * same length. Section 9.3.2
 *
 * Copyright (c) 2008-2016, MOVES Institute, Naval Postgraduate School. All
 * rights reserved. This work is licensed under the BSD open source license,
 * available at https://www.movesinstitute.org/licenses/bsd.html
 *
 * @author DMcG
 */
public class LiveEntityPdu extends PduSuperclass implements Serializable {

    /**
     * Subprotocol used to decode the PDU. Section 13 of EBV.
     */
    protected int subprotocolNumber;

    /**
     * zero-filled array of padding
     */
    protected short padding = (short) 0;

    /**
     * Constructor
     */
    public LiveEntityPdu() {
    }

    public int getMarshalledSize() {
        int marshalSize = 0;

        marshalSize = super.getMarshalledSize();
        marshalSize = marshalSize + 2;  // subprotocolNumber
        marshalSize = marshalSize + 1;  // padding

        return marshalSize;
    }

    public void setSubprotocolNumber(int pSubprotocolNumber) {
        subprotocolNumber = pSubprotocolNumber;
    }

    public int getSubprotocolNumber() {
        return subprotocolNumber;
    }

    public void setPadding(short pPadding) {
        padding = pPadding;
    }

    public short getPadding() {
        return padding;
    }

    public void marshal(DataOutputStream dos) {
        super.marshal(dos);
        try {
            dos.writeShort((short) subprotocolNumber);
            dos.writeByte((byte) padding);
        } // end try 
        catch (Exception e) {
            System.out.println(e);
        }
    } // end of marshal method

    public void unmarshal(DataInputStream dis) {
        super.unmarshal(dis);

        try {
            subprotocolNumber = (int) dis.readUnsignedShort();
            padding = (short) dis.readUnsignedByte();
        } // end try 
        catch (Exception e) {
            System.out.println(e);
        }
    } // end of unmarshal method 

    /**
     * Packs a Pdu into the ByteBuffer.
     *
     * @throws java.nio.BufferOverflowException if buff is too small
     * @throws java.nio.ReadOnlyBufferException if buff is read only
     * @see java.nio.ByteBuffer
     * @param buff The ByteBuffer at the position to begin writing
     * @since ??
     */
    public void marshal(java.nio.ByteBuffer buff) {
        super.marshal(buff);
        buff.putShort((short) subprotocolNumber);
        buff.put((byte) padding);
    } // end of marshal method

    /**
     * Unpacks a Pdu from the underlying data.
     *
     * @throws java.nio.BufferUnderflowException if buff is too small
     * @see java.nio.ByteBuffer
     * @param buff The ByteBuffer at the position to begin reading
     * @since ??
     */
    public void unmarshal(java.nio.ByteBuffer buff) {
        super.unmarshal(buff);

        subprotocolNumber = (int) (buff.getShort() & 0xFFFF);
        padding = (short) (buff.get() & 0xFF);
    } // end of unmarshal method 


    /*
  * The equals method doesn't always work--mostly it works only on classes that consist only of primitives. Be careful.
     */
    @Override
    public boolean equals(Object obj) {

        if (this == obj) {
            return true;
        }

        if (obj == null) {
            return false;
        }

        if (getClass() != obj.getClass()) {
            return false;
        }

        return equalsImpl(obj);
    }

    @Override
    public boolean equalsImpl(Object obj) {
        boolean ivarsEqual = true;

        if (!(obj instanceof LiveEntityPdu)) {
            return false;
        }

        final LiveEntityPdu rhs = (LiveEntityPdu) obj;

        if (!(subprotocolNumber == rhs.subprotocolNumber)) {
            ivarsEqual = false;
        }
        if (!(padding == rhs.padding)) {
            ivarsEqual = false;
        }

        return ivarsEqual && super.equalsImpl(rhs);
    }
} // end of class
