/*
 * Decompiled with CFR 0.152.
 */
package edu.stanford.protege.webprotege.ipc.impl;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.rabbitmq.client.AMQP;
import com.rabbitmq.client.Channel;
import edu.stanford.protege.webprotege.authorization.ActionId;
import edu.stanford.protege.webprotege.authorization.AuthorizationStatus;
import edu.stanford.protege.webprotege.authorization.GetAuthorizationStatusRequest;
import edu.stanford.protege.webprotege.authorization.GetAuthorizationStatusResponse;
import edu.stanford.protege.webprotege.authorization.Resource;
import edu.stanford.protege.webprotege.authorization.Subject;
import edu.stanford.protege.webprotege.common.Request;
import edu.stanford.protege.webprotege.common.Response;
import edu.stanford.protege.webprotege.common.UserId;
import edu.stanford.protege.webprotege.ipc.AuthorizedCommandHandler;
import edu.stanford.protege.webprotege.ipc.CommandExecutionException;
import edu.stanford.protege.webprotege.ipc.CommandExecutor;
import edu.stanford.protege.webprotege.ipc.CommandHandler;
import edu.stanford.protege.webprotege.ipc.ExecutionContext;
import java.io.IOException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeoutException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.rabbit.listener.api.ChannelAwareMessageListener;
import org.springframework.http.HttpStatus;
import reactor.core.publisher.Mono;

public class RabbitMqCommandHandlerWrapper<Q extends Request<R>, R extends Response>
implements ChannelAwareMessageListener {
    private static final Logger logger = LoggerFactory.getLogger(RabbitMqCommandHandlerWrapper.class);
    private final List<CommandHandler<? extends Request, ? extends Response>> handlers;
    private final ObjectMapper objectMapper;
    private final CommandExecutor<GetAuthorizationStatusRequest, GetAuthorizationStatusResponse> authorizationStatusExecutor;

    public RabbitMqCommandHandlerWrapper(List<CommandHandler<? extends Request, ? extends Response>> handlers, ObjectMapper objectMapper, CommandExecutor<GetAuthorizationStatusRequest, GetAuthorizationStatusResponse> authorizationStatusExecutor) {
        this.handlers = handlers;
        this.objectMapper = objectMapper;
        this.authorizationStatusExecutor = authorizationStatusExecutor;
    }

    public void onMessage(Message message, Channel channel) throws Exception {
        logger.info("Received message " + message);
        String replyChannel = message.getMessageProperties().getReplyTo();
        if (replyChannel == null) {
            String errorMessage = "webprotege_replyChannel header is missing.  Cannot reply to message.";
            this.replyWithValidationError(message, channel, errorMessage);
            return;
        }
        String correlationId = message.getMessageProperties().getCorrelationId();
        if (correlationId == null) {
            String errorMessage = "webprotege_correlationId header is missing.  Cannot process message.";
            this.replyWithValidationError(message, channel, errorMessage);
            return;
        }
        String userId = (String)message.getMessageProperties().getHeaders().get("webprotege_userId");
        if (userId == null) {
            String errorMessage = "webprotege_userId header is missing.  Cannot process message.  Returning Forbidden Error Code.  Message reply topic: " + replyChannel;
            this.replyWithValidationError(message, channel, errorMessage);
            return;
        }
        String accessToken = String.valueOf(message.getMessageProperties().getHeaders().get("webprotege_accessToken"));
        if (accessToken == null) {
            String errorMessage = "webprotege_accessToken header is missing.  Cannot process message.  Returning Forbidden Error Code.  Message reply topic: " + replyChannel;
            this.replyWithValidationError(message, channel, errorMessage);
            return;
        }
        String messageType = (String)message.getMessageProperties().getHeaders().get("webprotege_methodName");
        if (messageType == null) {
            String errorMessage = "webprotege_methodName header is missing.  Cannot process message.  Returning Forbidden Error Code.  Message reply topic: " + replyChannel;
            this.replyWithValidationError(message, channel, errorMessage);
            return;
        }
        CommandHandler<Request, Response> handler = this.extractHandler(messageType);
        logger.info("Dispatch handling to {}", handler.getClass());
        this.parseAndHandleRequest(handler, message, channel, new UserId(userId), accessToken);
    }

    private void replyWithValidationError(Message message, Channel channel, String errorMessage) throws IOException, TimeoutException {
        AMQP.BasicProperties replyProps = new AMQP.BasicProperties.Builder().correlationId(message.getMessageProperties().getCorrelationId()).build();
        logger.error(errorMessage);
        channel.basicPublish("", message.getMessageProperties().getReplyTo(), replyProps, errorMessage.getBytes());
    }

    private void parseAndHandleRequest(CommandHandler<Q, R> handler, Message message, Channel channel, UserId userId, String accessToken) {
        try {
            Request request = (Request)this.objectMapper.readValue(message.getBody(), handler.getRequestClass());
            if (handler instanceof AuthorizedCommandHandler) {
                AuthorizedCommandHandler authorizedCommandHandler = (AuthorizedCommandHandler)handler;
                this.authorizeAndReplyToRequest(handler, message, channel, userId, request, authorizedCommandHandler, accessToken);
            } else {
                this.handleAndReplyToRequest(handler, channel, message, userId, request, accessToken);
            }
        }
        catch (IOException e) {
            logger.error("Could not parse request", (Throwable)e);
            this.replyWithErrorResponse(message, channel, userId, HttpStatus.BAD_REQUEST);
        }
    }

    private CommandHandler<? extends Request, ? extends Response> extractHandler(String messageType) {
        return this.handlers.stream().filter(handler -> handler.getChannelName().equalsIgnoreCase(messageType)).findFirst().orElseThrow(() -> new RuntimeException("Invalid message type " + messageType));
    }

    private void authorizeAndReplyToRequest(CommandHandler<Q, R> handler, Message message, Channel channel, UserId userId, Q request, AuthorizedCommandHandler<Q, R> authenticatingCommandHandler, String accessToken) {
        Resource resource = authenticatingCommandHandler.getTargetResource(request);
        Subject subject = Subject.forUser((UserId)userId);
        Collection<ActionId> requiredActionId = authenticatingCommandHandler.getRequiredCapabilities();
        GetAuthorizationStatusRequest authRequest = new GetAuthorizationStatusRequest(resource, subject, (ActionId)requiredActionId.stream().findFirst().orElse(null));
        ExecutionContext executionContext = new ExecutionContext(userId, "");
        CompletableFuture<GetAuthorizationStatusResponse> authResponseFuture = this.authorizationStatusExecutor.execute(authRequest, executionContext);
        authResponseFuture.whenComplete((authResponse, authError) -> {
            if (authError != null) {
                logger.warn("An error occurred when requesting the authorization status for {} on {}. Error: {}", new Object[]{userId, resource, authError.getMessage()});
                this.replyWithErrorResponse(message, channel, userId, HttpStatus.INTERNAL_SERVER_ERROR);
            } else if (authResponse.authorizationStatus() == AuthorizationStatus.AUTHORIZED) {
                this.handleAndReplyToRequest(handler, channel, message, userId, request, accessToken);
            } else {
                logger.info("Permission denied when attempting to execute a request.  User: {}, Request: {}", (Object)userId, (Object)request);
                this.replyWithErrorResponse(message, channel, userId, HttpStatus.FORBIDDEN);
            }
        });
    }

    private void handleAndReplyToRequest(CommandHandler<Q, R> handler, Channel channel, Message message, UserId userId, Q request, String accessToken) {
        ExecutionContext executionContext = new ExecutionContext(userId, accessToken);
        try {
            Mono<R> response = handler.handleRequest(request, executionContext);
            response.subscribe(r -> this.replyWithSuccessResponse(channel, message, userId, r), throwable -> {
                if (throwable instanceof CommandExecutionException) {
                    CommandExecutionException ex = (CommandExecutionException)throwable;
                    logger.info("The command handler threw a CommandExecutionException exception while handling a request.  Sending an error as the reply.  Code: {}, Message: {},  Request: {}", new Object[]{ex.getStatusCode(), throwable.getMessage(), request});
                    this.replyWithErrorResponse(message, channel, userId, ex.getStatus());
                } else {
                    this.replyWithErrorResponse(message, channel, userId, HttpStatus.INTERNAL_SERVER_ERROR);
                }
            });
        }
        catch (Throwable throwable2) {
            logger.error("Uncaught exception when handling request", throwable2);
            this.replyWithErrorResponse(message, channel, userId, HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }

    private void replyWithErrorResponse(Message message, Channel channel, UserId userId, HttpStatus status) {
        try {
            CommandExecutionException executionException = new CommandExecutionException(status);
            String value = this.serializeCommandExecutionException(executionException);
            HashMap<String, String> headersMap = new HashMap<String, String>();
            headersMap.put("webprotege_error", String.valueOf(value));
            headersMap.put("webprotege_userId", String.valueOf(userId.id()));
            AMQP.BasicProperties replyProps = new AMQP.BasicProperties.Builder().correlationId(message.getMessageProperties().getCorrelationId()).headers(headersMap).build();
            channel.basicPublish("webprotege-exchange", message.getMessageProperties().getReplyTo(), replyProps, value.getBytes());
        }
        catch (Exception e) {
            logger.error("Error replyWithErrorResponse ", (Throwable)e);
        }
    }

    private void replyWithSuccessResponse(Channel channel, Message message, UserId userId, R response) {
        try {
            byte[] value = this.objectMapper.writeValueAsBytes(response);
            AMQP.BasicProperties replyProps = new AMQP.BasicProperties.Builder().correlationId(message.getMessageProperties().getCorrelationId()).build();
            channel.basicPublish("webprotege-exchange", message.getMessageProperties().getReplyTo(), replyProps, value);
        }
        catch (Exception e) {
            logger.error("Error handling replyWithSuccessResponse ", (Throwable)e);
            this.replyWithErrorResponse(message, channel, userId, HttpStatus.INTERNAL_SERVER_ERROR);
        }
    }

    private String serializeCommandExecutionException(CommandExecutionException exception) {
        try {
            return this.objectMapper.writeValueAsString((Object)exception);
        }
        catch (JsonProcessingException e) {
            logger.error("Error while serializing CommandExecutionException", (Throwable)e);
            return "{\n    \"statusCode\" : 500\n}\n".strip();
        }
    }
}

