package edu.uci.qa.crypt;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;

import org.slf4j.LoggerFactory;

import joptsimple.OptionParser;
import joptsimple.OptionSet;

public class Main {

  public static void main(String[] args) throws Exception {
    OptionParser parser = new OptionParser() {
      {
        acceptsAll(asList("?", "h", "help"), "Shows the help");
        
        acceptsAll(asList("v", "version"), "Shows the implementation version and default settings");
        
        acceptsAll(asList("verbose"), "Makes logging verbose");
        
        acceptsAll(asList("e", "encrypt"), "Encrypt file (or --input/stdin if no file is specified) "
            + "and write to stdout (or the file specified with --output).").withOptionalArg()
          .ofType(File.class);
        
        acceptsAll(asList("d", "decrypt"), "Decrypt file (or --input/stdin if no file is specified) "
            + "and write to stdout (or the file specified with --output).").withOptionalArg()
          .ofType(File.class);
        
        acceptsAll(asList("i", "input"), "The input string to encrypt/decrypt").withRequiredArg()
          .ofType(File.class);
        
        acceptsAll(asList("o", "output"), "Write output to file.").withRequiredArg()
          .ofType(File.class);
        
        acceptsAll(asList("p", "passphrase"), "passphrase to use for encryption.").withRequiredArg()
          .ofType(String.class);
        
        acceptsAll(asList("options"), "Read options from a file and do not try to read them from the "
            + "default options file(s)").withRequiredArg().ofType(File.class)
          .defaultsTo(new File(".crypt"));
        
        acceptsAll(asList("s", "salt"), "salt to use for encryption").withRequiredArg()
           .ofType(String.class).defaultsTo("salty");
        
        acceptsAll(asList("cipher-algo"), "Algorithm for the cipher").withRequiredArg()
          .ofType(String.class).defaultsTo("AES/CBC/PKCS5Padding");
        
        acceptsAll(asList("spec-algo"), "Algorithm for the secret key spec").withRequiredArg()
          .ofType(String.class).defaultsTo("AES");
        
        acceptsAll(asList("factory-algo"), "Algorithm for the secret key factory").withRequiredArg()
          .ofType(String.class).defaultsTo("PBKDF2WithHmacSHA512");
        
        acceptsAll(asList("iterations"), "Number of iterations for the PBE. More iterations the more "
            + "secure").withRequiredArg().ofType(Integer.class).defaultsTo(40000);
        
        acceptsAll(asList("key-length")).withRequiredArg().ofType(Integer.class).defaultsTo(128);
      }
    };
    
    OptionSet options = null;
    
    try {
      options = parser.parse(args);
    } catch (joptsimple.OptionException e) {
      LoggerFactory.getLogger(Main.class.getName()).error(e.getLocalizedMessage());
    }
    
    if(options.has("?") || options.nonOptionArguments().size() > 1) {
      try {
        parser.printHelpOn(System.out);
      } catch (IOException e) {
        LoggerFactory.getLogger(Main.class.getName()).error(null, e);
      }
      return;
    }
    
    CryptConfig config = new CryptConfig((File) options.valueOf("options"));
    if (options.has("passphrase"))
      config.passphrase = (String) options.valueOf("passphrase");
    
    if (options.has("salt") || config.salt == null)
      config.salt = (String) options.valueOf("salt");
    if (options.has("cipher-algo") || config.cipherAlgo == null)
      config.cipherAlgo = (String) options.valueOf("cipher-algo");
    if (options.has("spec-algo") || config.specAlgo == null)
      config.specAlgo = (String) options.valueOf("spec-algo");
    if (options.has("factory-algo") || config.factoryAlgo == null)
      config.factoryAlgo = (String) options.valueOf("factory-algo");
    if (options.has("iterations") || config.iterations == null)
      config.iterations = (Integer) options.valueOf("iterations");
    if (options.has("key-length") || config.keyLength == null)
      config.keyLength = (Integer) options.valueOf("key-length");
    
    if (options.has("v")) {
      System.out.println(Main.class.getPackage().getImplementationVersion());
      System.out.println("Pass: " + config.passphrase);
      System.out.println("Salt: " + config.salt);
      System.out.println("Cipher: " + config.cipherAlgo);
      System.out.println("factory: " + config.factoryAlgo);
      System.out.println("Spec: " + config.specAlgo);
      System.out.println("Iterations:" + config.iterations);
      System.out.println("Key Length: " + config.keyLength);
      return;
    }
    
    if (config.passphrase == null) {
      System.out.println("No passphrase was specified either by file or command line!");
      return;
    }
    
    if (options.has("e") && options.has("d")) {
      System.out.println("Cannot both encrypt and decrypt in the same call!");
      return;
    }
    
    String inputStr = null;
    if (options.nonOptionArguments().size() == 1) {
      inputStr = (String) options.nonOptionArguments().get(0);
      if (options.has("input")) {
        inputStr = (String) options.valueOf("input");
      }
    }
    InputStream input = null;
    if (options.hasArgument("e") || options.hasArgument("d")) {
      File fileIn = (File) options.valueOf("e");
      if (fileIn == null) {
        fileIn = (File) options.valueOf("d");
      }
      input = new FileInputStream(fileIn);
    }
    
    if (input == null) {
      byte[] data = null;
      if (inputStr != null) {
        if (options.has("d")) {
          data = Base64.getDecoder().decode(inputStr);
        } else {
          data = inputStr.getBytes("UTF-8");
        }
      }
      if (data != null) {
        input = new ByteArrayInputStream(data);
      }
    }
    if (input == null) {
      try {
        parser.printHelpOn(System.out);
      } catch (IOException e) {
        LoggerFactory.getLogger(Main.class.getName()).error(null, e);
      }
      return;
    }
    ByteArrayOutputStream output = new ByteArrayOutputStream();
    if (options.has("d")) {
      AES.decrypt(config.passphrase.toCharArray(), input, output);
    } else {
      AES.encrypt(config.keyLength, config.passphrase.toCharArray(), input, output);
    }
    
    if (options.has("output")) {
      FileOutputStream fos = new FileOutputStream((File)options.valueOf("output"));
      output.writeTo(fos);
      fos.close();
    } else {
      if (options.has("d")) {
        System.out.println(new String(output.toByteArray()));
      } else {
        System.out.println(Base64.getEncoder().encodeToString(output.toByteArray()));
      }
    }
  }
  
  private static List<String> asList(String... params) {
    return Arrays.asList(params);
  }
}
