package edu.uiuc.ncsa.myproxy.oa4mp.client;

import edu.uiuc.ncsa.myproxy.oa4mp.client.storage.AssetProvider;
import edu.uiuc.ncsa.security.core.Identifier;
import edu.uiuc.ncsa.security.core.exceptions.GeneralException;
import edu.uiuc.ncsa.security.delegation.client.request.DelegatedAssetRequest;
import edu.uiuc.ncsa.security.delegation.client.request.DelegatedAssetResponse;
import edu.uiuc.ncsa.security.delegation.client.request.DelegationRequest;
import edu.uiuc.ncsa.security.delegation.client.request.DelegationResponse;
import edu.uiuc.ncsa.security.delegation.token.AuthorizationGrant;
import edu.uiuc.ncsa.security.delegation.token.MyX509Certificates;
import edu.uiuc.ncsa.security.delegation.token.Verifier;
import org.apache.commons.codec.binary.Base64;
import org.bouncycastle.jce.PKCS10CertificationRequest;

import java.net.URI;
import java.security.KeyPair;
import java.util.HashMap;
import java.util.Map;

import static edu.uiuc.ncsa.myproxy.oa4mp.client.ClientEnvironment.CALLBACK_URI_KEY;
import static edu.uiuc.ncsa.security.util.pkcs.CertUtil.createCertRequest;
import static edu.uiuc.ncsa.security.util.pkcs.KeyUtil.generateKeyPair;

/**
 * The OAuth for MyProxy service. Note that the {@link ClientEnvironment} is queried for its properties
 * for each call, so that changes on a per request basis will be performed.
 * <p>Created by Jeff Gaynor<br>
 * on May 16, 2011 at  3:27:16 PM
 */
public class OA4MPService {

    public static final String SKIN_PARAMETER = "skin";

    /**
     * Basic constructor for this service.
     *
     * @param environment
     */
    public OA4MPService(ClientEnvironment environment) {
        this.environment = environment;
    }

    public ClientEnvironment getEnvironment() {
        return environment;
    }

    ClientEnvironment environment;


    /**
     * Request a certificate from the user portal. This will also generate the private key and cert request. These
     * are not stored by this service. The additionalParameters argument are passed as key/value pairs
     * in the initial request
     * and are not otherwise processed.
     *
     * @return
     */
    public OA4MPResponse requestCert(Map additionalParameters) {
        if (additionalParameters == null) {
            additionalParameters = new HashMap();
        }
        try {
            KeyPair keyPair = generateKeyPair();
            PKCS10CertificationRequest certReq = createCertRequest(keyPair);
            OA4MPResponse mpdsResponse = new OA4MPResponse();
            mpdsResponse.setPrivateKey(keyPair.getPrivate());
            additionalParameters.put(ClientEnvironment.CERT_REQUEST_KEY, Base64.encodeBase64String(certReq.getEncoded()));
            if (!additionalParameters.containsKey(getEnvironment().getConstants().get(CALLBACK_URI_KEY))) {
                additionalParameters.put(getEnvironment().getConstants().get(CALLBACK_URI_KEY), getEnvironment().getCallback().toString());
            }
            DelegationRequest daReq = new DelegationRequest();
            daReq.setParameters(additionalParameters);
            daReq.setClient(getEnvironment().getClient());
            daReq.setBaseUri(getEnvironment().getAuthorizationUri());
            DelegationResponse daResp = (DelegationResponse) getEnvironment().getDelegationService().process(daReq);
            String skin = getEnvironment().getSkin();
            String r = daResp.getRedirectUri().toString();
            if (skin != null) {
                r = r + "&" + SKIN_PARAMETER + "=" + skin;
            }
            mpdsResponse.setRedirect(URI.create(r));
            return mpdsResponse;
        } catch (Throwable e) {
            e.printStackTrace();
            if (e instanceof RuntimeException) {
                throw (RuntimeException) e;
            }
            throw new GeneralException("Error generating request", e);
        }

    }


    /**
     * This will make the request with whatever defaults are in effect for the client. You can override these
     * by supplying them as key-value pairs in the {@link #requestCert(java.util.Map)} call.
     *
     * @return
     */
    public OA4MPResponse requestCert() {
        Map m = new HashMap();
        if (0 <= getEnvironment().getCertLifetime()) {
            m.put(ClientEnvironment.CERT_LIFETIME_KEY, getEnvironment().getCertLifetime());
        }
        return requestCert(m);
    }

    /**
     * A convenience method to do the {@link #requestCert()} call and create an asset with the given identifier. This
     * will throw an exception if there is no asset store configured.
     *
     * @param identifier
     * @return
     */
    public OA4MPResponse requestCert(Identifier identifier) {
        Map m = new HashMap();
        if (0 <= getEnvironment().getCertLifetime()) {
            m.put(ClientEnvironment.CERT_LIFETIME_KEY, getEnvironment().getCertLifetime());
        }
        return requestCert(identifier, m);
    }


    /**
     * A convenience method that allows for a map of additional parameters.
     * @param identifier
     * @param additionalParameters
     * @return
     */
    public OA4MPResponse requestCert(Identifier identifier, Map additionalParameters) {
        OA4MPResponse response = requestCert(additionalParameters);
        if (!getEnvironment().hasAssetStore()) {
            throw new IllegalStateException("Error: There is no asset store configured.");
        }
        AssetProvider assetProvider = new AssetProvider();
        Asset asset = assetProvider.get(identifier);
        asset.setPrivateKey(response.getPrivateKey());
        asset.setRedirect(response.getRedirect());
        getEnvironment().getAssetStore().save(asset);
        return response;
    }

    /**
     * Retrieve the certificate chain from the server. This is done after the {@link #requestCert()} and
     * user authorization.
     *
     * @param tempToken
     * @param verifier
     * @return
     */
    public AssetResponse getCert(String tempToken, String verifier) {
        AuthorizationGrant ag = getEnvironment().getTokenForge().getAuthorizationGrant(tempToken);
        Verifier v = getEnvironment().getTokenForge().getVerifier(verifier);
        DelegatedAssetRequest dar = new DelegatedAssetRequest();
        Map m = new HashMap();
        dar.setAuthorizationGrant(ag);
        dar.setClient(getEnvironment().getClient());
        dar.setVerifier(v);
        dar.setParameters(m);
        Map m1 = new HashMap();

        dar.setAssetParameters(m1);
        DelegatedAssetResponse daResp = (DelegatedAssetResponse) getEnvironment().getDelegationService().process(dar);

        AssetResponse par = new AssetResponse();
        MyX509Certificates myX509Certificate = (MyX509Certificates) daResp.getProtectedAsset();
        par.setX509Certificates(myX509Certificate.getX509Certificates());
        par.setUsername(daResp.getAdditionalInformation().get("username"));
        return par;
    }

    /**
     * Performs the {@link #getCert(String, String)} call then updates the asset associated with
     * the given identifier. This throws an exception is there is no asset or if the asset store
     * is not enabled.
     *
     * @param tempToken
     * @param verifier
     * @param identifier
     * @return
     */
    public AssetResponse getCert(String tempToken, String verifier, Identifier identifier) {
        AssetResponse assetResponse = getCert(tempToken, verifier);

        if (!getEnvironment().hasAssetStore()) {
            throw new IllegalStateException("Error: No asset store enabled.");
        }
        if (identifier == null) {
            throw new IllegalArgumentException("Error: null identifier encountered");
        }

        Asset asset = getEnvironment().getAssetStore().get(identifier);
        if (asset == null) {
            throw new IllegalArgumentException("Error:No asset with the given identifier \"" + identifier + "\" found.");
        }
        asset.setUsername(assetResponse.getUsername());
        asset.setCertificates(assetResponse.getX509Certificates());
        getEnvironment().getAssetStore().save(asset);
        return assetResponse;
    }

}
