/*
  $Id: PasswordValidator.java 1724 2010-10-29 17:47:59Z dfisher $

  Copyright (C) 2003-2010 Virginia Tech.
  All rights reserved.

  SEE LICENSE FOR MORE INFORMATION

  Author:  Middleware Services
  Email:   middleware@vt.edu
  Version: $Revision: 1724 $
  Updated: $Date: 2010-10-29 13:47:59 -0400 (Fri, 29 Oct 2010) $
*/
package edu.vt.middleware.password;

import java.io.RandomAccessFile;
import edu.vt.middleware.dictionary.FileWordList;
import edu.vt.middleware.dictionary.TernaryTreeDictionary;

/**
 * <code>PasswordValidator</code> provides methods to running rule validation
 * against password data.
 *
 * @author  Middleware Services
 * @version  $Revision: 1724 $ $Date: 2010-10-29 13:47:59 -0400 (Fri, 29 Oct 2010) $
 */

public final class PasswordValidator
{


  /** Default constructor. */
  private PasswordValidator() {}


  /**
   * Validates the supplied password data against the supplied rule.
   *
   * @param  rule  <code>Rule</code> to validate password with
   * @param  passwordData  <code>PasswordData</code> to validate
   *
   * @return  <code>RuleResult</code>
   */
  public static RuleResult validate(
    final Rule rule,
    final PasswordData passwordData)
  {
    return rule.validate(passwordData);
  }


  /**
   * This provides command line access to an <code>PasswordValidator</code>.
   *
   * @param  args  <code>String[]</code>
   *
   * @throws  Exception  if an error occurs
   */
  public static void main(final String[] args)
    throws Exception
  {
    final RuleList ruleList = new RuleList();
    String username = null;
    String password = null;
    try {
      if (args.length < 2) {
        throw new ArrayIndexOutOfBoundsException();
      }
      for (int i = 0; i < args.length; i++) {
        if ("-l".equals(args[i])) {
          final int min = Integer.parseInt(args[++i]);
          final int max = Integer.parseInt(args[++i]);
          final LengthRule rule = new LengthRule(min, max);
          ruleList.getRules().add(rule);
        } else if ("-c".equals(args[i])) {
          final CharacterCharacteristicsRule rule =
            new CharacterCharacteristicsRule();
          rule.getRules().add(
            new DigitCharacterRule(Integer.parseInt(args[++i])));
          rule.getRules().add(
            new AlphabeticalCharacterRule(Integer.parseInt(args[++i])));
          rule.getRules().add(
            new NonAlphanumericCharacterRule(Integer.parseInt(args[++i])));
          rule.getRules().add(
            new UppercaseCharacterRule(Integer.parseInt(args[++i])));
          rule.getRules().add(
            new LowercaseCharacterRule(Integer.parseInt(args[++i])));
          rule.setNumberOfCharacteristics(Integer.parseInt(args[++i]));
          ruleList.getRules().add(rule);
        } else if ("-d".equals(args[i])) {
          final TernaryTreeDictionary dict = new TernaryTreeDictionary(
            new FileWordList(new RandomAccessFile(args[++i], "r"), false));
          final DictionarySubstringRule rule = new DictionarySubstringRule(
            dict);
          rule.setMatchBackwards(true);
          rule.setWordLength(Integer.parseInt(args[++i]));
          ruleList.getRules().add(rule);
        } else if ("-u".equals(args[i])) {
          ruleList.getRules().add(new UsernameRule(true, true));
          username = args[++i];
        } else if ("-k".equals(args[i])) {
          ruleList.getRules().add(new SequenceRule(true, true));
        } else if ("-h".equals(args[i])) {
          throw new ArrayIndexOutOfBoundsException();
        } else {
          password = args[i];
        }
      }

      if (password == null) {
        throw new ArrayIndexOutOfBoundsException();
      } else {
        RuleResult result = null;
        final PasswordData pd = new PasswordData(new Password(password));
        if (username == null) {
          result = PasswordValidator.validate(ruleList, pd);
        } else {
          pd.setUsername(username);
          result = PasswordValidator.validate(ruleList, pd);
        }
        if (result.isValid()) {
          System.out.println("Valid password");
        } else {
          for (RuleResultDetail rrd : result.getDetails()) {
            System.out.println(rrd.getMessage());
          }
        }
      }

    } catch (ArrayIndexOutOfBoundsException e) {
      System.out.println(
        "Usage: java " + PasswordValidator.class.getName() + " <password> \\");
      System.out.println("       -l (Set the min & max password length) \\");
      System.out.println("          <min> \\");
      System.out.println("          <max> \\");
      System.out.println(
        "       -c (Set the characters which must be present" +
        " in the password) \\");
      System.out.println("          (Each of the following must be >= 0) \\");
      System.out.println("          <digits> \\");
      System.out.println("          <alphabetical> \\");
      System.out.println("          <non-alphanumeric> \\");
      System.out.println("          <uppercase> \\");
      System.out.println("          <lowercase> \\");
      System.out.println(
        "          <num> (Number of these rules to" +
        " enforce) \\");
      System.out.println("       -d (Test password against a dictionary) \\");
      System.out.println("          <file> (dictionary files) \\");
      System.out.println(
        "          <num> (number of characters in matching" +
        " words) \\");
      System.out.println("       -u (Test for a user id) \\");
      System.out.println("          <userid> \\");
      System.out.println("       -k (Test for keyboard sequences) \\");
      System.out.println("       -h (Print this message) \\");
      System.exit(1);
    }
  }
}
