/*
 * Decompiled with CFR 0.152.
 */
package ee.sk.mid;

import ee.sk.mid.MidAuthentication;
import ee.sk.mid.MidAuthenticationError;
import ee.sk.mid.MidAuthenticationIdentity;
import ee.sk.mid.MidAuthenticationResult;
import ee.sk.mid.MidClient;
import ee.sk.mid.MidSignatureVerifier;
import ee.sk.mid.exception.MidInternalErrorException;
import ee.sk.mid.exception.MidMissingOrInvalidParameterException;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.PublicKey;
import java.security.SignatureException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;
import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MidAuthenticationResponseValidator {
    private static final Logger logger = LoggerFactory.getLogger(MidAuthenticationResponseValidator.class);
    private List<X509Certificate> trustedCACertificates;

    public MidAuthenticationResponseValidator(MidClient client) {
        KeyStore trustStore = client.getTrustStore();
        if (trustStore == null) {
            throw new MidMissingOrInvalidParameterException("You need to add a trust store to client ");
        }
        this.initializeTrustedCACertificatesFromTrustStore(trustStore);
    }

    public MidAuthenticationResponseValidator(KeyStore trustStore) {
        if (trustStore == null) {
            throw new MidMissingOrInvalidParameterException("trustStore cannot be null");
        }
        this.initializeTrustedCACertificatesFromTrustStore(trustStore);
    }

    public MidAuthenticationResponseValidator(List<X509Certificate> trustedCACertificates) {
        this.trustedCACertificates = trustedCACertificates;
    }

    public MidAuthenticationResult validate(MidAuthentication authentication) {
        this.validateAuthentication(authentication);
        MidAuthenticationResult authenticationResult = new MidAuthenticationResult();
        MidAuthenticationIdentity identity = this.constructAuthenticationIdentity(authentication.getCertificate());
        authenticationResult.setAuthenticationIdentity(identity);
        if (!this.isResultOk(authentication)) {
            authenticationResult.setValid(false);
            authenticationResult.addError(MidAuthenticationError.INVALID_RESULT);
        }
        if (!this.isSignatureValid(authentication)) {
            authenticationResult.setValid(false);
            authenticationResult.addError(MidAuthenticationError.SIGNATURE_VERIFICATION_FAILURE);
        }
        if (!this.isCertificateValid(authentication.getCertificate())) {
            authenticationResult.setValid(false);
            authenticationResult.addError(MidAuthenticationError.CERTIFICATE_EXPIRED);
        }
        if (!this.isCertificateTrusted(authentication.getCertificate())) {
            authenticationResult.setValid(false);
            authenticationResult.addError(MidAuthenticationError.CERTIFICATE_NOT_TRUSTED);
        }
        return authenticationResult;
    }

    private void validateAuthentication(MidAuthentication authentication) {
        if (authentication.getCertificate() == null) {
            logger.error("Certificate is not present in the authentication response");
            throw new MidInternalErrorException("Certificate is not present in the authentication response");
        }
        if (authentication.getSignatureValueInBase64().isEmpty()) {
            logger.error("Signature is not present in the authentication response");
            throw new MidInternalErrorException("Signature is not present in the authentication response");
        }
        if (authentication.getHashType() == null) {
            logger.error("Hash type is not present in the authentication response");
            throw new MidInternalErrorException("Hash type is not present in the authentication response");
        }
    }

    MidAuthenticationIdentity constructAuthenticationIdentity(X509Certificate certificate) {
        MidAuthenticationIdentity identity = new MidAuthenticationIdentity();
        try {
            LdapName ln = new LdapName(certificate.getSubjectDN().getName());
            for (Rdn rdn : ln.getRdns()) {
                String type;
                switch (type = rdn.getType().toUpperCase()) {
                    case "GIVENNAME": {
                        identity.setGivenName(rdn.getValue().toString());
                        break;
                    }
                    case "SURNAME": {
                        identity.setSurName(rdn.getValue().toString());
                        break;
                    }
                    case "SERIALNUMBER": {
                        identity.setIdentityCode(this.getIdentityNumber(rdn.getValue().toString()));
                        break;
                    }
                    case "C": {
                        identity.setCountry(rdn.getValue().toString());
                        break;
                    }
                }
            }
            return identity;
        }
        catch (InvalidNameException e) {
            logger.error("Error getting authentication identity from the certificate", (Throwable)e);
            throw new MidInternalErrorException("Error getting authentication identity from the certificate", e);
        }
    }

    private String getIdentityNumber(String serialNumber) {
        return serialNumber.replaceAll("^PNO[A-Z][A-Z]-", "");
    }

    private boolean isResultOk(MidAuthentication authentication) {
        return "OK".equalsIgnoreCase(authentication.getResult());
    }

    private boolean isSignatureValid(MidAuthentication authentication) {
        PublicKey publicKey = authentication.getCertificate().getPublicKey();
        switch (publicKey.getAlgorithm()) {
            case "RSA": {
                return MidSignatureVerifier.verifyWithRSA(publicKey, authentication);
            }
            case "EC": {
                return MidSignatureVerifier.verifyWithECDSA(publicKey, authentication);
            }
        }
        throw new IllegalArgumentException("Unsupported algorithm " + publicKey.getAlgorithm());
    }

    private void initializeTrustedCACertificatesFromTrustStore(KeyStore trustStore) {
        ArrayList<X509Certificate> certificatesFromTrustStore = new ArrayList<X509Certificate>();
        try {
            Enumeration<String> aliases = trustStore.aliases();
            while (aliases.hasMoreElements()) {
                String alias = aliases.nextElement();
                X509Certificate certificate = (X509Certificate)trustStore.getCertificate(alias);
                logger.debug("adding trusted ca certificate: {}", (Object)certificate);
                certificatesFromTrustStore.add(certificate);
            }
        }
        catch (KeyStoreException e) {
            logger.error("Error initializing trusted CA certificates", (Throwable)e);
            throw new MidInternalErrorException("Error initializing trusted CA certificates", e);
        }
        this.trustedCACertificates = certificatesFromTrustStore;
    }

    private boolean isCertificateValid(X509Certificate certificate) {
        return !certificate.getNotAfter().before(new Date());
    }

    private boolean isCertificateTrusted(X509Certificate certificate) {
        for (X509Certificate trustedCACertificate : this.trustedCACertificates) {
            try {
                certificate.verify(trustedCACertificate.getPublicKey());
                return true;
            }
            catch (SignatureException e) {
            }
            catch (GeneralSecurityException e) {
                logger.warn("Error verifying signer's certificate: " + certificate.getSubjectDN() + " against CA certificate: " + trustedCACertificate.getSubjectDN(), (Throwable)e);
            }
        }
        return false;
    }
}

