/*
 * Decompiled with CFR 0.152.
 */
package ee.sk.smartid;

import ee.sk.smartid.AuthenticationIdentity;
import ee.sk.smartid.CertificateLevel;
import ee.sk.smartid.SmartIdAuthenticationResponse;
import ee.sk.smartid.SmartIdAuthenticationResult;
import ee.sk.smartid.exception.TechnicalErrorException;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.List;
import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AuthenticationResponseValidator {
    private static final Logger logger = LoggerFactory.getLogger(AuthenticationResponseValidator.class);
    private List<X509Certificate> trustedCACertificates = new ArrayList<X509Certificate>();

    public AuthenticationResponseValidator() {
        this.initializeTrustedCACertificatesFromKeyStore();
    }

    public SmartIdAuthenticationResult validate(SmartIdAuthenticationResponse authenticationResponse) {
        this.validateAuthenticationResponse(authenticationResponse);
        SmartIdAuthenticationResult authenticationResult = new SmartIdAuthenticationResult();
        AuthenticationIdentity identity = this.constructAuthenticationIdentity(authenticationResponse.getCertificate());
        authenticationResult.setAuthenticationIdentity(identity);
        if (!this.verifyResponseEndResult(authenticationResponse)) {
            authenticationResult.setValid(false);
            authenticationResult.addError(SmartIdAuthenticationResult.Error.INVALID_END_RESULT);
        }
        if (!this.verifySignature(authenticationResponse)) {
            authenticationResult.setValid(false);
            authenticationResult.addError(SmartIdAuthenticationResult.Error.SIGNATURE_VERIFICATION_FAILURE);
        }
        if (!this.verifyCertificateExpiry(authenticationResponse.getCertificate())) {
            authenticationResult.setValid(false);
            authenticationResult.addError(SmartIdAuthenticationResult.Error.CERTIFICATE_EXPIRED);
        }
        if (!this.isCertificateTrusted(authenticationResponse.getCertificate())) {
            authenticationResult.setValid(false);
            authenticationResult.addError(SmartIdAuthenticationResult.Error.CERTIFICATE_NOT_TRUSTED);
        }
        if (!this.verifyCertificateLevel(authenticationResponse)) {
            authenticationResult.setValid(false);
            authenticationResult.addError(SmartIdAuthenticationResult.Error.CERTIFICATE_LEVEL_MISMATCH);
        }
        return authenticationResult;
    }

    public List<X509Certificate> getTrustedCACertificates() {
        return this.trustedCACertificates;
    }

    public void addTrustedCACertificate(X509Certificate certificate) {
        this.trustedCACertificates.add(certificate);
    }

    public void addTrustedCACertificate(byte[] certificateBytes) throws CertificateException {
        CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
        X509Certificate caCertificate = (X509Certificate)certFactory.generateCertificate(new ByteArrayInputStream(certificateBytes));
        this.addTrustedCACertificate(caCertificate);
    }

    public void addTrustedCACertificate(File certificateFile) throws IOException, CertificateException {
        this.addTrustedCACertificate(Files.readAllBytes(certificateFile.toPath()));
    }

    public void clearTrustedCACertificates() {
        this.trustedCACertificates.clear();
    }

    private void initializeTrustedCACertificatesFromKeyStore() {
        try (InputStream is = AuthenticationResponseValidator.class.getResourceAsStream("/trusted_certificates.jks");){
            KeyStore keystore = KeyStore.getInstance(KeyStore.getDefaultType());
            keystore.load(is, "changeit".toCharArray());
            Enumeration<String> aliases = keystore.aliases();
            while (aliases.hasMoreElements()) {
                String alias = aliases.nextElement();
                X509Certificate certificate = (X509Certificate)keystore.getCertificate(alias);
                this.addTrustedCACertificate(certificate);
            }
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            logger.error("Error initializing trusted CA certificates", (Throwable)e);
            throw new TechnicalErrorException("Error initializing trusted CA certificates", e);
        }
    }

    private void validateAuthenticationResponse(SmartIdAuthenticationResponse authenticationResponse) {
        if (authenticationResponse.getCertificate() == null) {
            logger.error("Certificate is not present in the authentication response");
            throw new TechnicalErrorException("Certificate is not present in the authentication response");
        }
        if (StringUtils.isEmpty((CharSequence)authenticationResponse.getSignatureValueInBase64())) {
            logger.error("Signature is not present in the authentication response");
            throw new TechnicalErrorException("Signature is not present in the authentication response");
        }
        if (authenticationResponse.getHashType() == null) {
            logger.error("Hash type is not present in the authentication response");
            throw new TechnicalErrorException("Hash type is not present in the authentication response");
        }
    }

    private boolean verifyResponseEndResult(SmartIdAuthenticationResponse authenticationResponse) {
        return "OK".equalsIgnoreCase(authenticationResponse.getEndResult());
    }

    private boolean verifySignature(SmartIdAuthenticationResponse authenticationResponse) {
        try {
            PublicKey signersPublicKey = authenticationResponse.getCertificate().getPublicKey();
            Signature signature = Signature.getInstance("NONEwith" + signersPublicKey.getAlgorithm());
            signature.initVerify(signersPublicKey);
            byte[] signedHash = Base64.decodeBase64((String)authenticationResponse.getSignedHashInBase64());
            byte[] signedDigestWithPadding = AuthenticationResponseValidator.addPadding(authenticationResponse.getHashType().getDigestInfoPrefix(), signedHash);
            signature.update(signedDigestWithPadding);
            return signature.verify(authenticationResponse.getSignatureValue());
        }
        catch (GeneralSecurityException e) {
            logger.error("Signature verification failed");
            throw new TechnicalErrorException("Signature verification failed", e);
        }
    }

    private boolean verifyCertificateExpiry(X509Certificate certificate) {
        return !certificate.getNotAfter().before(new Date());
    }

    private boolean isCertificateTrusted(X509Certificate certificate) {
        for (X509Certificate trustedCACertificate : this.trustedCACertificates) {
            try {
                certificate.verify(trustedCACertificate.getPublicKey());
                return true;
            }
            catch (SignatureException e) {
            }
            catch (GeneralSecurityException e) {
                logger.warn("Error verifying signer's certificate: " + certificate.getSubjectDN() + " against CA certificate: " + trustedCACertificate.getSubjectDN(), (Throwable)e);
            }
        }
        return false;
    }

    private boolean verifyCertificateLevel(SmartIdAuthenticationResponse authenticationResponse) {
        CertificateLevel certLevel = new CertificateLevel(authenticationResponse.getCertificateLevel());
        String requestedCertificateLevel = authenticationResponse.getRequestedCertificateLevel();
        return StringUtils.isEmpty((CharSequence)requestedCertificateLevel) || certLevel.isEqualOrAbove(requestedCertificateLevel);
    }

    private static byte[] addPadding(byte[] digestInfoPrefix, byte[] digest) {
        return ArrayUtils.addAll((byte[])digestInfoPrefix, (byte[])digest);
    }

    AuthenticationIdentity constructAuthenticationIdentity(X509Certificate certificate) {
        AuthenticationIdentity identity = new AuthenticationIdentity();
        try {
            LdapName ln = new LdapName(certificate.getSubjectDN().getName());
            for (Rdn rdn : ln.getRdns()) {
                if (rdn.getType().equalsIgnoreCase("GIVENNAME")) {
                    identity.setGivenName(rdn.getValue().toString());
                    continue;
                }
                if (rdn.getType().equalsIgnoreCase("SURNAME")) {
                    identity.setSurName(rdn.getValue().toString());
                    continue;
                }
                if (rdn.getType().equalsIgnoreCase("SERIALNUMBER")) {
                    identity.setIdentityCode(rdn.getValue().toString().split("-", 2)[1]);
                    continue;
                }
                if (!rdn.getType().equalsIgnoreCase("C")) continue;
                identity.setCountry(rdn.getValue().toString());
            }
            return identity;
        }
        catch (InvalidNameException e) {
            logger.error("Error getting authentication identity from the certificate", (Throwable)e);
            throw new TechnicalErrorException("Error getting authentication identity from the certificate", e);
        }
    }
}

