/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma;

import org.dom4j.DocumentException;
import org.dom4j.io.SAXReader;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;

import com.vladium.emma.util.DecorateUtil;
import com.vladium.emma.util.EmmaRunner;
import com.vladium.emma.util.ReportUtil;

/**
 * Used to overcome that the Emma coverage file is output by another process and
 * therefore does not display in the workbench. Refreshes the file on a regular basis.
 * 
 * @author Erik Beijnoff erik@beijnoff.com
 * @since 2005-11-03
 */
public class CoverageRefreshJob extends Job{
	//The project to check
	private IProject project= null;

	//Indicates when the coverage file was modifed the last time
	private long coverageModified= IResource.NULL_STAMP;

	//Indicates when the report file was modifed the last time
	private long reportModified= IResource.NULL_STAMP;

	//Count to prevent job from scheduling itself infintely
	private int tryCount= 0;
	private static final int TRY_LIMIT= 20;
	
	/**
	 * Constructor
	 * 
	 * @param name the name of the thread
	 * @param project the project that the job is checking coverage for
	 */
	public CoverageRefreshJob(String name, IProject project){
		super(name);
		this.project= project;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.core.internal.jobs.InternalJob#run(org.eclipse.core.runtime.IProgressMonitor)
	 */
	protected IStatus run(IProgressMonitor monitor){
		try{
			//Refresh coverage files for all projects
			if(project.isOpen()){
				boolean emmaEnabled= Properties.BUILDER_ENABLED.getPersistentAsBoolean(project);
				
				if(emmaEnabled){
					String coverageFilePath= Properties.INTERNAL_COVERAGE_FILE.getPersistent(project);
					
					final IFile coverageFile= project.getFile(coverageFilePath);
					coverageFile.refreshLocal(IResource.DEPTH_INFINITE, null);
					
					boolean redecorationDone= false;
					if(coverageFile.exists()){
						long newCoverageModified= coverageFile.getModificationStamp();
						if(newCoverageModified != coverageModified){
							coverageModified= newCoverageModified;

							final IJavaProject javaProject= JavaCore.create(project);
							final IPath xmlReportPath= (IPath)Properties.INTERNAL_XML_REPORT.getDefault();

							ReportUtil.buildReport(javaProject, EmmaRunner.XML, false, xmlReportPath);

							final IFile outFile= project.getFile(xmlReportPath);
							if(outFile.exists()){
								long newReportModified= outFile.getModificationStamp();
								if(newReportModified != reportModified){
									reportModified= newReportModified;
									tryCount= 0; 
									redecorationDone= true;

									DecorateUtil.redecorateCoverage(project, new SAXReader().read(outFile.getContents()));
								}
							}
						}
					}
					
					if(!redecorationDone){
						reschedule();
					}
				}
			}

			return Status.OK_STATUS;
		}catch(CoreException e){
			return new Status(IStatus.ERROR, EmmaPlugin.ID, 0, e.getMessage(), e);
		}catch(DocumentException e){
			return new Status(IStatus.ERROR, EmmaPlugin.ID, 0, e.getMessage(), e);
		}catch(IllegalStateException e){
			//May be thrown on shutdown, ignore.
			return Status.CANCEL_STATUS;
		}finally{
			monitor.done();
		}
	}
	
	/* 
	 * (non-Javadoc)
	 * Schedule to run a short time later if coverage file and report hasn't been
	 * generated yet. It seems as if this job sometimes runs too fast to catch the generated file.
	 */
	private void reschedule(){
		if(tryCount < TRY_LIMIT){
			tryCount++;
			
			schedule(200);
		}else{
			tryCount= 0;
		}
	}
}