/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchListener;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.jdt.launching.IVMInstall;
import org.eclipse.jdt.launching.IVMInstallType;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.ui.IStartup;

/**
 * The startup class added as early startup in plugin.xml
 */
public class CoverageRefreshManager implements IStartup{
	//Housekeeping threads to make sure the coverage.ec file is up to date
	protected final Map coverageRefreshJobs= new HashMap();

	//The shared instance.
	private static CoverageRefreshManager instance;

	/**
	 * The constructor.
	 */
	public CoverageRefreshManager(){
		super();
		instance= this;
	}

	/**
	 * Returns the shared instance.
	 */
	public static CoverageRefreshManager getInstance(){
		return instance;
	}

	/**
	 * Enables or disables the coverage refresh job that continually keeps checking the 
	 * coverage.ec file to see if it has been updated.
	 * 
	 * TODO - A closed project does not get any job added to it. Enablemenet needs to 
	 * 			be done when the project is reopened
	 */
	public void setCoverageJobStatus(IProject project){
		if(project.isOpen()){
			try{
				final boolean enabled= Properties.BUILDER_ENABLED.getPersistentAsBoolean(project);

				final String projectName= project.getName();
				Job job= (Job)coverageRefreshJobs.get(projectName);
				if(job == null){
					job= new CoverageRefreshJob("Internal coverage job for project " + projectName, project);
					job.setSystem(true);

					coverageRefreshJobs.put(projectName, job);
				}

				if(enabled && project.isOpen()){
					job.schedule();
				}
			}catch(CoreException e){
				EmmaPlugin.log(e.getStatus());
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.IStartup#earlyStartup()
	 */
	public void earlyStartup(){
		IWorkspace workspace= ResourcesPlugin.getWorkspace();
		final IProject[] projects= workspace.getRoot().getProjects();
		for(int i= 0; i < projects.length; i++){
			final IProject project= projects[i];

			setCoverageJobStatus(project);
		}


		final String covName= Properties.INTERNAL_COVERAGE_JVM_ARG.getDefaultAsString();
		final String covValue= Properties.INTERNAL_COVERAGE_FILE.getDefaultAsString();

		final String verbName= Properties.INTERNAL_VERBOSITY_JVM_ARG.getDefaultAsString();
		final String verbValue= Properties.INTERNAL_VERBOSITY_JVM_VALUE.getDefaultAsString();

		//TODO - Should handle case when new jres are added
		final IVMInstallType[] types= JavaRuntime.getVMInstallTypes();
		for(int i= 0; i < types.length; i++){
			final IVMInstallType type= types[i];
			final IVMInstall[] jres= type.getVMInstalls();

			for(int j= 0; j < jres.length; j++){
				final IVMInstall jre= jres[j];
				final List newVmArgs= new ArrayList();

				final String[] vmArgs= jre.getVMArguments();
				if(vmArgs != null){
					for(int k= 0; k < vmArgs.length; k++){
						final String vmArg= vmArgs[k];

						if(!vmArg.startsWith(covName) && !vmArg.startsWith(verbName)){
							newVmArgs.add(vmArg);
						}
					}
				}

				newVmArgs.add(covName + covValue);
				newVmArgs.add(verbName + verbValue);

				jre.setVMArguments((String[])newVmArgs.toArray(new String[newVmArgs.size()]));
			}
		}

		setupCoverageFileChangeListener();
	}

	/**
	 * Adds a listener to the launch manager that will kick off the applicable CoverageRefreshJob when a launch has been removed.
	 * A launch is removed when it has completed running.
	 */
	private void setupCoverageFileChangeListener(){
		DebugPlugin.getDefault().getLaunchManager().addLaunchListener(new ILaunchListener(){
			boolean firstInvocation= true;
			public void launchRemoved(ILaunch launch){
				runCoverageJob(launch);
			}

			public void launchChanged(ILaunch launch){
				// no-op
			}

			public void launchAdded(ILaunch launch){
				//The first time a launch is done, the launchRemoved listener method
				//does not seem to be executed. Strange, but true. Therefore the first
				//invocation is done here to get the coverage job in phase and then never run 
				//again from this method.
				if(firstInvocation){
					runCoverageJob(launch);

					firstInvocation= false;
				}
			}

			private void runCoverageJob(ILaunch launch){
				try{
					final String name= launch.getLaunchConfiguration().getAttribute(IJavaLaunchConfigurationConstants.ATTR_PROJECT_NAME, (String)null);
					final CoverageRefreshJob job= (CoverageRefreshJob)coverageRefreshJobs.get(name);
					if(job != null){
						job.schedule();
					}
				}catch(CoreException e){
					EmmaPlugin.log(e.getStatus());
				}
			}
		});
	}
}
