/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

import com.vladium.emma.filter.Filter;
import com.vladium.emma.util.SemiRegExConverter;

/**
 * The main plugin class to be used in the desktop.
 */
public class EmmaPlugin extends AbstractUIPlugin{
	//The same identifier as specified in the manifest.mf. Used throughout the code 
	//wherever the unique prefix is needed
	public static final String ID= "com.vladium.emma";

	//Class path variable referring to the emma home location
	public final static String EMMA_HOME= "EMMA_HOME";

	//Storage for the filter groups for all projects
	private final Map projectFilters= new HashMap();
	
	//The shared instance.
	private static EmmaPlugin instance;

	/**
	 * The constructor.
	 */
	public EmmaPlugin(){
		super();
		if(instance != null){
			throw new IllegalStateException("Plug-in class already exists. This class should only be instantiated by Eclipse itself.");
		}
		instance= this;
	}

	/**
	 * Returns the shared instance.
	 */
	public static EmmaPlugin getDefault(){
		return instance;
	}

	/**
	 * Convenience logging method
	 */
	public static void log(IStatus status){
		getDefault().getLog().log(status);
	}

	/** 
	 * Gets the shell for the window that is active for the moment. 
	 * Convenience method for example for actions.
	 */
	public static Shell getActiveShell(){
		return PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
	}

	/**
	 * All groups can have a filter associated with them. The plugin class acts as
	 * a storage point for them. Another option would have been to add the 
	 * filters as session properties directly to the projects.
	 * 
	 * @param project the project to find the filter for
	 */
	public Filter getProjectFilter(IProject project) throws CoreException{
		final String name= project.getName();
		Filter filter= (Filter)projectFilters.get(name);
		if(filter == null){
			filter= new Filter();
			projectFilters.put(name, filter);

			final boolean isIncRegExEnabled= Properties.INCLUDE_FILTER_IS_REGEX.getPersistentAsBoolean(project);

			final Set incFilters= Properties.INCLUDE_FILTER_LIST.getPersistentAsSet(project);

			for(Iterator iter= incFilters.iterator(); iter.hasNext();){
				String filterRegExPattern= (String)iter.next();

				if(filterRegExPattern != null && filterRegExPattern.length() > 0){
					if(!isIncRegExEnabled){
						filterRegExPattern= SemiRegExConverter.createRegExFromSemi(filterRegExPattern).pattern();
					}

					filter.addInclusion(filterRegExPattern);
				}
			}

			final boolean isExcRegExEnabled= Properties.EXCLUDE_FILTER_IS_REGEX.getPersistentAsBoolean(project);

			final Set excFilters= Properties.EXCLUDE_FILTER_LIST.getPersistentAsSet(project);

			for(Iterator iter= excFilters.iterator(); iter.hasNext();){
				String filterRegExPattern= (String)iter.next();

				if(filterRegExPattern != null && filterRegExPattern.length() > 0){
					if(!isExcRegExEnabled){
						filterRegExPattern= SemiRegExConverter.createRegExFromSemi(filterRegExPattern).pattern();
					}

					filter.addExclusion(filterRegExPattern);
				}
			}
		}

		return filter;
	}

	/**
	 * Clear the filter group for a specific project
	 * 
	 * @param project the project to clear the filter for
	 */
	public void clearProjectFilter(IProject project) throws CoreException{
		projectFilters.remove(project.getName());
	}

	/*
	 * (non-Javadoc)
	 * @see org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext)
	 */
	public void start(BundleContext context) throws Exception{
		super.start(context);
	}

	/*
	 * (non-Javadoc)
	 * @see org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
	 */
	public void stop(BundleContext context) throws Exception{
		super.stop(context);
		instance= null;
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.ui.plugin.AbstractUIPlugin#initializeDefaultPreferences(org.eclipse.jface.preference.IPreferenceStore)
	 */
	protected void initializeDefaultPreferences(IPreferenceStore store){
		//Does nothing for the moment
	}
}
