/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.actions;

import java.io.IOException;
import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageDeclaration;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.IObjectActionDelegate;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.FileEditorInput;

import com.vladium.emma.EmmaPlugin;
import com.vladium.emma.Messages;
import com.vladium.emma.Properties;
import com.vladium.emma.editors.HtmlBrowser;
import com.vladium.emma.report.XmlReportParser;
import com.vladium.emma.util.EmmaRunner;
import com.vladium.emma.util.ReportUtil;
import com.vladium.emma.util.StreamUtil;

public class ShowReportAction implements IObjectActionDelegate{
	private static final String REF_START= "<a href=\"[^\"]+\">\\Q";
	private static final String REF_END= "\\E</a>";

	private IJavaElement selectedElement;

	/**
	 * @see IObjectActionDelegate#setActivePart(IAction, IWorkbenchPart)
	 */
	public void setActivePart(IAction action, IWorkbenchPart targetPart){
		//Do nothing
	}

	/**
	 * @see IActionDelegate#run(IAction)
	 */
	public void run(IAction action){
		try{
			final IJavaProject javaProject= selectedElement.getJavaProject();
			final IProject project= javaProject.getProject();
			final IPath reportPath= (IPath)Properties.INTERNAL_HTML_REPORT.getDefault();

			ReportUtil.buildReport(javaProject, EmmaRunner.HTML, true, reportPath);

			final IFile outFile= project.getFile(reportPath);
			if(outFile.exists()){
				IFile displayPage= outFile;

				if(!(selectedElement instanceof IJavaProject)){
					String packageName= "";
					if(selectedElement instanceof IPackageFragment){
						final IPackageFragment fragment= (IPackageFragment)selectedElement;
						packageName= fragment.getElementName();
					}else if(selectedElement instanceof ICompilationUnit){
						final ICompilationUnit sourceFile= (ICompilationUnit)selectedElement;
						final IPackageDeclaration[] packages= sourceFile.getPackageDeclarations();
						if(packages.length > 0){
							packageName= packages[0].getElementName();
						}
					}

					if(packageName.equals("")){
						packageName= XmlReportParser.DEFAULT_PACKAGE_NAME;
					}

					displayPage= parseForReference(project, outFile, packageName);

					if(selectedElement instanceof ICompilationUnit && displayPage.exists()){
						displayPage= parseForReference(project, displayPage, selectedElement.getElementName());
					}
				}

				//Show report page
				if(!displayPage.exists()){
					MessageDialog.openInformation(EmmaPlugin.getActiveShell(), Messages.report_no_match_title, Messages.report_no_match_text);
				}else{
					final IFile fileToDisplay= displayPage;
					final IEditorDescriptor desc= EmmaPlugin.getDefault().getWorkbench().getEditorRegistry().findEditor(HtmlBrowser.ID);
					Display.getDefault().asyncExec(new Runnable(){
						public void run(){
							final IWorkbenchWindow dw= PlatformUI.getWorkbench().getActiveWorkbenchWindow();
							if(dw != null){
								try{
									final IWorkbenchPage activePage= dw.getActivePage();

									if(activePage != null){
										activePage.openEditor(new FileEditorInput(fileToDisplay), desc.getId());
									}
								}catch(PartInitException e){
									EmmaPlugin.log(e.getStatus());
								}
							}
						}
					});
				}
			}
		}catch(CoreException e){
			EmmaPlugin.log(e.getStatus());
		}catch(IOException e){
			EmmaPlugin.log(new Status(IStatus.ERROR, EmmaPlugin.ID, 0, e.getMessage(), e));
		}
	}
	/**
	 * @see org.eclipse.ui.IActionDelegate#selectionChanged(IAction, ISelection)
	 */
	public void selectionChanged(IAction action, ISelection selection){
		if(selection != null && selection instanceof IStructuredSelection){
			final IStructuredSelection ss= (IStructuredSelection)selection;
			final Iterator iter= ss.iterator();
			if(iter.hasNext()){
				final Object selected= iter.next();

				if(selected instanceof IJavaElement){
					selectedElement= (IJavaElement)selected;
				}
			}
		}
	}

	/**
	 * Searches for a ahref reference in the generated report and finds the file that
	 * the href is pointing at.
	 * 
	 * @param findInFile the file to get the content to parse from
	 * @param reference the reference to a package or a source file
	 */
	public IFile parseForReference(IProject project, IFile findInFile, String reference) throws CoreException, IOException{
		final String contents= StreamUtil.getString(findInFile.getContents());

		//Special treatment of white space, regex matching problems. All whitespace in
		//reference is replaced with end quote, any character, start new quote
		String localRef= "";
		final String[] refSplit= reference.split("\\s", 0);
		for(int i= 0; i < refSplit.length; i++){
			localRef+= refSplit[i];

			if(i != refSplit.length - 1){
				localRef+= "\\E.\\Q";
			}
		}
		final Pattern packagePattern= Pattern.compile(REF_START + localRef + REF_END, Pattern.CASE_INSENSITIVE);
		String path= null;
		final Matcher packageMatcher= packagePattern.matcher(contents);
		if(packageMatcher.find()){
			final String packageHref= packageMatcher.group();
			path= packageHref.split("\"")[1];
		}

		final IPath reportRootFolder= findInFile.getProjectRelativePath().removeLastSegments(1).addTrailingSeparator();
		return project.getFile(reportRootFolder + path);
	}
}