/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.builders;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IResourceVisitor;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;

import com.vladium.emma.Properties;
import com.vladium.emma.util.EclipseFileUtil;
import com.vladium.emma.util.EmmaRunner;
import com.vladium.emma.util.FilterUtil;
import com.vladium.emma.util.JavaProjectUtil;

class MainVisitor implements IResourceVisitor, IResourceDeltaVisitor{
	/**
	 * @see IResourceVisitor#visit(IResource)
	 */
	public boolean visit(IResource res) throws CoreException{
		instrumentResource(res);
		return true;
	}

	/**
	 * @see IResourceDeltaVisitor#visit(IResourceDelta)
	 */
	public boolean visit(IResourceDelta res) throws CoreException{
		instrumentResource(res.getResource());
		return true;
	}

	/**
	 * Main instrumenting method. All classes in the package that the resource resides in 
	 * gets instrumented if the resource is a class
	 */
	public void instrumentResource(IResource resource) throws CoreException{
		//We use the source reference to find the compiled file.
		//This is because there's a bug in the package finding mechanism
		//for binary files in Eclipse. If the output path is not also on the source
		//path the method on IClassFile.getType().getFullyQualifiedName() does not 
		//include the package name. We can not then correctly apply the filtering 
		//mechanism.
		//Since Eclipse registers the manipulation of the source file after the 
		//manipulation of the class file the instrumentation can be done in this order
		//without the internal Eclipse compiler writing over our changes.
		if("java".equals(resource.getFullPath().getFileExtension())){
			final ICompilationUnit sourceFile= (ICompilationUnit)JavaCore.create(resource);

			if(sourceFile != null){
				final IPackageFragmentRoot packageRoot= (IPackageFragmentRoot)sourceFile.getAncestor(IJavaElement.PACKAGE_FRAGMENT_ROOT);
				final IJavaProject javaProject= sourceFile.getJavaProject();

				//Check if this file lives in a source folder or is just an ordinary file named .java
				final IPackageFragmentRoot[] packageRoots= JavaProjectUtil.getRootsOfType(javaProject, IPackageFragmentRoot.K_SOURCE);
				for(int i= 0; i < packageRoots.length; i++){
					if(packageRoot.equals(packageRoots[i])){
						final IType type= sourceFile.findPrimaryType();

						if(type != null && FilterUtil.isIncluded(sourceFile)){
							final String className= type.getFullyQualifiedName();
							final IProject project= resource.getProject();

							//Output location is found either from the package root and if it 
							//does not exists there, project default is used
							IPath outputLocation= packageRoot.getRawClasspathEntry().getOutputLocation();
							if(outputLocation == null){
								outputLocation= javaProject.getOutputLocation();
							}
							//Trim the project virtual name from the path
							outputLocation= outputLocation.removeFirstSegments(1);

							//Find path if folder is external linked
							final IFolder folder = project.getFolder(outputLocation.segment(0));
							outputLocation= EclipseFileUtil.getAbsoluteLocation(folder);
							
							final String projectRoot= EclipseFileUtil.getAbsoluteLocation(project).addTrailingSeparator().toOSString();

							//Build commands for Emma
							final List args= new ArrayList();

							args.add("-instrpath");
							String cpToFilePath= "";
							if(className.indexOf(".") != -1){
								cpToFilePath= File.separator + className.substring(0, className.lastIndexOf("."));

								final StringBuffer sb= new StringBuffer();
								final String[] packages= cpToFilePath.split("\\.", -1);
								for(int j= 0; j < packages.length; j++){
									sb.append(packages[j]);

									if(j != packages.length - 1){
										sb.append(File.separator);
									}
								}
								cpToFilePath= sb.toString();
							}
							args.add(outputLocation.toOSString() + cpToFilePath);

							final String metaDataPath= projectRoot + Properties.INTERNAL_META_FILE.getPersistent(project);

							args.add("-outfile");
							args.add(metaDataPath);

							args.add("-outmode");
							args.add("overwrite");

							args.add("-silent");

							//Add filter to instrument only present class and not whole directory
							//and it's subdirectories, also include inner classes
							final String fileName= resource.getFullPath().removeFileExtension().lastSegment();
							args.add("-filter");
							args.add("+" + fileName);
							args.add("-filter");
							args.add("+" + fileName + "$*");

							new EmmaRunner(EmmaRunner.INSTR).runEmma(args);
						}

						break;
					}
				}

			}
		}
	}
}