/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.decorators;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jface.viewers.IDecoration;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IDecoratorManager;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.themes.ITheme;

import com.vladium.emma.EmmaPlugin;
import com.vladium.emma.Properties;
import com.vladium.emma.ThemeProperties;
import com.vladium.emma.property.Property;

public class CoverageDecorator extends BaseDecorator implements org.eclipse.jface.util.IPropertyChangeListener, com.vladium.emma.property.IPropertyChangeListener{
	//The same identifier as specified in the plugin.xml. Used throughout the code 
	//wherever the unique id is needed
	public static final String ID= "com|vladium|emma|decorators|coverageDecorator";

	private Font passedFont;
	private Font failedFont;
	private Color passedColor;
	private Color failedColor;

	private static String[] fonts= new String[]{ThemeProperties.COVERAGE_PASSED_FONT, ThemeProperties.COVERAGE_FAILED_FONT};

	private static String[] colors= new String[]{ThemeProperties.COVERAGE_PASSED_COLOR, ThemeProperties.COVERAGE_FAILED_COLOR};

	private boolean packageClassFlag;
	private boolean packageMethodFlag;
	private boolean packageBlockFlag;
	private boolean packageLineFlag;

	private boolean sourceClassFlag;
	private boolean sourceMethodFlag;
	private boolean sourceBlockFlag;
	private boolean sourceLineFlag;

	private boolean nameDecorationActive;

	/**
	 * Constructor
	 */
	public CoverageDecorator(){
		ensureFontAndColorsCreated(fonts, colors);

		PlatformUI.getWorkbench().getThemeManager().getCurrentTheme().addPropertyChangeListener(this);

		Properties.NAME_DECORATION_ACTIVE.addPropertyChangeListener(this);

		Properties.PACKAGE_CLASS_FLAG.addPropertyChangeListener(this);
		Properties.PACKAGE_METHOD_FLAG.addPropertyChangeListener(this);
		Properties.PACKAGE_BLOCK_FLAG.addPropertyChangeListener(this);
		Properties.PACKAGE_LINE_FLAG.addPropertyChangeListener(this);

		Properties.SOURCE_CLASS_FLAG.addPropertyChangeListener(this);
		Properties.SOURCE_METHOD_FLAG.addPropertyChangeListener(this);
		Properties.SOURCE_BLOCK_FLAG.addPropertyChangeListener(this);
		Properties.SOURCE_LINE_FLAG.addPropertyChangeListener(this);

		final EmmaPlugin plugin= EmmaPlugin.getDefault();
		nameDecorationActive= Properties.NAME_DECORATION_ACTIVE.getPersistentAsBoolean(plugin);

		packageClassFlag= Properties.PACKAGE_CLASS_FLAG.getPersistentAsBoolean(plugin);
		packageMethodFlag= Properties.PACKAGE_METHOD_FLAG.getPersistentAsBoolean(plugin);
		packageBlockFlag= Properties.PACKAGE_BLOCK_FLAG.getPersistentAsBoolean(plugin);
		packageLineFlag= Properties.PACKAGE_LINE_FLAG.getPersistentAsBoolean(plugin);

		sourceClassFlag= Properties.SOURCE_CLASS_FLAG.getPersistentAsBoolean(plugin);
		sourceMethodFlag= Properties.SOURCE_METHOD_FLAG.getPersistentAsBoolean(plugin);
		sourceBlockFlag= Properties.SOURCE_BLOCK_FLAG.getPersistentAsBoolean(plugin);
		sourceLineFlag= Properties.SOURCE_LINE_FLAG.getPersistentAsBoolean(plugin);
	}

	/*
	 * (non-Javadoc)
	 * @see com.vladium.emma.property.PropertyChangeListener#propertyChange(com.vladium.emma.property.PropertyChangeEvent)
	 */
	public void propertyChange(com.vladium.emma.property.PropertyChangeEvent event){
		boolean newValue= event.getNewValueAsBoolean();

		if(event.sentFrom(Properties.NAME_DECORATION_ACTIVE)){
			nameDecorationActive= newValue;
		}else if(event.sentFrom(Properties.PACKAGE_CLASS_FLAG)){
			packageClassFlag= newValue;
		}else if(event.sentFrom(Properties.PACKAGE_METHOD_FLAG)){
			packageMethodFlag= newValue;
		}else if(event.sentFrom(Properties.PACKAGE_BLOCK_FLAG)){
			packageBlockFlag= newValue;
		}else if(event.sentFrom(Properties.PACKAGE_LINE_FLAG)){
			packageLineFlag= newValue;
		}else if(event.sentFrom(Properties.SOURCE_CLASS_FLAG)){
			sourceClassFlag= newValue;
		}else if(event.sentFrom(Properties.SOURCE_METHOD_FLAG)){
			sourceMethodFlag= newValue;
		}else if(event.sentFrom(Properties.SOURCE_BLOCK_FLAG)){
			sourceBlockFlag= newValue;
		}else if(event.sentFrom(Properties.SOURCE_LINE_FLAG)){
			sourceLineFlag= newValue;
		}
	}

	/* (non-Javadoc)
	 * @see com.vladium.emma.decorators.BaseDecorator#getId()
	 */
	public String getId(){
		return ID;
	}

	/**
	 * This method will ensure that the fonts and colors used by the decorator
	 * are cached in the registries. This avoids having to syncExec when
	 * decorating since we ensure that the fonts and colors are pre-created.
	 * 
	 * @param fonts fonts ids to cache
	 * @param colors color ids to cache
	 */
	private void ensureFontAndColorsCreated(final String[] paramFonts, final String[] paramColors){
		Display.getDefault().syncExec(new Runnable(){
			public void run(){
				final ITheme theme= PlatformUI.getWorkbench().getThemeManager().getCurrentTheme();
				for(int i= 0; i < paramColors.length; i++){
					theme.getColorRegistry().get(paramColors[i]);

				}
				for(int i= 0; i < paramFonts.length; i++){
					theme.getFontRegistry().get(paramFonts[i]);
				}
			}
		});

		final ITheme current= PlatformUI.getWorkbench().getThemeManager().getCurrentTheme();

		passedFont= current.getFontRegistry().get(ThemeProperties.COVERAGE_PASSED_FONT);
		failedFont= current.getFontRegistry().get(ThemeProperties.COVERAGE_FAILED_FONT);
		passedColor= current.getColorRegistry().get(ThemeProperties.COVERAGE_PASSED_COLOR);
		failedColor= current.getColorRegistry().get(ThemeProperties.COVERAGE_FAILED_COLOR);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.util.IPropertyChangeListener#propertyChange(org.eclipse.jface.util.PropertyChangeEvent)
	 */
	public void propertyChange(org.eclipse.jface.util.PropertyChangeEvent event){
		if(isEventOfInterest(event)){
			ensureFontAndColorsCreated(fonts, colors);
			refresh();
		}
	}

	private boolean isEventOfInterest(org.eclipse.jface.util.PropertyChangeEvent event){
		final String p= event.getProperty();
		return p.equals(ThemeProperties.COVERAGE_PASSED_FONT) || p.equals(ThemeProperties.COVERAGE_FAILED_FONT) || p.equals(ThemeProperties.COVERAGE_PASSED_COLOR) || p.equals(ThemeProperties.COVERAGE_FAILED_COLOR);
	}

	public static CoverageDecorator getInstance(){
		final IDecoratorManager decoratorManager= EmmaPlugin.getDefault().getWorkbench().getDecoratorManager();
		return (CoverageDecorator)decoratorManager.getBaseLabelProvider(ID);
	}

	/**
	 * This method should only be called by the decorator thread.
	 * 
	 * @see org.eclipse.jface.viewers.ILightweightLabelDecorator#decorate(java.lang.Object, org.eclipse.jface.viewers.IDecoration)
	 */
	public void decorate(Object element, IDecoration decoration){
		try{
			if(element instanceof IResource){
				final IResource resource= (IResource)element;
				final IProject project= resource.getProject();

				if(resource.exists() && project.isOpen() && Properties.BUILDER_ENABLED.getPersistentAsBoolean(project)){
					final IJavaElement javaElement= JavaCore.create(resource);

					boolean passed= true;
					final StringBuffer sb= new StringBuffer();
					if(javaElement instanceof IPackageFragment || javaElement instanceof IPackageFragmentRoot){
						passed= appendCoverageValue(sb, resource, Properties.COVERAGE_SUFFIX_CLASS, Properties.PACKAGE_CLASS_THRESHOLD, packageClassFlag, passed);
						passed= appendCoverageValue(sb, resource, Properties.COVERAGE_SUFFIX_METHOD, Properties.PACKAGE_METHOD_THRESHOLD, packageMethodFlag, passed);
						passed= appendCoverageValue(sb, resource, Properties.COVERAGE_SUFFIX_BLOCK, Properties.PACKAGE_BLOCK_THRESHOLD, packageBlockFlag, passed);
						passed= appendCoverageValue(sb, resource, Properties.COVERAGE_SUFFIX_LINE, Properties.PACKAGE_LINE_THRESHOLD, packageLineFlag, passed);
					}else if(javaElement instanceof ICompilationUnit){
						passed= appendCoverageValue(sb, resource, Properties.COVERAGE_SUFFIX_CLASS, Properties.SOURCE_CLASS_THRESHOLD, sourceClassFlag, passed);
						passed= appendCoverageValue(sb, resource, Properties.COVERAGE_SUFFIX_METHOD, Properties.SOURCE_METHOD_THRESHOLD, sourceMethodFlag, passed);
						passed= appendCoverageValue(sb, resource, Properties.COVERAGE_SUFFIX_BLOCK, Properties.SOURCE_BLOCK_THRESHOLD, sourceBlockFlag, passed);
						passed= appendCoverageValue(sb, resource, Properties.COVERAGE_SUFFIX_LINE, Properties.SOURCE_LINE_THRESHOLD, sourceLineFlag, passed);
					}

					if(sb.length() > 0){
						decoration.addSuffix(" (" + sb.toString() + ")");

						if(nameDecorationActive){
							Color color= null;
							Font font= null;

							if(passed){
								font= passedFont;
								color= passedColor;
							}else{
								font= failedFont;
								color= failedColor;
							}

							decoration.setFont(font);
							decoration.setForegroundColor(color);
						}
					}
				}
			}
		}catch(CoreException e){
			EmmaPlugin.log(e.getStatus());
		}
	}

	/**
	 * Utility method to append a decoration value to a decoration buffer. Return value 
	 * indicates if the coverage limit has been passed. The method does two jobs and 
	 * exists just to minimize code.
	 */
	private boolean appendCoverageValue(StringBuffer coverageBuffer, IResource resource, Property property, Property threshold, boolean appendValue, boolean previousPassed) throws CoreException{
		boolean passed= previousPassed;
		final Integer value= property.getSessionAsInt(resource);
		if(value != null){
			if(appendValue){
				if(coverageBuffer.length() > 0){
					coverageBuffer.append(",");
				}
				coverageBuffer.append(value + "%");
			}
						
			final IProject project= resource.getProject();
			final int thresholdValue= threshold.getPersistentAsInteger(project).intValue();			
			if(value.intValue() < thresholdValue){
				passed= false;
			}
		}
		return passed;
	}
}