/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.filter;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.regex.Pattern;

/**
 * A grouping of a number of inclusion and exclusion filters in the form 
 * of regular expressions. <br/><br/>
 * 
 * The filter approves a pattern if it IS included AND NOT IS excluded. If no inclusion
 * patterns are defined, a default inclusion of all is done. <br/><br/>
 * 
 * So in other words, first all inclusions are searched for the pattern. If no inclusions
 * are done, the pattern automatically validates. If the pattern is included, all 
 * exclusion patterns are searched. If no match is found there, the pattern is 
 * accepted as included. If an exclusion pattern is found, it is not accepted as included.
 * 
 * @author Erik Beijnoff erik@beijnoff.com
 * @since 2005-nov-04
 */
public class Filter{
	private final Map inclusions= new HashMap();
	private final Map exclusions= new HashMap();

	/**
	 * Add an inclusion pattern
	 * 
	 * @param filterRegex any regular expression string pattern
	 */
	public void addInclusion(String filterRegex){
		inclusions.put(filterRegex, Pattern.compile(filterRegex.trim()));
	}

	/**
	 * Removes an inclusion pattern
	 */
	public void removeInclusion(String filterRegex){
		inclusions.remove(filterRegex);
	}

	/**
	 * Add an exclusion pattern
	 * 
	 * @param filterRegex any regular expression string pattern
	 */
	public void addExclusion(String filterRegex){
		exclusions.put(filterRegex, Pattern.compile(filterRegex.trim()));
	}

	/**
	 * Removes an exclusion pattern
	 */
	public void removeExclusion(String filterRegex){
		exclusions.remove(filterRegex);
	}

	/**
	 * Main pattern matching method. The method tries to match the whole pattern
	 * on the inclusion and exclusion patterns. 
	 * 
	 * @param stringPattern the pattern to test against the filter
	 * @return if the pattern matches, that is if it is approved by the filter group
	 */
	public boolean matches(String stringPattern){
		boolean accepted= true;

		if(!inclusions.isEmpty()){
			accepted= false;

			for(Iterator iter= inclusions.values().iterator(); iter.hasNext();){
				final Pattern pattern= (Pattern)iter.next();

				if(pattern.matcher(stringPattern).matches()){
					accepted= true;
					break;
				}
			}
		}

		if(accepted){
			for(Iterator iter= exclusions.values().iterator(); iter.hasNext();){
				final Pattern pattern= (Pattern)iter.next();

				if(pattern.matcher(stringPattern).matches()){
					accepted= false;
					break;
				}
			}
		}

		return accepted;
	}
}