/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.properties;

import java.util.Iterator;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;

import com.vladium.emma.EmmaPlugin;
import com.vladium.emma.Messages;
import com.vladium.emma.Properties;
import com.vladium.emma.joblisteners.AfterBuildJobListener;
import com.vladium.emma.swtcomponents.StringListComposite;
import com.vladium.emma.util.BuildRunner;
import com.vladium.emma.util.EclipseFileUtil;
import com.vladium.emma.util.SwtUtil;

/**
 * The page where project specific settings are done. This is where the plugin is 
 * activated on a project basis.
 * 
 * @author Erik Beijnoff erik@beijnoff.com
 * @since 2005-okt-03 
 */
public class ProjectPropertyFilterPage extends BasePropertyPage{
	private StringListComposite incFilterList;
	private Button incRegExButton;

	private StringListComposite excFilterList;
	private Button excRegExButton;

	/**
	 * @see org.eclipse.jface.preference.PreferencePage#createContents(Composite)
	 */
	protected Control createContents(Composite parent){
		final Composite composite= new Composite(parent, SWT.NULL);
		final GridLayout gridLayout= new GridLayout();
		gridLayout.numColumns= 1;
		composite.setLayout(gridLayout);

		try{
			SwtUtil.createGridLabel(composite, Messages.properties_filter_desc1, 200, -1, -1);
			SwtUtil.createGridSeparator(composite, -1, -1);
			SwtUtil.createGridLabel(composite, Messages.properties_filter_desc2, 200, -1, -1);
			SwtUtil.createGridLabel(composite, Messages.properties_filter_desc3, 200, -1, -1);

			//Inclusion filters		
			SwtUtil.createGridSeparator(composite, 12, -1);
			SwtUtil.createGridLabel(composite, Messages.properties_filter_incpatterns, 200, -1, -1);
			incFilterList= new StringListComposite(composite, SWT.NONE);

			incFilterList.setText(StringListComposite.TEXTKEY_ADD, Messages.buttonlabel_add);
			incFilterList.setText(StringListComposite.TEXTKEY_EDIT, Messages.buttonlabel_edit);
			incFilterList.setText(StringListComposite.TEXTKEY_REMOVE, Messages.buttonlabel_remove);
			incFilterList.setText(StringListComposite.TEXTKEY_VALUE_TITLE, Messages.new_value);
			incFilterList.setText(StringListComposite.TEXTKEY_VALUE_LABEL, Messages.enter_new_value);

			incFilterList.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			final Set incFilters= Properties.INCLUDE_FILTER_LIST.getPersistentAsSet(getProject());

			incFilterList.setValues(incFilters);


			// Enablement of regular expression support for filters
			incRegExButton= new Button(composite, SWT.CHECK);
			incRegExButton.setSelection(Properties.INCLUDE_FILTER_IS_REGEX.getPersistentAsBoolean(getProject()));
			incRegExButton.setText(Messages.properties_filter_regexenabled);
			incRegExButton.addSelectionListener(new SelectionAdapter(){
				public void widgetSelected(SelectionEvent e){
					if(!validateRegExes(incRegExButton.getSelection(), incFilterList.getValues())){
						incRegExButton.setSelection(false);
					}
				}
			});

			incFilterList.setValidator(new RegExValidator(incRegExButton));


			//Exclusion filters
			SwtUtil.createGridSeparator(composite, 12, -1);
			SwtUtil.createGridLabel(composite, Messages.properties_filter_excpatterns, 200, -1, -1);
			excFilterList= new StringListComposite(composite, SWT.NONE);

			excFilterList.setText(StringListComposite.TEXTKEY_ADD, Messages.buttonlabel_add);
			excFilterList.setText(StringListComposite.TEXTKEY_EDIT, Messages.buttonlabel_edit);
			excFilterList.setText(StringListComposite.TEXTKEY_REMOVE, Messages.buttonlabel_remove);
			excFilterList.setText(StringListComposite.TEXTKEY_VALUE_TITLE, Messages.new_value);
			excFilterList.setText(StringListComposite.TEXTKEY_VALUE_LABEL, Messages.enter_new_value);

			excFilterList.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

			final Set excFilters= Properties.EXCLUDE_FILTER_LIST.getPersistentAsSet(getProject());

			excFilterList.setValues(excFilters);


			// Enablement of regular expression support for filters
			excRegExButton= new Button(composite, SWT.CHECK);
			excRegExButton.setSelection(Properties.EXCLUDE_FILTER_IS_REGEX.getPersistentAsBoolean(getProject()));
			excRegExButton.setText(Messages.properties_filter_regexenabled);
			excRegExButton.addSelectionListener(new SelectionAdapter(){
				public void widgetSelected(SelectionEvent e){
					if(!validateRegExes(excRegExButton.getSelection(), excFilterList.getValues())){
						excRegExButton.setSelection(false);
					}
				}
			});

			excFilterList.setValidator(new RegExValidator(excRegExButton));
		}catch(CoreException e){
			EmmaPlugin.log(e.getStatus());
		}
		return composite;
	}

	/**
	 * @see org.eclipse.jface.preference.PreferencePage#performDefaults()
	 */
	protected void performDefaults(){
		incFilterList.setValues(Properties.INCLUDE_FILTER_LIST.getDefaultAsSet());
		incRegExButton.setSelection(Properties.INCLUDE_FILTER_IS_REGEX.getDefaultAsBoolean());

		excFilterList.setValues(Properties.EXCLUDE_FILTER_LIST.getDefaultAsSet());
		excRegExButton.setSelection(Properties.EXCLUDE_FILTER_IS_REGEX.getDefaultAsBoolean());
	}

	/**
	 * @see org.eclipse.jface.preference.PreferencePage#performOk()
	 */
	public boolean performOk(){
		try{
			final IProject project= getProject();

			final boolean incRegExEnabled= incRegExButton.getSelection();
			final Set incFilters= incFilterList.getValues();
			if(!validateRegExes(incRegExEnabled, incFilters)){
				return false;
			}

			final boolean excRegExEnabled= excRegExButton.getSelection();
			final Set excFilters= excFilterList.getValues();
			if(!validateRegExes(excRegExEnabled, excFilters)){
				return false;
			}

			Properties.INCLUDE_FILTER_LIST.setPersistent(project, incFilters);
			Properties.INCLUDE_FILTER_IS_REGEX.setPersistent(project, incRegExEnabled + "");

			Properties.EXCLUDE_FILTER_LIST.setPersistent(project, excFilters);
			Properties.EXCLUDE_FILTER_IS_REGEX.setPersistent(project, excRegExEnabled + "");

			EmmaPlugin.getDefault().clearProjectFilter(project);

			//Remove the meta data file so that it can be rebuilt
			final String metaFileName= Properties.INTERNAL_META_FILE.getPersistent(project);
			EclipseFileUtil.delete(project, metaFileName);

			//Rebuild project
			BuildRunner.startBuild(Messages.build_project_title + " " + project.getName(), project, new AfterBuildJobListener(project));
		}catch(CoreException e){
			EmmaPlugin.log(e.getStatus());
		}

		return true;
	}

	/**
	 * Internal helper method that evaluates if a regular expression is in the
	 * correct form. A popup is displayed if errors occurs and the method returns false;
	 */
	private boolean validateRegExes(boolean isRegExEnabled, Set filters){
		boolean validates= true;

		if(isRegExEnabled){
			for(Iterator iter= filters.iterator(); iter.hasNext();){
				final String filterRegEx= (String)iter.next();

				try{
					Pattern.compile(filterRegEx);
				}catch(PatternSyntaxException e){
					MessageDialog.openError(EmmaPlugin.getActiveShell(), Messages.regexerror_title, Messages.regexerror_desc + e.getMessage());
					validates= false;
					break;
				}
			}
		}

		return validates;
	}
}