/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.properties;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.FocusAdapter;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Text;

import com.vladium.emma.CoverageRefreshManager;
import com.vladium.emma.EmmaPlugin;
import com.vladium.emma.Messages;
import com.vladium.emma.Properties;
import com.vladium.emma.builders.MainBuilder;
import com.vladium.emma.decorators.CoverageDecorator;
import com.vladium.emma.joblisteners.AfterBuildJobListener;
import com.vladium.emma.property.Property;
import com.vladium.emma.util.BuildRunner;
import com.vladium.emma.util.SwtUtil;

/**
 * The page where project specific settings are done. This is where the plugin is 
 * activated on a project basis.
 * 
 * @author Erik Beijnoff erik@beijnoff.com
 * @since 2005-10-03
 */
public class ProjectPropertyMainPage extends BasePropertyPage{
	private Button enabledButton;

	private Text packageClassThreshold;
	private Text packageMethodThreshold;
	private Text packageBlockThreshold;
	private Text packageLineThreshold;

	private Text sourceClassThreshold;
	private Text sourceMethodThreshold;
	private Text sourceBlockThreshold;
	private Text sourceLineThreshold;

	private String revertValue;

	/**
	 * @see org.eclipse.jface.preference.PreferencePage#createContents(Composite)
	 */
	protected Control createContents(Composite parent){
		final Composite composite= new Composite(parent, SWT.NULL);
		final GridLayout gridLayout= new GridLayout();
		gridLayout.numColumns= 1;
		composite.setLayout(gridLayout);

		// Owner text field
		try{
			enabledButton= new Button(composite, SWT.CHECK);
			enabledButton.setSelection(Properties.BUILDER_ENABLED.getPersistentAsBoolean(getProject()));
			enabledButton.setText(Messages.checkboxlabel_enable);

			final Group group2= SwtUtil.createGridGroup(composite, Messages.validation_limits, 3, 20);

			final Composite inner1= new Composite(group2, SWT.NULL);
			final GridLayout innerLayout1= new GridLayout();
			innerLayout1.numColumns= 2;
			inner1.setLayout(innerLayout1);

			SwtUtil.createGridLabel(inner1, Messages.preferences_coverage_package_header, -1, -1, 2);
			packageClassThreshold= createTextBox(inner1, Messages.preferences_coverage_class, Properties.PACKAGE_CLASS_THRESHOLD);
			packageMethodThreshold= createTextBox(inner1, Messages.preferences_coverage_method, Properties.PACKAGE_METHOD_THRESHOLD);
			packageBlockThreshold= createTextBox(inner1, Messages.preferences_coverage_block, Properties.PACKAGE_BLOCK_THRESHOLD);
			packageLineThreshold= createTextBox(inner1, Messages.preferences_coverage_line, Properties.PACKAGE_LINE_THRESHOLD);

			SwtUtil.createVertGridSeparator(group2, 40, -1);

			final Composite inner2= new Composite(group2, SWT.NULL);
			final GridLayout innerLayout2= new GridLayout();
			innerLayout2.numColumns= 2;
			inner2.setLayout(innerLayout2);

			SwtUtil.createGridLabel(inner2, Messages.preferences_coverage_source_header, -1, -1, 2);
			sourceClassThreshold= createTextBox(inner2, Messages.preferences_coverage_class, Properties.SOURCE_CLASS_THRESHOLD);
			sourceMethodThreshold= createTextBox(inner2, Messages.preferences_coverage_method, Properties.SOURCE_METHOD_THRESHOLD);
			sourceBlockThreshold= createTextBox(inner2, Messages.preferences_coverage_block, Properties.SOURCE_BLOCK_THRESHOLD);
			sourceLineThreshold= createTextBox(inner2, Messages.preferences_coverage_line, Properties.SOURCE_LINE_THRESHOLD);
		}catch(CoreException e){
			EmmaPlugin.log(e.getStatus());
		}
		return composite;
	}

	private Text createTextBox(Composite parent, String text, Property property) throws CoreException{
		final Text textBox= SwtUtil.createGridTextField(parent, property.getPersistent(getProject()), 20, -1);
		textBox.setTextLimit(3);
		((GridData)textBox.getLayoutData()).horizontalIndent= 5;

		textBox.addFocusListener(new FocusAdapter(){
			public void focusGained(FocusEvent event){
				revertValue= textBox.getText();
			}
			public void focusLost(FocusEvent event){
				boolean wrongInput= false;

				try{
					int newValue= new Integer(textBox.getText()).intValue();

					if(newValue < 0 || newValue > 100){
						wrongInput= true;
					}
				}catch(NumberFormatException e){
					wrongInput= true;
				}

				if(wrongInput){
					MessageDialog.openError(EmmaPlugin.getActiveShell(), Messages.not_percentage_title, Messages.not_percentage_desc);

					textBox.setText(revertValue);
				}
			}
		});

		SwtUtil.createGridLabel(parent, text);
		return textBox;
	}

	/**
	 * @see org.eclipse.jface.preference.PreferencePage#performDefaults()
	 */
	protected void performDefaults(){
		enabledButton.setSelection(Properties.BUILDER_ENABLED.getDefaultAsBoolean());

		packageClassThreshold.setText(Properties.PACKAGE_CLASS_THRESHOLD.getDefaultAsString());
		packageMethodThreshold.setText(Properties.PACKAGE_METHOD_THRESHOLD.getDefaultAsString());
		packageBlockThreshold.setText(Properties.PACKAGE_BLOCK_THRESHOLD.getDefaultAsString());
		packageLineThreshold.setText(Properties.PACKAGE_LINE_THRESHOLD.getDefaultAsString());

		sourceClassThreshold.setText(Properties.SOURCE_CLASS_THRESHOLD.getDefaultAsString());
		sourceMethodThreshold.setText(Properties.SOURCE_METHOD_THRESHOLD.getDefaultAsString());
		sourceBlockThreshold.setText(Properties.SOURCE_BLOCK_THRESHOLD.getDefaultAsString());
		sourceLineThreshold.setText(Properties.SOURCE_LINE_THRESHOLD.getDefaultAsString());
	}

	/**
	 * @see org.eclipse.jface.preference.PreferencePage#performOk()
	 */
	public boolean performOk(){
		try{
			final IProject project= getProject();

			Properties.BUILDER_ENABLED.setPersistent(project, enabledButton.getSelection() + "");

			if(enabledButton.getSelection()){
				enableBuilder();
			}else{
				disableBuilder();
			}

			Properties.PACKAGE_CLASS_THRESHOLD.setPersistent(project, packageClassThreshold.getText());
			Properties.PACKAGE_METHOD_THRESHOLD.setPersistent(project, packageMethodThreshold.getText());
			Properties.PACKAGE_BLOCK_THRESHOLD.setPersistent(project, packageBlockThreshold.getText());
			Properties.PACKAGE_LINE_THRESHOLD.setPersistent(project, packageLineThreshold.getText());

			Properties.SOURCE_CLASS_THRESHOLD.setPersistent(project, sourceClassThreshold.getText());
			Properties.SOURCE_METHOD_THRESHOLD.setPersistent(project, sourceMethodThreshold.getText());
			Properties.SOURCE_BLOCK_THRESHOLD.setPersistent(project, sourceBlockThreshold.getText());
			Properties.SOURCE_LINE_THRESHOLD.setPersistent(project, sourceLineThreshold.getText());

			CoverageDecorator.getInstance().refresh();
		}catch(CoreException e){
			EmmaPlugin.log(e.getStatus());
		}

		return true;
	}

	/**
	 * Enable the Emma instrumentation for the selected project
	 */
	private void enableBuilder() throws CoreException{
		final IProject project= getProject();
		final IProjectDescription desc= project.getDescription();
		final ICommand[] builders= desc.getBuildSpec();

		boolean found= false;
		for(int i= 0; i < builders.length; ++i){
			if(builders[i].getBuilderName().equals(MainBuilder.ID)){
				found= true;
				break;
			}
		}

		if(!found){
			addJarToClassPath(true);

			//Add builder to project
			final ICommand command= desc.newCommand();
			command.setBuilderName(MainBuilder.ID);

			//Add it after other builders.
			final ICommand[] newBuilders= new ICommand[builders.length + 1];
			System.arraycopy(builders, 0, newBuilders, 0, builders.length);
			newBuilders[newBuilders.length - 1]= command;
			desc.setBuildSpec(newBuilders);
			project.setDescription(desc, null);

			//Rebuild project
			BuildRunner.startBuild(Messages.build_project_title + " " + project.getName(), project, new AfterBuildJobListener(project));

			CoverageRefreshManager.getInstance().setCoverageJobStatus(project);
		}
	}

	/**
	 * Disable the Emma instrumentation for the selected project
	 */
	private void disableBuilder() throws CoreException{
		final IProject project= getProject();
		final IProjectDescription desc= project.getDescription();
		final ICommand[] builders= desc.getBuildSpec();

		boolean found= false;
		for(int i= 0; i < builders.length; ++i){
			if(builders[i].getBuilderName().equals(MainBuilder.ID)){
				found= true;
				break;
			}
		}

		if(found){
			addJarToClassPath(false);

			//Remove builder from project
			final ICommand[] newBuilders= new ICommand[builders.length - 1];

			int posIndex= 0;
			for(int i= 0; i < builders.length; i++){
				if(!builders[i].getBuilderName().equals(MainBuilder.ID)){
					newBuilders[posIndex]= builders[i];
					posIndex++;
				}
			}

			// Add new build list
			desc.setBuildSpec(newBuilders);
			project.setDescription(desc, null);

			//Rebuild project
			BuildRunner.startBuild(Messages.build_project_title + " " + project.getName(), project);

			CoverageRefreshManager.getInstance().setCoverageJobStatus(project);
		}
	}

	/**
	 * Adds/removes bundled emma jar into work directory and adds it from/to classpath. 
	 * Class path entries can be found under [project]&gt; Properties&gt; Java Build Path.
	 */
	private void addJarToClassPath(boolean add) throws CoreException{
		final IPath newClassPath= new Path(EmmaPlugin.EMMA_HOME).append((IPath)Properties.EMMA_JAR_PATH.getDefault());
		final List newEntries= new ArrayList();

		final IJavaProject jProject= ((IJavaElement)getElement()).getJavaProject();
		final IClasspathEntry[] entries= jProject.getRawClasspath();
		for(int i= 0; i < entries.length; i++){
			final IClasspathEntry entry= entries[i];
			final String entryOSPath= entry.getPath().toString();

			if(!entryOSPath.equalsIgnoreCase(newClassPath.toString())){
				newEntries.add(entry);
			}
		}

		if(add){
			newEntries.add(JavaCore.newVariableEntry(newClassPath, null, null));
		}

		jProject.setRawClasspath((IClasspathEntry[])newEntries.toArray(new IClasspathEntry[]{}), null);
	}
}