/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.property;

import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.regex.Pattern;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.ui.plugin.AbstractUIPlugin;

/**
 * Defines a property with an optional default value.
 * 
 * @author Erik Beijnoff erik@beijnoff.com
 * @since 2005-okt-03
 */
public class Property{
	//This is a way to handle strings concatenated togheter with a separator without
	//mixing up the content with the separator. Any ambigious text in the values are
	//escaped when added togheter. When the value is fetched, the strings are unescaped
	private static final String VALUE_SEPARATOR= "-|-";
	private static final Pattern VALUE_SEPARATOR_REGEX= Pattern.compile("\\-\\|\\-");
	private static final Pattern PIPE_REGEX= Pattern.compile("\\|");
	private static final Pattern DOUBLE_PIPE_REGEX= Pattern.compile("\\|\\|");
	private static final String PIPE= "|";
	private static final String DOUBLE_PIPE= "||";

	private final QualifiedName key;
	private final Object defaultValue;

	private final Set propertyChangeListeners= new HashSet();

	/**
	 * Constructor
	 * 
	 * @param key plugin unique name of the property
	 * @param defaultValue the default value when no property is set for a resource
	 */
	public Property(QualifiedName key, Object defaultValue){
		this.key= key;
		this.defaultValue= defaultValue;
	}

	/**
	 * Return the key of this property
	 */
	public QualifiedName getKey(){
		return key;
	}

	/**
	 * Return default value
	 */
	public Object getDefault(){
		return defaultValue;
	}

	/**
	 * Return default value
	 */
	public String getDefaultAsString(){
		String result= null;
		if(defaultValue != null){
			result= defaultValue.toString();
		}
		return result;
	}

	/**
	 * Return default value
	 */
	public boolean getDefaultAsBoolean(){
		boolean result= false;
		if(defaultValue != null){
			result= ((Boolean)defaultValue).booleanValue();
		}
		return result;
	}

	/**
	 * Return default value
	 */
	public Set getDefaultAsSet(){
		return (Set)defaultValue;
	}

	/**
	 * Gets a string value from the persistent store for the specified project
	 */
	public String getPersistent(IResource resource) throws CoreException{
		String result= resource.getPersistentProperty(key);
		if(result == null){
			if(defaultValue != null){
				result= defaultValue.toString();
			}
		}
		return result;
	}

	/**
	 * Gets a boolean value from the persistent store for the specified project
	 */
	public boolean getPersistentAsBoolean(IResource resource) throws CoreException{
		boolean result= ((Boolean)defaultValue).booleanValue();
		final String storedProperty= resource.getPersistentProperty(key);
		if(storedProperty != null){
			result= new Boolean(storedProperty).booleanValue();
		}
		return result;
	}

	/**
	 * Gets a boolean value from the persistent store for the specified project
	 */
	public Integer getPersistentAsInteger(IResource resource) throws CoreException{
		Integer result= (Integer)defaultValue;
		final String storedProperty= resource.getPersistentProperty(key);
		if(storedProperty != null){
			result= Integer.valueOf(storedProperty);
		}
		return result;
	}

	/**
	 * Gets a persisted property as a unmodifiable set of strings.
	 */
	public Set getPersistentAsSet(IResource resource) throws CoreException{
		Set result= (Set)defaultValue;
		String storedValue= resource.getPersistentProperty(key);
		if(storedValue != null){
			result= new HashSet();

			final String[] splittedValues= VALUE_SEPARATOR_REGEX.split(storedValue, -1);
			for(int i= 0; i < splittedValues.length; i++){
				result.add(unescapeValue(splittedValues[i]));
			}
		}

		if(result != null){
			result= Collections.unmodifiableSet(result);
		}

		return result;
	}

	/**
	 * Set a property in this projects persistent store
	 */
	public void setPersistent(IResource resource, String value) throws CoreException{
		final Object oldValue= getPersistent(resource);

		resource.setPersistentProperty(key, value);

		final Object newValue= getPersistent(resource);
		firePropertyChange(resource, oldValue, newValue);
	}

	/**
	 * Set a property in this projects persistent store
	 */
	public void setPersistent(IResource resource, Set values) throws CoreException{
		String concatValue= null;
		if(values != null && !values.isEmpty()){
			concatValue= "";
			for(Iterator iter= values.iterator(); iter.hasNext();){
				final String value= (String)iter.next();

				if(concatValue != ""){
					concatValue+= VALUE_SEPARATOR;
				}
				concatValue+= escapeValue(value);
			}
		}

		setPersistent(resource, concatValue);
	}

	/**
	 * Adds a property to this projects persistent store. The new value is 
	 * added to already existing values.
	 */
	public void addPersistent(IResource resource, String value) throws CoreException{
		Set cloned= new HashSet();
		final Set existing= getPersistentAsSet(resource);
		if(existing != null){
			cloned= new HashSet(existing);
		}

		cloned.add(value);

		setPersistent(resource, cloned);
	}

	/**
	 * Removes a value from this projects persistent store. The new value is 
	 * removed from already existing values.
	 */
	public void removePersistent(IResource resource, String value) throws CoreException{
		final Set existing= getPersistentAsSet(resource);
		if(existing != null){
			final Set cloned= new HashSet(existing);

			cloned.remove(value);

			setPersistent(resource, cloned);
		}
	}

	/**
	 * Get a preference from the default preference store for the plugin
	 */
	public String getPersistent(AbstractUIPlugin plugin){
		String result= plugin.getPreferenceStore().getString(getKey().toString());
		if(result == null || result.equals("")){
			if(getDefault() != null){
				result= getDefault().toString();
			}
		}
		return result;
	}

	/**
	 * Gets a boolean value from the persistent store for the specified plugin
	 */
	public boolean getPersistentAsBoolean(AbstractUIPlugin plugin){
		String stored= plugin.getPreferenceStore().getString(getKey().toString());

		boolean result= false;
		if(stored != null && !stored.equals("")){
			result= new Boolean(stored).booleanValue();
		}else{
			if(getDefault() != null){
				result= ((Boolean)getDefault()).booleanValue();
			}
		}
		return result;
	}

	/**
	 * Sets a preference in the default preference store for the plugin
	 */
	public void setPersistent(AbstractUIPlugin plugin, String value){
		final String oldValue= getPersistent(plugin);
		final String keyName= getKey().toString();
		final IPreferenceStore store= plugin.getPreferenceStore();
		if(value == null){
			store.setToDefault(keyName);
		}else if(!value.equals(oldValue)){
			store.setValue(keyName, value);
		}

		final Object newValue= getPersistent(plugin);
		firePropertyChange(plugin, oldValue, newValue);
	}

	/**
	 * Set a property in this resource session store
	 */
	public void setSession(IResource resource, Object value) throws CoreException{
		final Object oldValue= getSession(resource);

		resource.setSessionProperty(key, value);

		final Object newValue= getSession(resource);
		firePropertyChange(resource, oldValue, newValue);
	}

	/**
	 * Gets a object value from the session store for the specified project
	 */
	public Object getSession(IResource resource) throws CoreException{
		Object result= resource.getSessionProperty(key);
		if(result == null){
			result= defaultValue;
		}
		return result;
	}

	/**
	 * Gets a boolean value from the persistent store for the specified project
	 */
	public Integer getSessionAsInt(IResource resource) throws CoreException{
		Integer result= ((Integer)defaultValue);
		final Object storedProperty= resource.getSessionProperty(key);
		if(storedProperty != null){
			result= (Integer)storedProperty;
		}
		return result;
	}

	public void addPropertyChangeListener(IPropertyChangeListener listener){
		if(!propertyChangeListeners.contains(listener)){
			propertyChangeListeners.add(listener);
		}
	}

	public void removePropertyChangeListener(IPropertyChangeListener listener){
		propertyChangeListeners.remove(listener);
	}

	private void firePropertyChange(Object resource, Object oldValue, Object newValue){
		for(Iterator iter= propertyChangeListeners.iterator(); iter.hasNext();){
			final IPropertyChangeListener listener= (IPropertyChangeListener)iter.next();
			listener.propertyChange(new PropertyChangeEvent(this, resource, oldValue, newValue));
		}
	}

	/**
	 * Escapes all pipes "|" in a string by doubling them "||"
	 */
	private String escapeValue(String value){
		return PIPE_REGEX.matcher(value).replaceAll(DOUBLE_PIPE);
	}

	/**
	 * Unescapes from double pipe "||" form to single pipe form "|" for the string
	 */
	private String unescapeValue(String value){
		return DOUBLE_PIPE_REGEX.matcher(value).replaceAll(PIPE);
	}
}
