/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.report;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.dom4j.Document;
import org.dom4j.Element;

/**
 * Used to create a java queryable structure from an xml coverage report.
 * 
 * @author Erik Beijnoff erik@beijnoff.com
 * @since 2005-nov-14
 */
public class XmlReportParser{
	public static final String ELM_REPORT= "report";

	public static final String ELM_PACKAGES= "packages";
	public static final String ELM_CLASSES= "classes";
	public static final String ELM_METHODS= "methods";
	public static final String ELM_SRCFILES= "srcfiles";
	public static final String ELM_SRCLINES= "srclines";

	public static final String ELM_STATS= "stats";

	public static final String ELM_DATA= "data";
	public static final String ELM_ALL= "all";

	public static final String ELM_COVERAGE= "coverage";

	public static final String ELM_PACKAGE= "package";
	public static final String ELM_SRCFILE= "srcfile";
	public static final String ELM_CLASS= "class";
	public static final String ELM_METHOD= "method";

	public static final String ATTR_NAME= "name";
	public static final String ATTR_TYPE= "type";
	public static final String ATTR_VALUE= "value";

	public static final String TYPE_CLASS= "class";
	public static final String TYPE_METHOD= "method";
	public static final String TYPE_BLOCK= "block";
	public static final String TYPE_LINE= "line";

	public static final String DEFAULT_PACKAGE_NAME= "default package";

	private final Map packageData= new HashMap();
	private final Map classData= new HashMap();

	/**
	 * Constructor
	 * 
	 * @param doc the xml report document 
	 */
	public XmlReportParser(Document doc){
		final Element root= doc.getRootElement();
		final Element data= root.element(ELM_DATA);
		final Element all= data.element(ELM_ALL);

		final Set allPackages= new HashSet();
		final Set allClasses= new HashSet();

		for(Iterator iter= all.elementIterator(); iter.hasNext();){
			final Element elm= (Element)iter.next();

			if(elm.getName().equals(ELM_PACKAGE)){
				parsePackage(elm, allPackages, allClasses);
			}
		}
	}

	/**
	 * Gets all package data that hasn't been marked as read. Sets the internal state to read
	 * so a concecutive call will result in an empty collection. Also removes all data
	 * that is tagged as removed.
	 */
	public synchronized Collection getPackageData(){
		return Collections.unmodifiableCollection(packageData.values());
	}

	/**
	 * Gets all class data that hasn't been marked as read. Sets the internal state to read
	 * so a concecutive call will result in an empty collection. Also removes all data
	 * that is tagged as removed.
	 */
	public synchronized Collection getClassData(){
		return Collections.unmodifiableCollection(classData.values());
	}

	private void parsePackage(Element packageElm, Set allPackages, Set allClasses){
		String packageName= packageElm.attributeValue(ATTR_NAME);
		//The default package is named "default package". Set to "".
		if(packageName.equals(DEFAULT_PACKAGE_NAME)){
			packageName= "";
		}

		allPackages.add(packageName);

		CoverageData coverageData= (CoverageData)packageData.get(packageName);
		if(coverageData == null){
			coverageData= new CoverageData(packageName, null);
			packageData.put(packageName, coverageData);
		}

		for(Iterator iter= packageElm.elementIterator(); iter.hasNext();){
			final Element childElm= (Element)iter.next();
			final String name= childElm.getName();

			if(name.equals(ELM_COVERAGE)){
				addCoverageData(childElm, coverageData);
			}else if(name.equals(ELM_SRCFILE)){
				parseSourceFile(packageName, childElm, allClasses);
			}
		}
	}

	private void parseSourceFile(String packageName, Element sourceFileElm, Set allClasses){
		final String fileName= sourceFileElm.attributeValue(ATTR_NAME);
		final String key= packageName + ":" + fileName;

		allClasses.add(key);

		CoverageData coverageData= (CoverageData)classData.get(key);
		if(coverageData == null){
			coverageData= new CoverageData(packageName, fileName);
			classData.put(key, coverageData);
		}

		for(Iterator iter= sourceFileElm.elementIterator(); iter.hasNext();){
			final Element childElm= (Element)iter.next();
			final String name= childElm.getName();

			if(name.equals(ELM_COVERAGE)){
				addCoverageData(childElm, coverageData);
			}
		}
	}

	private void addCoverageData(Element coverageElm, CoverageData data){
		final String type= coverageElm.attributeValue(ATTR_TYPE);
		final String value= coverageElm.attributeValue(ATTR_VALUE);

		if(type.startsWith(TYPE_CLASS)){
			data.setClassCoverage(getPercentValue(value));
		}else if(type.startsWith(TYPE_METHOD)){
			data.setMethodCoverage(getPercentValue(value));
		}else if(type.startsWith(TYPE_BLOCK)){
			data.setBlockCoverage(getPercentValue(value));
		}else if(type.startsWith(TYPE_LINE)){
			data.setLineCoverage(getPercentValue(value));
		}
	}

	private int getPercentValue(String value){
		return Integer.parseInt(value.split("%")[0]);
	}
}