/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.swtcomponents;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;

/**
 * A composite that holds a list of string values in a list with buttons to add, change
 * and remove the values.
 */
public class StringListComposite extends Composite{
	//Keyes used to index and change help texts
	public static final String TEXTKEY_ADD= "add";
	public static final String TEXTKEY_EDIT= "edit";
	public static final String TEXTKEY_REMOVE= "remove";
	public static final String TEXTKEY_VALUE_TITLE= "title";
	public static final String TEXTKEY_VALUE_LABEL= "label";

	//Holds all the texts used in the dialog
	private Map texts= new HashMap();
	{
		texts.put(TEXTKEY_ADD, "Add...");
		texts.put(TEXTKEY_EDIT, "Edit...");
		texts.put(TEXTKEY_REMOVE, "Remove");
		texts.put(TEXTKEY_VALUE_TITLE, "New value");
		texts.put(TEXTKEY_VALUE_LABEL, "Enter a new value");
	}

	private Table table;
	protected TableViewer tableViewer;
	private Set values= new HashSet();

	private Button addButton;
	private Button editButton;
	private Button removeButton;

	private IInputValidator validator;

	/**
	 * Constructor
	 * 
	 * @param parent the parent composite
	 * @param style a normal swt style bitmap
	 */
	public StringListComposite(Composite parent, int style){
		super(parent, style);
		createWidgets();
	}

	/**
	 * Add a validator to this composite
	 */
	public void setValidator(IInputValidator validator){
		this.validator= validator;
	}

	/**
	 * Used to change the default texts for the dialog
	 * 
	 * @param textKey one of the tetx keyes declared in this class
	 * @param value the new textual value for the key
	 */
	public void setText(String textKey, String value){
		texts.put(textKey, value);
	}

	/*
	 * (non-Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setEnabled(boolean)
	 */
	public void setEnabled(boolean enabled){
		super.setEnabled(enabled);
		enableButtons();
	}

	/**
	 * Set the content of the string list
	 */
	public void setValues(Set values){
		this.values= new HashSet(values);

		tableViewer.setInput(this.values);
		tableViewer.refresh();
	}

	/**
	 * Get the content of the string list
	 */
	public Set getValues(){
		return Collections.unmodifiableSet(values);
	}

	private String getText(String textKey){
		return (String)texts.get(textKey);
	}

	private void createWidgets(){
		final GridLayout layout= new GridLayout();
		layout.horizontalSpacing= 6;
		layout.verticalSpacing= 6;
		layout.marginWidth= 0;
		layout.marginHeight= 0;
		layout.numColumns= 2;
		setLayout(layout);

		table= new Table(this, SWT.BORDER | SWT.V_SCROLL | SWT.H_SCROLL | SWT.MULTI | SWT.FULL_SELECTION);
		final GridData data= new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_FILL);
		table.setLayoutData(data);

		table.setHeaderVisible(false);
		table.setLinesVisible(true);

		final TableLayout tableLayout= new TableLayout();
		new TableColumn(table, SWT.NONE);
		final ColumnWeightData colData= new ColumnWeightData(100, 60, false);
		tableLayout.addColumnData(colData);
		table.setLayout(tableLayout);

		tableViewer= new TableViewer(table);
		tableViewer.setContentProvider(new IStructuredContentProvider(){
			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
			 */
			public Object[] getElements(Object inputElement){
				return ((Collection)inputElement).toArray(new String[0]);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
			 */
			public void dispose(){
				//Do nothing
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
			 */
			public void inputChanged(Viewer viewer, Object oldInput, Object newInput){
				//Do nothing
			}
		});

		tableViewer.addSelectionChangedListener(new ISelectionChangedListener(){
			/*
			 * (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
			 */
			public void selectionChanged(SelectionChangedEvent event){
				enableButtons();
			}
		});

		final Composite buttonComp= new Composite(this, SWT.NONE);
		final GridLayout layout2= new GridLayout();
		layout2.horizontalSpacing= 0;
		layout2.verticalSpacing= 8;
		layout2.marginWidth= 0;
		layout2.marginHeight= 0;
		layout2.numColumns= 1;
		buttonComp.setLayout(layout2);
		final GridData data2= new GridData(GridData.HORIZONTAL_ALIGN_END | GridData.VERTICAL_ALIGN_FILL);
		buttonComp.setLayoutData(data2);

		addButton= createButton(buttonComp, getText(TEXTKEY_ADD));
		addButton.addSelectionListener(new SelectionAdapter(){
			public void widgetSelected(SelectionEvent e){
				final InputDialog dialog= new InputDialog(getShell(), getText(TEXTKEY_VALUE_TITLE), getText(TEXTKEY_VALUE_LABEL), "", null);

				boolean reopen= true;
				while(reopen){
					int clicked= dialog.open();

					if(clicked == Window.CANCEL || validator == null){
						reopen= false;
					}else if(validator.validate(dialog.getValue())){
						reopen= false;
					}
				}

				addToList(dialog.getValue());
			}
		});

		editButton= createButton(buttonComp, getText(TEXTKEY_EDIT));
		editButton.addSelectionListener(new SelectionAdapter(){
			public void widgetSelected(SelectionEvent e){
				final IStructuredSelection selection= (IStructuredSelection)tableViewer.getSelection();

				final StringBuffer sb= new StringBuffer();
				for(Iterator iter= selection.iterator(); iter.hasNext();){
					sb.append(iter.next());
				}

				final String selected= sb.toString();
				final InputDialog dialog= new InputDialog(getShell(), getText(TEXTKEY_VALUE_TITLE), getText(TEXTKEY_VALUE_LABEL), selected, null);

				if(dialog.open() != Window.CANCEL){
					removeFromList(selection);
					addToList(dialog.getValue());
				}
			}
		});
		editButton.setEnabled(false);

		removeButton= createButton(buttonComp, getText(TEXTKEY_REMOVE));
		removeButton.addSelectionListener(new SelectionAdapter(){
			public void widgetSelected(SelectionEvent e){
				removeFromList((IStructuredSelection)tableViewer.getSelection());

				tableViewer.refresh();
			}
		});
		removeButton.setEnabled(false);
	}

	private Button createButton(Composite comp, String label){
		final Button button= new Button(comp, SWT.PUSH);
		button.setText(label);
		final GridData data= new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.VERTICAL_ALIGN_BEGINNING);
		button.setLayoutData(data);
		return button;
	}

	private void removeFromList(IStructuredSelection selection){
		values.removeAll(selection.toList());
	}

	private void addToList(String value){
		if(value != null && !value.equals("")){
			values.add(value);
			tableViewer.refresh();
		}
	}

	private void enableButtons(){
		boolean enabled= getEnabled();

		if(enabled){
			boolean itemsSelected= !tableViewer.getSelection().isEmpty();

			addButton.setEnabled(true);
			editButton.setEnabled(itemsSelected);
			removeButton.setEnabled(itemsSelected);
		}else{
			addButton.setEnabled(false);
			editButton.setEnabled(false);
			removeButton.setEnabled(false);
		}
	}
}