/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.util;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IncrementalProjectBuilder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.core.runtime.jobs.IJobChangeListener;
import org.eclipse.core.runtime.jobs.Job;

public class BuildRunner{
	/**
	 * Starts a build in the background. 
	 * This method is equal to @see #getBuildJob(String, IProject).schedule()
	 * 
	 * @param project The project to build or <code>null</code> to build the workspace.
	 */
	public static void startBuild(String buildJobName, IProject project){
		getBuildJob(buildJobName, project).schedule();
	}

	/**
	 * Starts a build in the background with a listener attached to it. 
	 * This method is equal to @see #getBuildJob(String, IProject).schedule() but it
	 * attaches a listener before the job is scheduled.
	 * 
	 * @param project The project to build or <code>null</code> to build the workspace.
	 * @param listener the listener to report progress back to
	 */
	public static void startBuild(String buildJobName, IProject project, IJobChangeListener listener){
		final Job job= getBuildJob(buildJobName, project);
		job.addJobChangeListener(listener);
		job.schedule();
	}

	/**
	 * Returns a build job
	 * 
	 * @param project The project to build or <code>null</code> to build the workspace.
	 */
	public static Job getBuildJob(String buildJobTitle, IProject project){
		final Job buildJob= new BuildJob(buildJobTitle, project);
		buildJob.setRule(ResourcesPlugin.getWorkspace().getRuleFactory().buildRule());
		buildJob.setUser(true);
		return buildJob;
	}


	//Internal job extension class
	private static final class BuildJob extends Job{
		private final IProject project;

		/**
		 * Constructor
		 * 
		 * @param name the name of the build job, displayed in the progress window
		 * @param project the project to rebuild
		 */
		private BuildJob(String name, IProject project){
			super(name);
			this.project= project;
		}

		/**
		 * A check to see if another job contains the same project to build
		 */
		public boolean overlaps(BuildJob other){
			return other.project == null || other.project.equals(project);
		}

		/* (non-Javadoc)
		 * @see org.eclipse.core.runtime.jobs.Job#run(org.eclipse.core.runtime.IProgressMonitor)
		 */
		protected IStatus run(IProgressMonitor monitor){
			synchronized(getClass()){
				if(monitor.isCanceled()){
					return Status.CANCEL_STATUS;
				}

				//Cancel all other build jobs of this kind
				final Job[] buildJobs= Platform.getJobManager().find(ResourcesPlugin.FAMILY_MANUAL_BUILD);
				for(int i= 0; i < buildJobs.length; i++){
					final Job current= buildJobs[i];

					if(current != this && current instanceof BuildJob){
						final BuildJob buildJob= (BuildJob)current;

						if(buildJob.overlaps(this)){
							current.cancel();
						}
					}
				}
			}

			IStatus resultStatus= Status.OK_STATUS;
			try{
				monitor.beginTask(getName(), 2);

				if(project != null){
					project.build(IncrementalProjectBuilder.FULL_BUILD, new SubProgressMonitor(monitor, 1));

					ResourcesPlugin.getWorkspace().build(IncrementalProjectBuilder.INCREMENTAL_BUILD, new SubProgressMonitor(monitor, 1));
				}else{
					ResourcesPlugin.getWorkspace().build(IncrementalProjectBuilder.FULL_BUILD, new SubProgressMonitor(monitor, 2));
				}
			}catch(CoreException e){
				resultStatus= e.getStatus();
			}catch(OperationCanceledException e){
				resultStatus= Status.CANCEL_STATUS;
			}finally{
				monitor.done();
			}

			return resultStatus;
		}

		/*
		 * (non-Javadoc)
		 * @see org.eclipse.core.internal.jobs.InternalJob#belongsTo(java.lang.Object)
		 */
		public boolean belongsTo(Object family){
			return ResourcesPlugin.FAMILY_MANUAL_BUILD == family;
		}
	}
}