/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.util;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.dom4j.Document;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;

import com.vladium.emma.EmmaPlugin;
import com.vladium.emma.Properties;
import com.vladium.emma.decorators.CoverageDecorator;
import com.vladium.emma.report.CoverageData;
import com.vladium.emma.report.XmlReportParser;

/**
 * This class is used for common decoration utility tasks.
 * 
 * @author Erik Beijnoff
 * @since 2002-09-15
 */
public class DecorateUtil{
	/** Hidden constructor */
	private DecorateUtil(){
		//This object can not be instantiated
	}

	/**
	 * Rebuilds all line coverage decorations
	 * 
	 * @param project the project to redecorate for
	 * @param coverageReport the report produced from a merge of the emma coverage file.
	 * 		  If null is passed in, all coverage decorations are removed.
	 */
	public static void redecorateCoverage(IProject project, Document coverageReport){
		try{
			final IJavaProject javaProject= JavaCore.create(project);

			final List notifyResources= new ArrayList();

			final IPackageFragmentRoot[] sourceRoots= JavaProjectUtil.getRootsOfType(javaProject, IPackageFragmentRoot.K_SOURCE);

			//Clean all previous line coverage
			for(int i= 0; i < sourceRoots.length; i++){
				final IJavaElement[] packages= sourceRoots[i].getChildren();

				for(int j= 0; j < packages.length; j++){
					final IPackageFragment pack= (IPackageFragment)packages[j];
					final IResource pResource= pack.getResource();

					if(pResource != null && pResource.exists() && Properties.COVERAGE_SUFFIX_CLASS.getSession(pResource) != null){
						setReportData(pResource, null, notifyResources);
					}

					final ICompilationUnit[] sources= pack.getCompilationUnits();
					for(int k= 0; k < sources.length; k++){
						final ICompilationUnit sourceFile= sources[k];
						final IResource sResource= sourceFile.getResource();

						if(sResource != null && sResource.exists() && Properties.COVERAGE_SUFFIX_CLASS.getSession(sResource) != null){
							setReportData(sResource, null, notifyResources);
						}
					}
				}
			}

			if(coverageReport != null){
				final XmlReportParser reportDeltaParser= new XmlReportParser(coverageReport);

				//Add all packages with line coverage from report
				for(Iterator iter= reportDeltaParser.getPackageData().iterator(); iter.hasNext();){
					final CoverageData data= (CoverageData)iter.next();

					for(int i= 0; i < sourceRoots.length; i++){
						final IPackageFragment pack= sourceRoots[i].getPackageFragment(data.getPackageName());
						if(pack != null){
							setReportData(pack.getResource(), data, notifyResources);
						}
					}
				}

				//Add all classes with line coverage from report		
				for(Iterator iter= reportDeltaParser.getClassData().iterator(); iter.hasNext();){
					final CoverageData data= (CoverageData)iter.next();

					for(int i= 0; i < sourceRoots.length; i++){
						final IPackageFragment fragment= sourceRoots[i].getPackageFragment(data.getPackageName());
						if(fragment != null){
							final ICompilationUnit sourceFile= fragment.getCompilationUnit(data.getClassName());

							if(sourceFile != null){
								setReportData(sourceFile.getResource(), data, notifyResources);
							}
						}
					}
				}
			}

			CoverageDecorator.getInstance().refresh((IResource[])notifyResources.toArray(new IResource[notifyResources.size()]));
		}catch(CoreException e){
			EmmaPlugin.log(e.getStatus());
		}
	}


	private static void setReportData(IResource resource, CoverageData data, List notifyResources) throws CoreException{
		if(resource != null && resource.exists()){
			if(data == null){
				Properties.COVERAGE_SUFFIX_CLASS.setSession(resource, null);
				Properties.COVERAGE_SUFFIX_METHOD.setSession(resource, null);
				Properties.COVERAGE_SUFFIX_BLOCK.setSession(resource, null);
				Properties.COVERAGE_SUFFIX_LINE.setSession(resource, null);
			}else{
				Properties.COVERAGE_SUFFIX_CLASS.setSession(resource, new Integer(data.getClassCoverage()));
				Properties.COVERAGE_SUFFIX_METHOD.setSession(resource, new Integer(data.getMethodCoverage()));
				Properties.COVERAGE_SUFFIX_BLOCK.setSession(resource, new Integer(data.getBlockCoverage()));
				Properties.COVERAGE_SUFFIX_LINE.setSession(resource, new Integer(data.getLineCoverage()));
			}
			notifyResources.add(resource);
		}
	}
}