/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.util;

import java.io.File;

import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;

import com.vladium.emma.EmmaPlugin;

/**
 * This class is used for common eclipse file system manipulation tasks.
 * 
 * @author Erik Beijnoff
 * @since 2002-09-15
 */
public class EclipseFileUtil{
	/** Hidden constructor */
	private EclipseFileUtil(){
		//This object can not be instantiated
	}

	/**
	 * Get the full disk location of a resource
	 */
	public static IPath getAbsoluteLocation(IResource resource){
		IPath result= null;

		if(resource.isLocal(IResource.DEPTH_ZERO)){
			final IPath resolvedLocation= resource.getLocation();

			IPath location= resolvedLocation;
			if(resource.isLinked()){
				location= resource.getRawLocation();
			}

			if(location != null){
				result= location;

				if(resolvedLocation != null && !isPathVariable(resource)){
					// No path variable used.
					final File file= resolvedLocation.toFile();
					if(!file.exists()){
						result= null;
					}
				}
			}
		}

		return result;
	}

	private static boolean isPathVariable(IResource resource){
		if(!resource.isLinked()){
			return false;
		}

		final IPath location= resource.getLocation();
		if(location == null){
			return true;
		}

		final IPath rawLocation= resource.getRawLocation();
		if(location.equals(rawLocation)){
			return false;
		}

		return true;
	}

	/**
	 * Creates all parent folders for a path if they don't exist already
	 * 
	 * @param project the project to create the path in
	 * @param path the path that contains the info
	 */
	public static void createFolders(IProject project, IPath path){
		try{
			final String[] segments= path.segments();

			IPath tmpPath= new Path("");
			for(int i= 0; i < segments.length - 1; i++){
				final String segment= segments[i];

				if(i != 0){
					tmpPath= tmpPath.addTrailingSeparator();
				}
				tmpPath= tmpPath.append(segment);


				final IFolder folder= project.getFolder(tmpPath);
				if(!folder.exists()){
					folder.create(true, true, null);
				}
			}
		}catch(CoreException e){
			EmmaPlugin.log(e.getStatus());
		}
	}

	/**
	 * Deletes a file or folder and any subfiles and folders after first refreshing it. 
	 * This is needed since Eclipse may not be aware of the file if it has been created 
	 * by some third party intervention. Therefore a first refresh is needed.
	 * 
	 * @param project the project in which the fiel resides
	 * @param the path to the file locally within the project
	 */
	public static void delete(IProject project, IPath filePath){
		try{
			IResource resource= project.getFile(filePath);
			resource.refreshLocal(IResource.DEPTH_INFINITE, null);

			if(!resource.exists()){
				resource= project.getFolder(filePath);
			}

			if(resource.exists()){
				resource.delete(true, null);
			}
		}catch(CoreException e){
			EmmaPlugin.log(e.getStatus());
		}
	}

	/**
	 *@see #delete(IProject, IPath)
	 */
	public static void delete(IProject project, String filePath){
		delete(project, new Path(filePath));
	}
}