/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.util;


/**
 * This class is used for common tasks related to exceptions
 * 
 * @author Erik Beijnoff
 * @since 2002-09-15
 */
public class ExceptionUtil{
	/** Hidden constructor */
	private ExceptionUtil(){
		//This object can not be instantiated
	}

	/**
	 * Rethrows an exception, wrapped in a new instance of the exception type given. The 
	 * rethrown exception is given a generic explanatory message.
	 * 
	 * @param wrapperType the class of the new wrapper exception. Must inherit from Exception
	 * @param e the exception to wrap
	 */
	public static void throwAsRuntime(final Exception e){
		final String message= getPosition(1) + ", an exception of type " + e.getClass().getName() + " occured: " + e.getMessage();

		throw new RuntimeException(message, e);
	}

	/**
	 * Builds a common exception message with: <br/>
	 * 1) The position where the exception occured <br/>
	 * 2) The class name of the exception <br/>
	 * 3) Any message that the exception carries <br/>
	 * 
	 * @return the String containing the information declared above
	 */
	public static String getCommonMessage(final Exception e){
		return getPosition(1) + ", an exception of type " + e.getClass().getName() + " occured: " + e.getMessage();

	}

	/**
	 * This method prints the name of the current method that calls it.
	 * 
	 * @see #getPosition(int)
	 */
	public static String getPosition(){
		return getPosition(1);
	}

	/**
	 * This method prints the name of the method that has the stack index given as 
	 * parameter. It should only be used under exceptional circumstances as for example 
	 * when an Exception is thrown, since it needs to create an Exception itself to find 
	 * the current position in the code.
	 * 
	 * @param position the position in the call stack of the method to print the name for. 
	 * 		  if 0 is given as argument, this points to the method that is invoking this one.
	 * @return a String on the form "[className].[methodName]()"
	 */
	public static String getPosition(final int position){
		final StackTraceElement method= new Exception().getStackTrace()[position + 1];

		String className= method.getClassName();
		className= className.substring(className.lastIndexOf(".") + 1);
		final String methodName= method.getMethodName();

		return className + "." + methodName + "()";
	}

	/**
	 * This method prints the full name of the current method, including the 
	 * classpath. It should only be used under exceptional circumstances as 
	 * for example when an Exception is thrown, since it needs to create an 
	 * Exception itself to find the current position in the code.
	 * 
	 * @return a String on the form "[classPath].[className].[methodName]()"
	 */
	public static String getFullPosition(){
		final StackTraceElement method= new Exception().getStackTrace()[1];

		return method.getClassName() + "." + method.getMethodName() + "()";
	}
}