/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.util;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * This class is used for common File handling tasks
 * 
 * @author Erik Beijnoff
 * @since 2002-09-15
 */
public class FileUtil{
	/** Hidden constructor */
	private FileUtil(){
		//This object can not be instantiated
	}

	/**
	 * Get the content of a File as a byte array
	 * 
	 * @param file the file to parse into the byte array
	 * @return the parsed byte array
	 */
	public static byte[] getBytes(final File file) throws IOException{
		final long length= file.length();

		// You cannot create an array using a long.
		// It needs to be shorter or as long as an int.
		if(length > Integer.MAX_VALUE){
			throw new IllegalArgumentException("Input file is too big. " + "The biggest file size that can be streamed is approximately 2GB.");
		}

		final InputStream is= new FileInputStream(file);
		byte[] bytes= new byte[(int)length];
		int offset= 0;
		int numRead= 0;
		while(offset < bytes.length && (numRead= is.read(bytes, offset, bytes.length - offset)) >= 0){
			offset+= numRead;
		}

		// Ensure all the bytes have been read in
		if(offset < bytes.length){
			throw new IOException("Could not completely read file " + file.getName());
		}

		is.close();

		return bytes;
	}

	/**
	 * Writes an InputStream to a File
	 * 
	 * @param file the File to write to
	 * @param in the InputStream to read from
	 * @throws IllegalArgumentException if the file does not exist
	 */
	public static void writeToFile(final File file, final InputStream in) throws IOException{
		if(!file.exists()){
			throw new IllegalArgumentException("The referenced file does not exist: " + file.getAbsolutePath());
		}

		FileOutputStream fo= null;
		try{
			fo= new FileOutputStream(file);
			BufferedOutputStream out= new BufferedOutputStream(fo);

			//Do writing
			byte[] buffer= new byte[32 * 1024];
			int bytesRead= 0;
			while((bytesRead= in.read(buffer)) != -1){
				out.write(buffer, 0, bytesRead);
			}
		}finally{
			if(in != null){
				in.close();
			}

			if(fo != null){
				fo.close();
			}
		}
	}


	/**
	 * Writes a byte array to a File
	 * 
	 * @param file the File to write to
	 * @param content the byte array to write
	 * @throws IllegalArgumentException if the file does not exist
	 */
	public static void writeToFile(final File file, final byte[] content) throws IOException{
		if(!file.exists()){
			throw new IllegalArgumentException("The referenced file does not exist: " + file.getAbsolutePath());
		}

		final FileOutputStream fo= new FileOutputStream(file);
		final BufferedOutputStream out= new BufferedOutputStream(fo);
		out.write(content);

		out.close();
	}
}