/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.util;

import java.io.File;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.osgi.util.NLS;

import com.vladium.emma.EmmaPlugin;
import com.vladium.emma.Messages;
import com.vladium.emma.Properties;

/**
 * This class is used for common emma report tasks.
 * 
 * @author Erik Beijnoff
 * @since 2005-12-10
 */
public class ReportUtil{
	/** Hidden constructor */
	private ReportUtil(){
		//This object can not be instantiated
	}

	/**
	 * Takes all the normal neccessary actions to build a report from the Emma meta data
	 * 
	 * @param javaProject the project to build the report for
	 * @param type one of the report types found in @link EmmaRunner
	 */
	public static void buildReport(IJavaProject javaProject, int type, boolean displayCreateError, IPath outLocalPath){
		try{
			final IProject project= javaProject.getProject();
			final IPath projectAbsRoot= EclipseFileUtil.getAbsoluteLocation(project);

			final IPackageFragmentRoot[] roots= javaProject.getPackageFragmentRoots();

			//Find out how many of the roots that are folders of the correct type
			int noOfFolders= 0;
			for(int i= 0; i < roots.length; i++){
				if(roots[i].getKind() == IPackageFragmentRoot.K_SOURCE){
					noOfFolders++;
				}
			}

			//Index source folders
			int sourcePathPos= 0;
			final IPath[] sourcePaths= new IPath[noOfFolders];
			for(int i= 0; i < roots.length; i++){
				if(roots[i].getKind() == IPackageFragmentRoot.K_SOURCE){
					final IPath rootPath= roots[i].getPath().removeFirstSegments(1);
					String rootOSString= rootPath.toOSString();
					if(!rootOSString.equals("")){
						rootOSString= File.separator + rootOSString;
					}
					sourcePaths[sourcePathPos]= new Path(projectAbsRoot + rootOSString);
					sourcePathPos++;
				}
			}

			final String metaDataFilePath= Properties.INTERNAL_META_FILE.getPersistent(project);
			final IFile metaDataFile= project.getFile(metaDataFilePath);
			metaDataFile.refreshLocal(0, null);

			final String coverageFilePath= Properties.INTERNAL_COVERAGE_FILE.getPersistent(project);
			final IFile coverageFile= project.getFile(coverageFilePath);
			coverageFile.refreshLocal(0, null);

			final String absRootString= projectAbsRoot.addTrailingSeparator().toOSString();
			if(!metaDataFile.exists() || !coverageFile.exists()){
				if(displayCreateError){
					MessageDialog.openInformation(EmmaPlugin.getActiveShell(), Messages.report_missingfile_title, NLS.bind(Messages.report_missingfile_text, new Object[]{absRootString + metaDataFilePath, absRootString + coverageFilePath}));
				}
			}else{
				String outFile= absRootString + outLocalPath.toOSString();

				EmmaRunner.runReport(type, new Path(outFile), new Path(absRootString + metaDataFilePath), new Path(absRootString + coverageFilePath), sourcePaths);

				//Refresh output so that it shows in explorer
				project.getFile(outLocalPath).refreshLocal(IResource.DEPTH_ZERO, null);

				if(type == EmmaRunner.HTML){
					final IPath reportRootFolder= outLocalPath.removeLastSegments(1).addTrailingSeparator();
					project.getFolder(reportRootFolder.append("_files")).refreshLocal(IResource.DEPTH_INFINITE, null);
				}
			}
		}catch(CoreException e){
			EmmaPlugin.log(e.getStatus());
		}
	}
}