/* Copyright (C) 2005 Erik Beijnoff. All rights reserved.
 * 
 * This program and the accompanying materials are made available under
 * the terms of the Common Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/cpl-v10.html
 */
package com.vladium.emma.util;

import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

/**
 * Helper class to ease creation of swt constructs, such as creating tables, widgets
 * or layouyts
 * 
 * @author Erik Beijnoff
 * @since 2002-09-15
 */
public class SwtUtil{
	/** Hidden constructor */
	private SwtUtil(){
		//This object can not be instantiated
	}

	/**
	 * @see #createGridLabel(Composite, String, int, int, int, int)
	 */
	public static Label createGridLabel(Composite parent, String text){
		return createGridLabel(parent, text, -1, -1, -1);
	}

	/**
	 * Creates a normal text label for a grid layout.
	 * 
	 * @param parent the parent of the label
	 * @param text the text the label has
	 * @param widthHint the width before the label wraps. -1 means no width set.
	 * @param verticalIndent the top padding of the label in pixels. -1 means no value set.
	 * @param horizontalSpan the number of columns the widget should span	 
	 */
	public static Label createGridLabel(Composite parent, String text, int widthHint, int verticalIndent, int horizontalSpan){
		final Label label= new Label(parent, SWT.WRAP);
		label.setText(text);

		if(widthHint != -1 || verticalIndent != -1 || horizontalSpan != -1){
			final GridData gd= new GridData(GridData.FILL_HORIZONTAL);
			if(widthHint != -1){
				gd.widthHint= widthHint;
			}
			if(verticalIndent != -1){
				gd.verticalIndent= verticalIndent;
			}
			if(horizontalSpan != -1){
				gd.horizontalSpan= horizontalSpan;
			}
			label.setLayoutData(gd);
		}
		return label;
	}

	/**
	 * Creates a normal separator (horizontal line) for a grid layout.
	 * 
	 * @param parent the parent of the label
	 * @param verticalIndent the top padding of the label in pixels. -1 means no value set
	 * @param horizontalSpan the number of columns the widget should span
	 */
	public static Label createGridSeparator(Composite parent, int verticalIndent, int horizontalSpan){
		final Label label= new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);

		final GridData gd= new GridData(GridData.FILL_HORIZONTAL);
		if(verticalIndent != -1){
			gd.verticalIndent= verticalIndent;
		}
		if(horizontalSpan != -1){
			gd.horizontalSpan= horizontalSpan;
		}
		label.setLayoutData(gd);
		return label;
	}

	/**
	 * Creates a vertical separator (vertical line) for a grid layout.
	 * 
	 * @param parent the parent of the label
	 * @param horizontalIndent the top padding of the label in pixels. -1 means no value set
	 * @param verticalSpan the number of columns the widget should span
	 */
	public static Label createVertGridSeparator(Composite parent, int horizontalIndent, int verticalSpan){
		final Label label= new Label(parent, SWT.SEPARATOR | SWT.VERTICAL);

		final GridData gd= new GridData(GridData.FILL_VERTICAL);
		if(horizontalIndent != -1){
			gd.horizontalIndent= horizontalIndent;
		}
		if(verticalSpan != -1){
			gd.verticalSpan= verticalSpan;
		}
		label.setLayoutData(gd);
		return label;
	}

	/**
	 * Creates a normal group (a bordered part) with a grid layout.
	 * 
	 * @param parent the parent of the widget
	 * @param text the text in the border of the group
	 * @param numColumns the number of columns the grid in the widget has
	 * @param verticalIndent the top padding of the widget in pixels. -1 means no value set
	 */
	public static Group createGridGroup(Composite parent, String text, int numColumns, int verticalIndent){
		final Group group= new Group(parent, SWT.NONE);
		group.setText(text);

		final GridLayout layout= new GridLayout();
		layout.numColumns= numColumns;
		group.setLayout(layout);

		final GridData gd= new GridData(GridData.FILL_HORIZONTAL);
		if(verticalIndent != -1){
			gd.verticalIndent= verticalIndent;
		}
		group.setLayoutData(gd);
		return group;
	}

	/**
	 * Creates a normal text field with a grid layout.
	 * 
	 * @param parent the parent of the widget
	 * @param text the text in the border of the group
	 * @param widthHint the width of the field. -1 means that the grabHorizontalSpace property is set..
	 * @param horizontalSpan the number of columns the widget should span	 
	 */
	public static Text createGridTextField(Composite parent, String text, int widthHint, int horizontalSpan){
		final Text textField= new Text(parent, SWT.BORDER);
		textField.setText(text);

		GridData gd= null;
		if(widthHint == -1){
			gd= new GridData(GridData.FILL_HORIZONTAL);
			gd.grabExcessHorizontalSpace= true;
		}else{
			gd= new GridData();
			gd.widthHint= widthHint;
		}
		if(horizontalSpan != -1){
			gd.horizontalSpan= horizontalSpan;
		}
		textField.setLayoutData(gd);
		return textField;
	}
}